
/* $Id: nw.h,v 1.1 2008/10/31 13:58:14 jayrfink Exp $*/

#ifndef _NETWARD_H
#define _NETWARD_H

#define _BSD_SOURCE 1

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/time.h>
#include <netinet/in_systm.h>
#include <pcap.h>
#ifndef SUNOS
#include <net/ethernet.h>
#else
#include <sys/ethernet.h>
#endif
#include <netinet/in.h>
#include <netinet/ip.h>
#include <signal.h>
#include <math.h>
#include <unistd.h>
#include <arpa/inet.h>
#include <sys/socket.h>
#include <netdb.h>
#include <semaphore.h>
#include <fcntl.h>
#include <getopt.h>
#include <errno.h>
#include <netinet/udp.h>
#include <net/if.h>
#ifndef DARWIN
#include <net/ppp_defs.h>
#endif
#include <sys/ioctl.h>
#include <time.h>

#include "utils.h"

#define PACKAGE "netward"
#define DEFAULT_LOG_FILE "/var/log/netward.log"

#define MAX_RULES 16
#define MAX_RULE_LEN 256

/* TCP header */
struct tcphdr {
	uint16_t source;
	uint16_t dest;
	uint32_t seq;
	uint32_t ack_seq;
#  if __BYTE_ORDER == __LITTLE_ENDIAN
	uint16_t res1:4;
	uint16_t doff:4;
	uint16_t fin:1;
	uint16_t syn:1;
	uint16_t rst:1;
	uint16_t psh:1;
	uint16_t ack:1;
	uint16_t urg:1;
	uint16_t res2:2;
#  elif __BYTE_ORDER == __BIG_ENDIAN
	uint16_t doff:4;
	uint16_t res1:4;
	uint16_t res2:2;
	uint16_t urg:1;
	uint16_t ack:1;
	uint16_t psh:1;
	uint16_t rst:1;
	uint16_t syn:1;
	uint16_t fin:1;
#  else
#   error "Adjust your <bits/endian.h> defines"
#  endif
	uint16_t window;
	uint16_t check;
	uint16_t urg_ptr;
};

typedef struct ether_header eth_hdr;	/* Ethernet header */
typedef struct ip ip_hdr;	/* IP header       */
typedef struct tcphdr tcp_hdr;	/* TCP header      */
typedef struct udphdr udp_hdr;	/* UDP header      */

/* 
 * Internal Data structures 
 */

/* Match rule */
#define TYPE_NORMAL		0
#define TYPE_QUICK 		1

/* TCP flags */
#define TH_FIN    0x01
#define TH_SYN    0x02
#define TH_RST    0x04
#define TH_PUSH   0x08
#define TH_ACK    0x10
#define TH_URG    0x20

/* Redirector interfaces */
#define REDIR_IPFILTER	0x01
#define REDIR_IPFW		0x02

/* Port range structure */
struct port {
	uint16_t begin;
	uint16_t end;
};

/* Rule structure */
struct rule {
	int type;		/* Rule type */
	uint8_t proto;		/* Protocol */
	struct in_addr src_ip, dst_ip;	/* Src, Dst IP */
	uint32_t src_prefix, dst_prefix;	/* Src, Dst IP prefix */
	struct port src_port, dst_port;	/* Src, Dst port */
	uint8_t flags;		/* Packet flags */
	uint16_t score;		/* Packet score */
};

/* IPs Array Item */
struct host {
	uint32_t id;
	struct in_addr addr;
	char ipstr[16];		/* Standard length */
	uint16_t pts;
};

/* Options */
struct options {
	char *pcap_dev;		/* Device */
	char daemon;		/* Daemon mode disabled by default */
	char *log;		/* Logfile */
	uint32_t refresh;	/* Time between report refresh */
	uint32_t timeout;	/* Time to exit */
	uint16_t warning;	/* Score limit which make warning ddos */
	uint16_t critical;	/* Score limit which make critical ddos */
	char *script_warn;	/* Warning Script */
	char *script_crit;	/* Critical Script */
};

/* Struct net */
struct net {
	struct in_addr addr;	/* Net address */
	uint32_t mask;		/* Net mask */
};

/* Functions prototypes */
void nwcore(u_char * args, const struct pcap_pkthdr *header,
		   const u_char * packet);

/* Packet Engine and Checks */
int match_packet(const u_char * packet, const struct rule *r);
struct rule *parse_rule(char *str);
int init_rules(void);

void ckpkt(int sig);

/* Tree maintance */
int add_host(uint32_t pos, struct in_addr ip, int try_opt);
int find_host(uint32_t pos, struct in_addr ip);
int score_host(uint32_t pos, uint16_t score);
void pack_tree();
void optimize_tree(uint32_t beg, uint32_t end);
void browse_tree(uint32_t pos);
int cmp_ip(const void *s1, const void *s2);

/* Global variables */
extern struct options opt;	/* Command line options */
extern struct host *hosts;	/* Hosts score tables */
extern uint32_t hosts_num;	/* Number of hosts */
extern struct host *tmphosts;	/* Temporary hosts score tables */
extern struct net *netfilter;	/* Net filter to collect points only for specified networks */
extern uint32_t netfilter_num;	/* Number net filter networks */
extern uint32_t max_id;		/* Max items in score table */
extern uint32_t level_id;	/* Score tree level */
extern struct rule *rules;	/* Rules table */
extern uint32_t packets_max;	/* Maximum packets to read */
extern uint32_t vflag;		/* Verbose mode flag */
extern uint8_t rules_num;	/* Number of rules */
extern char default_rules[MAX_RULES][MAX_RULE_LEN];	/* Default rules */

/* Mutexes */
extern short m_packet;		/* Get packet function mutex */
extern short m_display;		/* Display function mutex */

#endif
