/*
 * sysmem.c -- display memory statistics on lcd display
 * Usage: sysmem [-p io_addr] [-d delay]
 *
 * Written by Benjamin Tse, July 1997
 *
 * NB: Assumes that the lcd has already been set up by lcdtime
 *
 * sysmem displays memory and swap statistics on the lcd in the given format:
 *  Mem  xxx% /  xxM
 *  Swap yyy% /  yyM
 * This is displayed from the sixteenth character on the first and second lines
 * of the lcd.
 *
 * The percentage use is updated every 5 seconds by default. This can be 
 * changed with the delay parameter.
 *
 * Legal stuff: At no stage was this program written, assembled or compiled on
 * any computer at the University of Melbourne, Australia. This program is
 * Copyright (C) 1997 Benjamin Tse (blt@students.cs.mu.oz.au) and covered by
 * GNU's GPL. In particular, this program is free software and comes WITHOUT
 * ANY WARRANTY.
 *
 * $Id: sysmem.c,v 1.1 1997/07/15 09:03:44 blt Exp blt $
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <getopt.h>

#include "port.h"
#include "lcd_sem.h"
#include "lcd.h"
#include "meminfo.h"

#define TRUE 1
#define FALSE 0
#define PORT 0x278
#define DEF_DELAY 5

/* default locations - the sixteenth character on the first line */
#define MEM_X 16
#define MEM_Y 0

int semid;

void usage(void);

void
main(int argc, char *argv[])
{
    unsigned int port;
    char c, memstring[16 + 1];
    unsigned long memtotal, swaptotal;
    float delay;

    port = PORT;	/* initialize defaults */
    delay = DEF_DELAY;

    while ((c = getopt(argc, argv, "p:d:")) != EOF)
    {
	switch (c)
	{
	 case ('p'):
	    if (sscanf(optarg, "%x", &port) != 1)
	    {
	        usage();
		exit(EXIT_FAILURE);
	    }
	    break;
	 case ('d'):
	    if (sscanf(optarg, "%f", &delay) != 1)
	    {
	        usage();
		exit(EXIT_FAILURE);
	    }
	    break;
	 default:
	    fprintf(stderr, "Unknown option\n");
	    usage();
	    exit(EXIT_FAILURE);
	}
    }

    /* set access permissions on the port */
    if ((ioperm(port, 1, 1)) == -1) {
	fprintf(stderr, "%s: port 0x%x invalid.\n", argv[0], port);
	perror(argv[0]);
	exit(EXIT_FAILURE);
    }
    if ((ioperm(port + 2, 1, 1)) == -1) {
	fprintf(stderr, "%s: port 0x%x invalid.\n", argv[0], port + 2);
	perror(argv[0]);
	exit(EXIT_FAILURE);
    }
    
    semid = sem_get();

    /*
     * 0123456789012345
     * Mem  100% / 128M
     * Swap  72% /  16M
     */
    meminfo(NULL, &memtotal, NULL, &swaptotal);
    sprintf(memstring, "Mem     %% /%4ldM", memtotal / (1024L * 1024L));
    sem_wait(semid);
    lcd_charaddr(port, MEM_X, MEM_Y);
    lcd_printstr(port, memstring);
    sem_signal(semid);

    sprintf(memstring, "Swap    %% /%4ldM", swaptotal / (1024L * 1024L));
    sem_wait(semid);
    lcd_charaddr(port, MEM_X, MEM_Y + 1);
    lcd_printstr(port, memstring);
    sem_signal(semid);

    for ( ; ; )
    {
        int i;
	unsigned long memused, swapused;
	float per_mem, per_swap;

	if ((i = meminfo(&memused, NULL, &swapused, NULL)) > 0)
	{
	    per_mem = (float) memused / (float) memtotal * 100.0;
	    per_swap = (float) swapused / (float) swaptotal * 100.0;

	    sprintf(memstring, "%3d", (int) per_mem);
	    sem_wait(semid);
	    lcd_charaddr(port, MEM_X + 5, MEM_Y);
	    lcd_printstr(port, memstring);
	    sem_signal(semid);

	    sprintf(memstring, "%3d", (int) per_swap);
	    sem_wait(semid);
	    lcd_charaddr(port, MEM_X + 5, MEM_Y + 1);
	    lcd_printstr(port, memstring);
	    sem_signal(semid);
	}
	/* else problems getting load averages */

	usleep(delay * 1000000L);	/* update every 5 seconds */
    }
}

void
usage(void)
{
    fprintf(stderr, "Usage: sysmem [-p io_port] [-d delay]\n");
}
