/*
 * sysload.c -- display system load on lcd display
 * Usage: sysload [-p io_addr] [-d delay]
 *
 * Written by Benjamin Tse, July 1997
 *
 * NB: Assumes that the lcd has already been set up by lcdtime
 *
 * sysload displays the system load on the lcd in the given format:
 *  System load x.xx
 * This is displayed from the first character of the second line of the lcd.
 *
 * The load is updated every 5 seconds by default. This can be changed with
 * the delay parameter.
 *
 * Legal stuff: At no stage was this program written, assembled or compiled on
 * any computer at the University of Melbourne, Australia. This program is
 * Copyright (C) 1997 Benjamin Tse (blt@students.cs.mu.oz.au) and covered by
 * GNU's GPL. In particular, this program is free software and comes WITHOUT
 * ANY WARRANTY.
 * 
 * $Id: sysload.c,v 1.1 1997/07/15 08:47:31 blt Exp blt $
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <getopt.h>

#include "port.h"
#include "lcd_sem.h"
#include "lcd.h"
#include "loadavg.h"

#define TRUE 1
#define FALSE 0
#define PORT 0x278
#define DEF_DELAY 5

/* default locations - the first character on the second line */
#define LOAD_X 0
#define LOAD_Y 1

int semid;

void usage(void);

void
main(int argc, char *argv[])
{
    unsigned int port;
    char c, loadstring[16 + 1];
    float delay;

    port = PORT;	/* initialize defaults */
    delay = DEF_DELAY;

    while ((c = getopt(argc, argv, "p:d:")) != EOF)
    {
	switch (c)
	{
	 case ('p'):
	    if (sscanf(optarg, "%x", &port) != 1)
	    {
	        usage();
		exit(EXIT_FAILURE);
	    }
	    break;
	 case ('d'):
	    if (sscanf(optarg, "%f", &delay) != 1)
	    {
	        usage();
		exit(EXIT_FAILURE);
	    }
	    break;
	 default:
	    fprintf(stderr, "Unknown option\n");
	    usage();
	    exit(EXIT_FAILURE);
	}
    }

    /* set access permissions on the port */
    if ((ioperm(port, 1, 1)) == -1) {
	fprintf(stderr, "%s: port 0x%x invalid.\n", argv[0], port);
	perror(argv[0]);
	exit(EXIT_FAILURE);
    }
    if ((ioperm(port + 2, 1, 1)) == -1) {
	fprintf(stderr, "%s: port 0x%x invalid.\n", argv[0], port + 2);
	perror(argv[0]);
	exit(EXIT_FAILURE);
    }
    
    semid = sem_get();

    sprintf(loadstring, "System load ");
    sem_wait(semid);
    lcd_charaddr(port, LOAD_X, LOAD_Y);
    lcd_printstr(port, loadstring);
    sem_signal(semid);

    for ( ; ; )
    {
        int i;
	float load_one, load_five, load_fifteen;

	if ((i = loadavg(&load_one, &load_five, &load_fifteen)) > 0)
	{
	    if (load_one < 100)
	    {
	        sprintf(loadstring, "System load %.*f", 
			((load_one < 10) ? 2:1), load_one);
		/* calc decimal places - if in range [0-10) need 2 digits
		   otherwise the range [10-100) needs 1 digit */
	    }
	    else
	        sprintf(loadstring, "System load 100+");

	    /* update just the load number - 12 is the offset of the load
	       number in the string */
	    sem_wait(semid);
	    lcd_charaddr(port, 12, LOAD_Y);
	    lcd_printstr(port, loadstring + 12); 
	    sem_signal(semid);
	}
	/* else problems getting load averages */

	usleep(delay * 1000000L);
    }
}

void
usage(void)
{
    fprintf(stderr, "Usage: sysload [-p io_port] [-d delay]\n");
}
