/*
 * meminfo.c -- Reads memory statistics from /proc/meminfo
 *
 * Written by Benjamin Tse (blt@students.cs.mu.oz.au), July 1997
 *
 * Gets memory stats from /proc/meminfo. meminfo() takes four arguments. Two
 * for the number of bytes used and in total, the other two for the number
 * of bytes of swap space used and in total. Each argument can be set to 
 * NULL to ignore that value. Returns the number of memory stats read in,
 * -1 on error.
 *
 * Assumes that the file has the format:
 *  <header line>
 *  Mem: <memused> <memtotal> <anything else>
 *  Swap: <swapused> <swaptotal> <anything else>
 *  <anything else>
 *
 * $Id: meminfo.c,v 1.1 1997/07/15 08:53:51 blt Exp blt $
 */

#include <stdio.h>

#define MAXLEN	100

int
meminfo(unsigned long *memused, unsigned long *memtotal, 
	unsigned long *swapused, unsigned long *swaptotal)
{
    FILE *fp;
    int num_read;
    char memstring[MAXLEN];

    if ((fp = fopen("/proc/meminfo", "r")) == NULL)
    {
        /* problems opening meminfo file */
        return -1;
    }

    /* throw away header line */
    if (fgets(memstring, MAXLEN - 1, fp) == NULL)
        return -1;

    num_read = 0;
    if (fgets(memstring, MAXLEN - 1, fp) != NULL)
    {
        /* memory line */
        if (memtotal != NULL)
	{
	    sscanf(memstring, "%*s %ld", memtotal);
	    ++num_read;
	}
	if (memused != NULL)
	{
	    sscanf(memstring, "%*s %*d %ld", memused);
	    ++num_read;
	}
    }
    if (fgets(memstring, MAXLEN - 1, fp) != NULL)
    {
        /* swap line */
        if (swaptotal != NULL)
	{
	    sscanf(memstring, "%*s %ld", swaptotal);
	    ++num_read;
	}
	if (swapused != NULL)
	{
	    sscanf(memstring, "%*s %*d %ld", swapused);
	    ++num_read;
	}
    }

    return num_read;
}
