/*
 * lcdtime.c -- time display for the parallel port lcd
 * Usage: lcdtime [-p io_addr]
 * 
 * Written by Benjamin Tse (blt@students.cs.mu.oz.au), October 1995, July 1997
 *
 * [Release 0.2  - Added month and day of month display
 * 
 * Legal stuff: At no stage was this program written, assembled or compiled on
 * any computer at the University of Melbourne, Australia. This program is
 * Copyright (C) 1995, 1997 Benjamin Tse and covered by GNU's GPL. In 
 * particular, this program is free software and comes WITHOUT ANY WARRANTY.
 * 
 * $Id: lcdtime.c,v 1.4 1997/07/15 09:07:17 blt Exp blt $
 */

#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <string.h>
#include <unistd.h>	/* for usleep, ioperm */
#include <getopt.h>

#include "port.h"
#include "lcd_sem.h"
#include "lcd.h"

#define TRUE  1
#define FALSE 0
#define PORT 0x278      /* default port number */

void usage(void);
void powerup(void);
void turnoff(void);

static char rcsId[] = "$Id: lcdtime.c,v 1.4 1997/07/15 09:07:17 blt Exp blt $";

int semid;              /* semaphore to mediate access to parallel port */

int
main(int argc, char *argv[])
{
    char c;
    unsigned int port;
    time_t l_time;
    
    port = PORT;
    
    while ((c = getopt(argc, argv, "p:")) != EOF)
    {
	switch (c)
	{
	 case ('p'):
	    if (sscanf(optarg, "%x", &port) != 1)
	    {
	        usage();
		exit(EXIT_FAILURE);
	    }
	    break;
	 default:
	    fprintf(stderr, "Unknown option\n");
	    usage();
	    exit(EXIT_FAILURE);
	}
    }
    
    /* set access permissions on the port */
    if ((ioperm(port, 1, 1)) == -1) {
	fprintf(stderr, "%s: port 0x%x invalid.\n", argv[0], port);
	perror(argv[0]);
	exit(EXIT_FAILURE);
    }
    if ((ioperm(port + 2, 1, 1)) == -1) {
	fprintf(stderr, "%s: port 0x%x invalid.\n", argv[0], port + 2);
	perror(argv[0]);
	exit(EXIT_FAILURE);
    }

    semid = sem_get();

    powerup();
    sem_wait(semid);
    lcd_charaddr(port, 0, 0);
    sem_signal(semid);

    l_time = time(NULL);
    for ( ; ; )
    {

	if (l_time != time(NULL))   /* update each second */
	{
	    struct tm *loctime;
	    char timestring[16 + 1], tstr[9], prevstr[9];

	    tstr[8] = prevstr[8] = '\0';

	    l_time = time(NULL);
	    loctime = localtime(&l_time);

	    strftime(timestring, 16 + 1, "%H:%M:%S  %b %d", loctime);
	    strncpy(tstr, timestring, 8);

	    sem_wait(semid);
	    if (tstr[4] != prevstr[4])
	    {
	        lcd_charaddr(port, 0, 0);      /* update whole time each min */
		lcd_printstr(port, timestring);
	    }
	    else
	    {
		lcd_charaddr(port, 6, 0);      /* update secs each second */
		lcd_printstr(port, tstr + 6);
	    }
	    sem_signal(semid);

	    strncpy(prevstr, tstr, 8);
	}
	usleep(200000L);   /* 0.2 sec */
    }
    /*    turnoff(); */

    return 0;
}

void
usage(void)
{
    fprintf(stderr, "Usage: lcdtime [-p io_port]\n");
}

/*
 * setup 8-bit, 2-line mode, small chars; turn on display, cursor off 
 * and not blinking; then clears screen and sends cursor home
 */

void
powerup(void)
{
    sem_wait(semid);
    lcd_control(PORT, TRUE, FALSE, FALSE);
    lcd_sysset(PORT, TRUE, TRUE, TRUE);
    lcd_clear(PORT);
    lcd_home(PORT);
    sem_signal(semid);
}

void
turnoff(void)
{
    sem_wait(semid);
    lcd_control(PORT, FALSE, 0, 0);
    sem_signal(semid);
}
