/*
 * lcdshift.c -- shift viewport of lcd display
 * Usage: lcdshift [-p io_addr] [-d delay] [-s]
 *
 * NB: Assumes that the lcd has already been set up by lcdtime.
 *
 * lcdshift scrolls the viewport of the lcd. By default it moves the
 * viewport right (ie. characters shift left) by 16 characters, waits
 * 2 seconds, then moves the viewport left by 16 characters. It then repeats
 * this sequence. The delay between shifts can be set by the delay parameter.
 *
 * Specifying the 's' argument forces the viewport to continually scroll
 * to the right by one character every delay seconds. This allows a scrolling
 * marquee type display.
 *
 * Written by Benjamin Tse (blt@students.cs.mu.oz.au), July 1997
 *
 * Legal stuff: At no stage was this program written, assembled or compiled on
 * any computer at the University of Melbourne, Australia. This program is
 * Copyright (C) 1997 Benjamin Tse (blt@students.cs.mu.oz.au) and covered by
 * GNU's GPL. In particular, this program is free software and comes WITHOUT
 * ANY WARRANTY.
 * 
 * $Id: lcdshift.c,v 1.1 1997/07/15 08:29:13 blt Exp blt $
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <getopt.h>

#include "port.h"
#include "lcd_sem.h"
#include "lcd.h"

#define TRUE 1
#define FALSE 0
#define PORT 0x278	/* default port number */

void usage(void);

int semid;		/* semaphore to mediate access to parallel port */

void
main(int argc, char *argv[])
{
    char c;
    unsigned int port;
    float delay;
    int scroll;

    port = PORT;	/* initialize defaults */
    scroll = FALSE;
    delay = 2.0;

    while ((c = getopt(argc, argv, "p:d:s")) != EOF)
    {
	switch (c)
	{
	 case ('p'):
	    if (sscanf(optarg, "%x", &port) != 1)
	    {
	        usage();
		exit(EXIT_FAILURE);
	    }
	    break;
	 case ('d'):
	    if (sscanf(optarg, "%f", &delay) != 1)
	    {
	        usage();
		exit(EXIT_FAILURE);
	    }
	    break;
	 case ('s'):
	    scroll = TRUE;
	    break;
	 default:
	    fprintf(stderr, "Unknown option\n");
	    usage();
	    exit(EXIT_FAILURE);
	}
    }

    /* set access permissions on the port */
    if ((ioperm(port, 1, 1)) == -1) {
	fprintf(stderr, "%s: port 0x%x invalid.\n", argv[0], port);
	perror(argv[0]);
	exit(EXIT_FAILURE);
    }
    if ((ioperm(port + 2, 1, 1)) == -1) {
	fprintf(stderr, "%s: port 0x%x invalid.\n", argv[0], port + 2);
	perror(argv[0]);
	exit(EXIT_FAILURE);
    }
    
    semid = sem_get();

    if (!scroll)
    {
	for ( ; ; )
	{
	    int i;
	    
	    for (i = 0; i < 16; i++)
	    {
		sem_wait(semid);
		lcd_shift(port, LEFT, DISPLAY);
		sem_signal(semid);
	    }
	    usleep(delay * 1000000L);

	    for (i = 0; i < 16; i++)
	    {
		sem_wait(semid);
		lcd_shift(port, RIGHT, DISPLAY);
		sem_signal(semid);
	    }
	    usleep(delay * 1000000L);
	}
    }
    else
    {
        /* scrolling display */
        for ( ; ; )
	{
	    sem_wait(semid);
	    lcd_shift(port, LEFT, DISPLAY);
	    sem_signal(semid);

	    usleep(delay * 1000000L);
	}
    }
}

void
usage(void)
{
    fprintf(stderr, "Usage: lcdshift [-p io_port] [-d delay] [-s]\n");
}
