/*
 * lcd module code for Hitachi HD44780 or SMOS SED1130 type LCD modules
 * 
 * Written by Benjamin Tse (blt@mundil.cs.mu.oz.au), August-October 1995
 * 
 * Reference:  LCD Module FAQ (ver 35A), Christopher Burian (cburian@uiuc.edu)
 *               ftp://ftp.ee.ualberta.ca/pub/cookbook/faq/lcd.doc
 * 
 * Legal stuff: At no stage was this program written, assembled or compiled on
 * any computer at the University of Melbourne, Australia. This program is
 * Copyright (C) 1995 Benjamin Tse (blt@mundil.cs.mu.oz.au) and covered by
 * GNU's GPL. In particular, this program is free software and comes WITHOUT
 * ANY WARRANTY.
 *
 * $Id: lcd.c,v 1.1 1997/07/15 03:21:26 blt Exp blt $
 */

#include <unistd.h>

#include "port.h"
#include "lcd.h"

#define SEC_LINE_ADDR  0x40  /* address of first character on second line */

static char rcsId[] = "$Id: lcd.c,v 1.1 1997/07/15 03:21:26 blt Exp blt $";

static inline void togglectrl(unsigned short int port);
static inline void togglechar(unsigned short int port);


struct lcd_stat
{
    int cur_x;		/* current cursor location (0,0) is first character */
    int cur_y;
    char display[HEIGHT][WIDTH];	/* lcd memory */
    int window_col;	/* x co-ord of leftmost character in view */
};

static struct lcd_stat mylcd;

/*
 * toggle lines for lcd control functions
 */

static inline void togglectrl(unsigned short int port)
{
    port_out(port, ENABLE | CTRL);
    usleep(TCYCE);
    port_out(port, CTRL);
}

/*
 * toggle lines for lcd character read/write operations
 */

static inline void togglechar(unsigned short int port)
{
    port_out(port, ENABLE | CHAR);
    usleep(TCYCE);
    port_out(port, CHAR);
}

/*
 * turn lcd on/off; set cursor on/off; set blinking on/off
 */

void
lcd_control(unsigned short int port, int lcd_on, int cursor_on, int cursor_blink)
{
    unsigned char val;

    port_out(port + 2, METER_OFF | RS);

    val = 8;
    if (lcd_on)
	val = val | 4;
    if (cursor_on)
        val = val | 2;
    if (cursor_blink)
        val = val | 1;
    port_out(port, val);
    togglectrl(port + 2);
}

/*
 * lcd system set for 4/8 bit operation; 1/2 line display; character size
 * (5x11 or 5x8 dots)
 */

void
lcd_sysset(unsigned short int port, int eightbit, int twoline, int smallchar)
{
    unsigned char val;

    port_out(port + 2, METER_OFF | RS);

    val = 32;
    if (eightbit)
        val = val | 16;
    if (twoline)
        val = val | 8;
    if (!smallchar)
        val = val | 4;
    port_out(port, val);
    togglectrl(port + 2);
}

/*
 * move viewport or cursor right/left by one character
 * function : CURSOR or DISPLAY   currently only moves display
 */

void
lcd_shift(unsigned short int port, int direction, int function)
{
    unsigned char val;

    port_out(port + 2, METER_OFF | RS);

    val = 16 | 8;
    if (direction == RIGHT)
	val = val | 4;
    port_out(port, val);
    togglectrl(port + 2);
}

/*
 * clear display
 */

void
lcd_clear(unsigned short int port)
{
    port_out(port + 2, METER_OFF | RS);
    port_out(port, 1);
    togglectrl(port + 2);
}

/* 
 * home cursor
 */

void
lcd_home(unsigned short int port)
{
    mylcd.cur_x = mylcd.cur_y = 0;
    /* clear array here */

    port_out(port + 2, METER_OFF | RS);
    port_out(port, 2 | 1);
    togglectrl(port + 2);
}

/*
 * print a character at the current location
 * presume cursor location incremented to the next address
 */

void
lcd_print(unsigned short int port, char c)
{
    mylcd.display[mylcd.cur_y][mylcd.cur_x] = c;
    mylcd.cur_x = (mylcd.cur_x + 1) % WIDTH;

    port_out(port + 2, METER_OFF | RS);
    port_out(port, c);
    togglechar(port + 2);
}

/*
 * prints a string at the current location
 */

void
lcd_printstr(unsigned short int port, const char *str)
{
    int i;
    
    for (i = 0; str[i] != '\0'; ++i)
        lcd_print(port, str[i]);
}


/*
 * set character address, (0,0) is the home address
 * doesn't actually check if values are sane
 */

void
lcd_charaddr(unsigned short int port, int x, int y)
{
    int val;
    
    mylcd.cur_x = x;
    mylcd.cur_y = y;
    
    val = x + y * SEC_LINE_ADDR;
    
    port_out(port + 2, METER_OFF | RS);
    port_out(port, 128 | ((char) val & 0x7f));
    togglectrl(port + 2);
}
