/*
 * lcdtext.c V1.0 -- display for the parallel port lcd (c)2/1997 M.Prinke
 * Usage: lcdtext [-p io_addr] [-t1 "first line"] [-t2 "second line"]
 *                [-h] [-v]
 * based on lcdtime.c by Benjamin Tse (blt@mundil.cs.mu.oz.au), October 1995
 * REFERENCES: port.h             by <damianf@wpi.edu>
 *             Data Sheet LTN211, Philips
 *             Various FAQs and TXTs about Hitachi's LCD Controller HD44780 -
 *                www.paranoia.com/~filipg is a good starting point
 * This program is Copyright (C) 1997 Matthias Prinke
 * <m.prinke@trashcan.mcnet.de> and covered by GNU's GPL.
 * In particular, this program is free software and comes WITHOUT
 * ANY WARRANTY.
 */
 
/* #define DOS */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#ifndef DOS
	#include <unistd.h>	/* for usleep, ioperm */
#endif

#include "port.h"
#include "lcd4.h"

#ifndef EXIT_FAILURE
	#define EXIT_FAILURE 20
#endif

#define TRUE  1
#define FALSE 0
#define DISPLAYWIDTH 16
#define PORT 0x378      /* default port number */
#define EIGHTBIT FALSE

void usage(void);
void powerup(int port);
void turnoff(int port);
void convert(unsigned char to[], unsigned char from[]);

int
main(int argc, unsigned char *argv[])
{
	unsigned int port;
	unsigned int p=0;
	int opt;
	unsigned char text1[80] = "\0";
	unsigned char text2[80] = "\0";
	unsigned char text[80] = "\0";

	port = PORT;

	/* change port adress ? */
	for (opt=1; opt<argc; opt++) {
		/* strange thing: DOS seems to dislike leading "0x" with hex input */ 
		if (sscanf(argv[opt], "-p%x", &p) < 3) /* not -p0x378 ? */
			if (strcmp(argv[opt], "-p") == 0)  /* -p 0x378 then ? */  
				if (++opt <= argc)
					sscanf(argv[opt], "%x", &p);
				else {
					usage();
					exit(20);
				}
	}

	if ((p == 0x278) || (p == 0x378) || ((p >= 0x300) && (p <= 0x3FF)))
		port = p;
	else
		if (p != 0)
{
			fprintf(stderr, "Invalid port address 0x%x !\n", p);
			usage();
			exit(20);  
		}

	/* text for first line */
	for (opt=1; opt<argc; opt++) {
		if (strcmp(argv[opt], "-t1") == 0)
			if (++opt <= argc)
{
				convert(text, argv[opt]);
				strncpy(text1, text, DISPLAYWIDTH);
			} else
{
				usage();
				exit(20);
			}
	}

	/* text for second line */
	for (opt=1; opt<argc; opt++)
{
		if (strcmp(argv[opt], "-t2") == 0) 
			if (++opt <= argc)
{
				convert(text, argv[opt]);
				strncpy(text2, text, DISPLAYWIDTH);
			} else
{
				usage();
				exit(20);
			}
	}

	/* short message */
	for (opt=1; opt<argc; opt++)
{
		if (strcmp(argv[opt], "-v") == 0) {
			printf("lcdtext V1.0 -- port address: 0x%x\n", port);
			break;
		}
	}

	/* cry for help ? */
	for (opt=1; opt<argc; opt++)
{
		if (strcmp(argv[opt], "-h") == 0) {
			usage();
			break;
		}
	}

	#ifndef DOS
		/* set access permissions on the port */
		if ((ioperm(port, 1, 1)) == -1) {
			fprintf(stderr, "%s: port 0x%x invalid.\n", argv[0], port);
			perror(argv[0]);
			exit(EXIT_FAILURE);
		}
	#endif

	powerup(port);

	lcd_charaddr(port, 0, 0);
	lcd_printstr(port, text1);
	lcd_charaddr(port, 0, 1);
	lcd_printstr(port, text2);

	/* turnoff(); */
	#ifndef DOS
		ioperm(port, 1, 0);
	#endif

	return 0;
}

/*
 * convert '\+'-sequence to special character
 */

void
convert (unsigned char to[], unsigned char from[])
{
	int j=0, i=0;
	do {
		if (from[i] == '\\')
			if (from[++i] == '+')
				to[j] = from[++i] + 128;
			else
				to[j] = from[--i];
		else
			to[j] = from[i];
		++i;
		++j;
	} while (from[i] != '\0');
}

void
usage(void)
{
	fprintf(stderr, "\nlcdtext V1.0 -- parallel port lcd program\n");
	fprintf(stderr, "Copyright (c)3/1997 M.Prinke <m.prinke@trashcan.mcnet.de>\n\n");
	fprintf(stderr, "based on lcdtime by Benjamin Tse(blt@mundil.cs.mu.oz.au)\n");
	fprintf(stderr, "Usage: lcdtext [-p io_addr] [-t1 \"first line\"] [-t2 \"second line\"] [-h] [-v] \n");
	fprintf(stderr, "Needs 4 bit-type connecting! ");
	fprintf(stderr, "Default port address is 0x%x.\n", PORT);
	fprintf(stderr, "This program is mainly for poor guys with DOS running.\n");
	fprintf(stderr, "Linux users will prefer lcdstat.\n\n");
	fprintf(stderr, "lcdtext comes with ABSOLUTELY NO WARRANTY.\n");
	fprintf(stderr, "This is free software under the terms of the GNU Public License.\n\n");
}

/*
 * setup 4-bit, 2-line mode, small chars; turn on display, cursor off
 * and not blinking; then clears screen and sends cursor home
 */

void
powerup(int port)
{
	/* syset first to turn to 4 bit mode */
	lcd_sysset(port, EIGHTBIT, TRUE, TRUE);
	lcd_control(port, TRUE, FALSE, FALSE);
	lcd_clear(port);
	lcd_home(port);
}

void
turnoff(int port)
{
	lcd_control(PORT, FALSE, 0, 0);
}
