/*
 * lcdstat4.c -- status display for the parallel port lcd (c)2/1997 M.Prinke
 * Usage: lcdstat4 [-p io_addr] [-v]
 * based on lcdtime.c by Benjamin Tse (blt@mundil.cs.mu.oz.au), October 1995
 * REFERENCES: port.h              by <damianf@wpi.edu>
 *             meter.c/ledstat.txt by Joseph W. Vigneau <joev@wpi.edi>
 *             Data Sheet LTN211, Philips
 *             Various FAQs and TXTs about Hitachi's LCD Controller HD44780 -
 *                www.paranoia.com/~filipg is a good starting point
 * This program is Copyright (C) 1997 Matthias Prinke
 * <m.prinke@trashcan.mcnet.de> and covered by GNU's GPL.
 * In particular, this program is free software and comes WITHOUT
 * ANY WARRANTY. 
 *
 * Note1: I commented out the semaphore stuff- which I don't understand, 
 *        sorry! I won't connect LEDs to the port either.
 */

#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <string.h>
#include <unistd.h>	/* for usleep, ioperm */
#include <getopt.h>

#include "port.h"
#include "lcd4.h"
#include "lcdstat.h"

#define TRUE  1
#define FALSE 0
#define PORT 0x378      /* default port number */
#define EIGHTBIT FALSE
#define VERSION "V1.0"

void usage(void);
void powerup(int port);
void turnoff(int port);

int
main(int argc, char *argv[])
{
	char c, *timestring, tstr[9], prevstr[9];
	unsigned int port;
	time_t l_time;
	char cpu[7], mem[7], swp[7], up[16];
	int upsec;
	int verbose = FALSE;
	
	port = PORT;

	while ((c = getopt(argc, argv, "p:v")) != EOF) {
		switch (c) {
			case ('p'):
				if (sscanf(optarg, "%x", &port) != 1) {
					usage();
					exit(EXIT_FAILURE);
				}
				break;
			case ('v'):
				verbose = TRUE;
				break;
			default:
				fprintf(stderr, "Unknown option\n");
				usage();
				exit(EXIT_FAILURE);
		}
	}

	if (verbose) 
		printf("lcdstat4 %s -- port address: 0x%x\n", VERSION, port);
	
	/* set access permissions on the port */
	if ((ioperm(port, 1, 1)) == -1) {
		fprintf(stderr, "%s: port 0x%x invalid.\n", argv[0], port);
		perror(argv[0]);
		exit(EXIT_FAILURE);
	}

	powerup(port);
	lcd_charaddr(port, 0, 0);

	tstr[8] = prevstr[8] = '\0';
	l_time = time(NULL);
	for ( ; ; ) {
		/* update each second */
		l_time = time(NULL);
		timestring = ctime(&l_time);
		strncpy(tstr, timestring + 11, 8);

		if (tstr[4] != prevstr[4]) {
			/* show time each min */
			lcd_clear(port);
			lcd_charaddr(port, 0, 0);
			lcd_printstr(port, "time:");
			lcd_charaddr(port, 6, 0);
			lcd_printstr(port, tstr);
			lcd_charaddr(port, 0, 1);
			upsec = (int) uptime();
			sprintf(up, "  up: %dd %dh %dm", upsec/86400, upsec/3600%60, upsec/60%60); 
			lcd_printstr(port, up);
			sleep(3); /* 3sec */
			lcd_clear(port);
			lcd_charaddr(port, 0, 0);
			lcd_printstr(port, "CPU:  MEM:  SWP:");
	    } else {
			lcd_charaddr(port, 0, 1);
			sprintf(cpu, "%3.2f  ", loadavg());
			lcd_printstr(port, cpu);
			lcd_charaddr(port, 6, 1);
			sprintf(mem, "%3.1f  ", memload());
			lcd_printstr(port, mem);
			lcd_charaddr(port, 12, 1);
			sprintf(swp, "%3.1f  ", swpload());
			lcd_printstr(port, swp);
		}
		strncpy(prevstr, tstr, 8);
		sleep(1);   /* 1 sec */
	} /* for loop */
	/* turnoff(); */
	/* 
	 * for future capatibility: "New pentium does endless loop in x secs." 
	 */
	ioperm(port, 1, 0);
	return 0;
}

void
usage(void)
{
	fprintf(stderr, "lcdstat %s -- status display on LCD at parallel port\n",VERSION);
	fprintf(stderr, "Copyright (c)3/1997 M.Prinke <m.prinke@trashcan.mcnet.de>\n\n");
	fprintf(stderr, "based on lcdtime by Benjamin Tse(blt@mundil.cs.mu.oz.au)\n");
	fprintf(stderr, "Usage: lcdstat4 [-p io_port] [-v]\n");
	fprintf(stderr, "Needs 4 bit-type connecting! ");
	fprintf(stderr, "Default port address is 0x%x.\n", PORT);
	fprintf(stderr, "lcdtext comes with ABSOLUTELY NO WARRANTY.\n");
	fprintf(stderr, "This is free software under the terms of the GNU Public License.\n\n");
}

/*
 * setup 4-bit, 2-line mode, small chars; turn on display, cursor off 
 * and not blinking; then clears screen and sends cursor home
 */

void
powerup(int port)
{
	// sem_wait(semid);
	/* syset first to turn to 4 bit mode */
	lcd_sysset(port, EIGHTBIT, TRUE, TRUE);
	lcd_control(port, TRUE, FALSE, FALSE);
	lcd_clear(port);
	lcd_home(port);
	// sem_signal(semid);
}

void
turnoff(int port)
{
	// sem_wait(semid);
	lcd_control(PORT, FALSE, 0, 0);
	// sem_signal(semid);
}
