/* lcd4.c
 * lcd module code for Hitachi HD44780 or SMOS SED1130 type LCD modules
 * The module is operated in it's 4 bit-mode to be connected to a single
 * 8 bit-port
 *
 * Code mainly adapted from the package lcdtime by
 * Benjamin Tse (blt@mundil.cs.mu.oz.au), August/October 1995
 * which uses the LCD-controller's 8 bit-mode.
 * References: port.h              by <damianf@wpi.edu>
 *             Data Sheet LTN211, Philips
 *             Various FAQs and TXTs about Hitachi's LCD Controller HD44780 -
 *                www.paranoia.com/~filipg is a good starting point              ???   
 * This program is Copyright (C) 1997 Matthias Prinke
 * <m.prinke@trashcan.mcnet.de> and covered by GNU's GPL.
 * In particular, this program is free software and comes WITHOUT
 * ANY WARRANTY.
 * Note: Here METER_OFF is really METER_OFF while in lcdtime it's
 *       actually /METER_OFF
 */

#include <unistd.h>
#include "port.h"
#include "lcd4.h"

#define SEC_LINE_ADDR  0x40  /* address of first character on second line */

struct lcd_stat
{
	int cur_x;		/* current cursor location (0,0) is first character */
	int cur_y;
	char display[HEIGHT][WIDTH];	/* lcd memory */
	int window_col;	/* x co-ord of leftmost character in view */
};

static struct lcd_stat mylcd;

/*
 * turn lcd on/off; set cursor on/off; set blinking on/off
 */

void
lcd_control(unsigned short int port, int lcd_on, int cursor_on, int cursor_blink)
{
	unsigned char val, h, l;

	port_out(port, METER_OFF);
	val = 8;
	if (lcd_on)
		val = val | 4;
	if (cursor_on)
		val = val | 2;
	if (cursor_blink)
		val = val | 1;
	h = 0;	/* high nibble */
	l = val&15; /*  low nibble */
	port_out(port, METER_OFF | ENABLE | h);
	usleep(TCYCLE); /* data acknowledged when ENABLE goes H->L */
	port_out(port, METER_OFF | h);  
	port_out(port, METER_OFF | ENABLE | l);
	usleep(TCYCLE);
	port_out(port, METER_OFF | l);
}

/*
 * lcd system set for 4/8 bit operation; 1/2 line display; character size
 * (5x11 or 5x8 dots)
 */

void
lcd_sysset(unsigned short int port, int eightbit, int twoline, int smallchar)
{
	unsigned char val, h, l;

	port_out(port, 0);
	usleep(TCYCLE);
	port_out(port, METER_OFF);    
	val = 32;
	if (eightbit)
		val = val | 16;
	if (twoline)
		val = val | 8;
	if (!smallchar)
		val = val | 4;
	h = val>>4;
	l = val&15;
	/* 
	 * the following lines switch the LCD to 8 bit mode and provide
     * a clean start whether the program is started while the display is
     * in the 8 bit- or in the 4 bit-mode
     */
	port_out(port, METER_OFF | ENABLE | 48);
	usleep(TCYCLE);
	port_out(port, METER_OFF | 48);
	usleep(5000);
	port_out(port, METER_OFF | ENABLE | 48);
	usleep(TCYCLE);
	port_out(port, METER_OFF | 48);
	usleep(150);
	port_out(port, METER_OFF | ENABLE | 48);
	usleep(TCYCLE);
	port_out(port, METER_OFF | 48);
	/* 
	 * now the LCD is in 8 bit-mode
	 */	
	port_out(port, METER_OFF | ENABLE | h); /* set 4 bit mode */
	usleep(TCYCLE);
	port_out(port, METER_OFF | h);  
	/* 
	 * now the LCD is in 4 bit-mode 
	 */
    port_out(port, METER_OFF | ENABLE | h); /* set nr. lines & font */
	usleep(TCYCLE);
	port_out(port, METER_OFF | h);
	port_out(port, METER_OFF | ENABLE | l);
	usleep(TCYCLE);
	port_out(port, METER_OFF | l);
}

/*
 * move viewport or cursor right/left by one character
 * function : CURSOR or DISPLAY   currently only moves display
 */

void
lcd_shift(unsigned short int port, int direction, int function)
{
	unsigned char val, h, l;

	port_out(port, METER_OFF);
    val = 16 | 8;
	if (direction == RIGHT)
		val = val | 4;
	h = val>>4;
	l = val&15;    
	port_out(port, METER_OFF | ENABLE | h);
	usleep(TCYCLE);
	port_out(port, METER_OFF | l);
	port_out(port, METER_OFF | ENABLE | l);
	usleep(TCYCLE);
	port_out(port, METER_OFF | l);
}

/*
 * clear display
 */

void
lcd_clear(unsigned short int port)
{
	unsigned char h, l;

	port_out(port, METER_OFF);
	h = 0;
	l = 1;    
	port_out(port, METER_OFF | ENABLE | h);
	usleep(TCYCLE);
	port_out(port, METER_OFF | h);
	port_out(port, METER_OFF | ENABLE | l);
	usleep(TCYCLE);
	port_out(port, METER_OFF | l);
}

/* 
 * home cursor
 */

void
lcd_home(unsigned short int port)
{
	unsigned char h, l;
	mylcd.cur_x = mylcd.cur_y = 0;
	/* clear array here */
   
	port_out(port, METER_OFF);
	h = 0;
	l = 2 | 1;
	port_out(port, METER_OFF | ENABLE | h);
	usleep(TCYCLE);
	port_out(port, METER_OFF | h);
	port_out(port, METER_OFF | ENABLE | l);
	usleep(TCYCLE);
	port_out(port, METER_OFF | l);
}

/*
 * print a character at the current location
 * presume cursor location incremented to the next address
 */

void
lcd_print(unsigned short int port, char c)
{
	unsigned char h, l;

	mylcd.display[mylcd.cur_y][mylcd.cur_x] = c;
	mylcd.cur_x = (mylcd.cur_x + 1) % WIDTH;

	port_out(port, METER_OFF);
	h = c>>4;
	l = c&15;
	port_out(port, METER_OFF | RS | ENABLE | h);
	usleep(TCYCLE);
	port_out(port, METER_OFF | RS | h);
	port_out(port, METER_OFF | RS | ENABLE | l);
	usleep(TCYCLE);
	port_out(port, METER_OFF | RS | l);
}

/*
 * prints a string at the current location
 */

void
lcd_printstr(unsigned short int port, const char *str)
{
	int i;

	for (i = 0; str[i] != '\0'; ++i)
		lcd_print(port, str[i]);
}


/*
 * set character address, (0,0) is the home address
 * doesn't actually check if values are sane
 */

void
lcd_charaddr(unsigned short int port, int x, int y)
{
	int val, h, l;
	
	mylcd.cur_x = x;
	mylcd.cur_y = y;
    val = x + y * SEC_LINE_ADDR;
	h = (128 | ((char) val & 0x7f)) >>4;
	l = (128 | ((char) val & 0x7f)) &15;    
	port_out(port, METER_OFF);
	port_out(port, METER_OFF | ENABLE | h);
	usleep(TCYCLE);
	port_out(port, METER_OFF | h);
	port_out(port, METER_OFF | ENABLE | l);
	usleep(TCYCLE);
	port_out(port, METER_OFF | l);
}
