/**********************************************************************/
/*   TimeMon (c)  1994  Helmut Maierhofer			      */
/*   KDE-ified M. Maierhofer 1998                                     */
/**********************************************************************/

/*
 * timemon.h
 *
 * Definitions for the timemon widget.
 */

#include "../config.h"

#include <string.h>

#include <qpainter.h>

#include <kapp.h>
#include <kwm.h>
#include <kprocess.h>
#include <kmsgbox.h>

#include "timemon.h"
#include "sample.h"

#include "timemon.moc"

// -- global variables ---------------------------------------------------
KApplication *app;
KConfig *conf;

// -- KTimeMonWidget definition ------------------------------------------

// Initialise the member variables
KTimeMonWidget::KTimeMonWidget(KTimeMonWidget *d, QWidget *w) :
  QWidget(w), delegate(d), sample(0),
  kernelColour("red1"), userColour("#ff3c1a"),
  niceColour("#ff6060"), cachedColour("RoyalBlue1"),
  usedColour("blue1"), buffersColour("#3c58ff"), 
  swapColour("cyan3"), bgColour(white)
{
  conf->setGroup("Interface");
  kernelColour = conf->readColorEntry("KernelColour", &kernelColour);
  userColour = conf->readColorEntry("UserColour", &userColour);
  niceColour = conf->readColorEntry("NiceColour", &niceColour);
  cachedColour = conf->readColorEntry("CachedColour", &cachedColour);
  usedColour = conf->readColorEntry("UsedColour", &usedColour);
  buffersColour = conf->readColorEntry("BuffersColour", &buffersColour);
  swapColour = conf->readColorEntry("SwapColour", &swapColour);
  bgColour = conf->readColorEntry("BgColour", &bgColour);

  mode = conf->readBoolEntry("Mode", true);
  vertical = conf->readBoolEntry("Vertical", true);

  setBackgroundMode(NoBackground); // to avoid flicker
}

// Update colour settings with the new ones from the config dialog
void KTimeMonWidget::updateColour(KConfDialog *d)
{
  kernelColour = d->getKernelColour();
  userColour = d->getUserColour();
  niceColour = d->getNiceColour();
  cachedColour = d->getCachedColour();
  usedColour = d->getUsedColour();
  buffersColour = d->getBuffersColour();
  swapColour = d->getSwapColour();
  bgColour = d->getBgColour();
}

// Call the delegate's function (e.g. if we are panelised)
void KTimeMonWidget::mousePressEvent(QMouseEvent *event)
{
  if (delegate != 0) delegate->mousePressEvent(event);
}

// Switch the mode and redraw the widget.
void KTimeMonWidget::switchMode()
{
  mode = !mode;
  update();
}

// Draw part of a bar, depending on the bar orientation.
inline void KTimeMonWidget::paintRect(int x, int y, int w, int h, QColor c,
			       QPainter *p)
{
  if (vertical) 
    p->fillRect(x, y, w, h, c);
  else 
    p->fillRect(width() - y - h, x, h, w, c);
}

// Repaint the object; get the current sample and paint the bar graphs
// correspondingly. Use a pixmap to minimise flicker.
void KTimeMonWidget::paintEvent(QPaintEvent *)
{
  int w, h, x, y, b;

  w = vertical ? width() : height();
  h = vertical ? height() : width();

  b = (w - 2) / 3;		// bar width (1 pixel gap)

  x = 0;

  KSample::Sample s;

  if (sample != 0) s = sample->getSample(h);
  else {			// default values (for configuration)
    s.user = h * 40; s.user /= 100;
    s.nice = h * 25; s.nice /= 100;
    s.kernel = h * 15; s.kernel /= 100;
    s.buffers = h * 20; s.buffers /= 100;
    s.used = h * 30; s.used /= 100;
    s.cached = h * 20; s.cached /= 100;
    s.sused = h * 25; s.sused /= 100;
    s.pout = h * 10; s.pout /= 100;
    s.pin = h * 20; s.pin /= 100;
    s.swout = h * 30; s.swout /= 100;
    s.swin = h * 15; s.swin /= 100;
    s.cswitches = h * 75; s.cswitches /= 100;
  }

  QPixmap pixmap(width(), height());
  pixmap.fill(bgColour);

  QPainter painter(&pixmap);

  if (mode) {			// normal mode
    y = h - s.kernel; paintRect(x, y, b, s.kernel, kernelColour, &painter);
    y -= s.user; paintRect(x, y, b, s.user, userColour, &painter);
    y -= s.nice; paintRect(x, y, b, s.nice, niceColour, &painter);

    x += b + 1;
    y = h - s.used; paintRect(x, y, b, s.used, usedColour, &painter);
    y -= s.buffers; paintRect(x, y, b, s.buffers, buffersColour, &painter);
    y -= s.cached; paintRect(x, y, b, s.cached, cachedColour, &painter);

    x += b + 1;
    y = h - s.sused; paintRect(x, y, b, s.sused, swapColour, &painter);

  } else {			// extended mode

    y = h - s.pout; paintRect(x, y, b, s.pout, kernelColour, &painter);
    y -= s.pin; paintRect(x, y, b, s.pin, niceColour, &painter);

    x += b + 1;
    y = h - s.swout; paintRect(x, y, b, s.swout, cachedColour, &painter);
    y -= s.swin; paintRect(x, y, b, s.swin, usedColour, &painter);

    x += b + 1;
    y = h - s.cswitches; paintRect(x, y, b, s.cswitches, swapColour, &painter);
  }

  painter.end();

  bitBlt(this, 0, 0, &pixmap);
}

// -- KTimeMon definition ------------------------------------------------

// Initialise the member variables, read the configuration data base,
// set up the widget, and start the timer.
KTimeMon::KTimeMon() :
  sample(0), panelWidget(0), configDialog(0), bgProcess(0)
{
  mouseAction[0] = SWITCH;
  mouseAction[1] = NOTHING;
  mouseAction[2] = MENU;

  conf->setGroup("Parameters");
  panelised = conf->readBoolEntry("Panelised", false);
  interval = conf->readUnsignedNumEntry("Interval", 500);
  autoScale = conf->readBoolEntry("AutoScale", true);
  pageScale = conf->readUnsignedNumEntry("PageScale", 10);
  swapScale = conf->readUnsignedNumEntry("SwapScale", 5);
  ctxScale = conf->readUnsignedNumEntry("ContextScale", 300);
  for (int i = 0; i < MAX_MOUSE_ACTIONS; i++) {
    QString n;
    n.setNum(i);

    mouseAction[i] = (MouseAction) 
      conf->readUnsignedNumEntry(QString("MouseAction")+n, mouseAction[i]);
    mouseActionCommand[i] = conf->readEntry(QString("MouseActionCommand")+n, 
					    "");
  }

  sample = new KSample(this, autoScale, pageScale, swapScale, ctxScale);
  setSample(sample);

  connect(&timer, SIGNAL(timeout()), this, SLOT(timeout()));
  timer.start(interval);

  static char aboutmsg[400];	// big enough for the message
  sprintf(aboutmsg, i18n("%s v%s\n\nby M. Maierhofer (m.maierhofer@tees.ac.uk)\nBased on timemon by H. Maierhofer"), PACKAGE, VERSION);

  menu.insertItem(i18n("Help"), app->getHelpMenu(true, aboutmsg), 1);
  menu.insertItem(i18n("Settings..."), 2);
  menu.insertItem(i18n("Docked in panel"), 3);
  menu.insertItem(i18n("Horizontal bars"), 4);
  menu.insertSeparator();
  menu.insertItem(i18n("Quit"), 5);

  menu.connectItem(2, this, SLOT(configure()));
  menu.connectItem(3, this, SLOT(panelise()));
  menu.connectItem(4, this, SLOT(orientation()));
  menu.connectItem(5, this, SLOT(quit()));

  menu.setCheckable(true);

  QSize size(48, 48);
  resize(conf->readSizeEntry("WidgetSize", &size));

  panelised = !panelised;	// a bit dodgy; simulate menu selection to
  panelise();			// check whether we are in the panel or not...

  vertical = !vertical;		// and similar for orientation
  orientation();
}

// Kill the timer and delete the member variables
KTimeMon::~KTimeMon()
{
  stop();
  delete sample;
  delete panelWidget;
  delete configDialog;
  delete bgProcess;
}


// Apply the settings from the configuration dialog and save them.
void KTimeMon::apply()
{
				// this should really never be the case...
  if (configDialog == 0) configDialog = new KConfDialog(this);

  timer.stop();
  interval = configDialog->getInterval();
  timer.start(interval);

  updateColour(configDialog);
  if (panelWidget != 0) panelWidget->updateColour(configDialog);

  sample->setScaling(configDialog->getAutoScale(), 
		     configDialog->getPageScale(),
		     configDialog->getSwapScale(), 
		     configDialog->getCtxScale());

  for (int i = 0; i < MAX_MOUSE_ACTIONS; i++) {
    mouseAction[i] = (MouseAction) configDialog->getMouseAction(i);
    mouseActionCommand[i] = configDialog->getMouseActionCommand(i);
  }

  update();
  writeConfiguration();
}

// Dump the current configuration entries to the data base.
void KTimeMon::writeConfiguration()
{
  conf->setGroup("Interface");
  conf->writeEntry("KernelColour", kernelColour);
  conf->writeEntry("UserColour", userColour);
  conf->writeEntry("NiceColour", niceColour);
  conf->writeEntry("CachedColour", cachedColour);
  conf->writeEntry("UsedColour", usedColour);
  conf->writeEntry("BuffersColour", buffersColour);
  conf->writeEntry("SwapColour", swapColour);
  conf->writeEntry("BgColour", bgColour);
  conf->writeEntry("Mode", mode);
  conf->writeEntry("Vertical", vertical);

  conf->setGroup("Parameters");
  conf->writeEntry("Panelised", panelised);
  conf->writeEntry("Interval", interval);
  conf->writeEntry("AutoScale", autoScale);
  conf->writeEntry("PageScale", pageScale);
  conf->writeEntry("SwapScale", swapScale);
  conf->writeEntry("ContextScale", ctxScale);
  conf->writeEntry("WidgetSize", size());
  for (int i = 0; i < MAX_MOUSE_ACTIONS; i++) {
    QString n;
    n.setNum(i);

    conf->writeEntry(QString("MouseAction")+n, (unsigned)mouseAction[i]);
    conf->writeEntry(QString("MouseActionCommand")+n, mouseActionCommand[i]);
  }
}

// Make the KSample object update its internal sample and repaint the
// object.
void KTimeMon::timeout()
{
  if (sample != 0) sample->updateSample();
  if (panelWidget != 0) panelWidget->update();
  else update();
}

// This is called when the session management strikes, and also when the
// main program exits with a code of 0 (i.e. there was no error).
void KTimeMon::save()
{
  writeConfiguration();
  conf->sync();
}

// Switch between panel mode and normal window mode.
void KTimeMon::panelise()
{
  panelised = !panelised;
  menu.setItemChecked(3, panelised);

  if (panelised) {
    hide();
    panelWidget = new KTimeMonWidget(this);
    panelWidget->setSample(sample);
    panelWidget->mode = mode;
    panelWidget->vertical = vertical;
    KWM::setDockWindow(panelWidget->winId());
    panelWidget->show();
  } else {
    show();
    delete panelWidget;
    panelWidget = 0;
  }

  if (panelWidget != 0) panelWidget->update();
  else update();
}

// Update the configuration dialog with the current values and show it.
void KTimeMon::configure()
{
  if (configDialog == 0) configDialog = new KConfDialog(this);
  configDialog->update();
  configDialog->show();
}

void KTimeMon::orientation()
{
  vertical = !vertical;
  if (panelWidget != 0) panelWidget->vertical = vertical;

  menu.setItemChecked(4, !vertical);

  if (panelWidget != 0) panelWidget->update();
  else update();
}

// Pop up the menu when the appropriate button has been pressed.
void KTimeMon::mousePressEvent(QMouseEvent *event)
{
  if (event == 0) return;

  int index = -1;
  if (event->button() == LeftButton) index = 0;
  else if (event->button() == MidButton) index = 1;
  else if (event->button() == RightButton) index = 2;

  if (index == -1) return;

  switch (mouseAction[index]) {
  case NOTHING: 
    break;
  case SWITCH: 
    switchMode(); 
    if (panelWidget != 0) panelWidget->switchMode();
    break;
  case MENU:
    { QWidget *w = (panelWidget != 0 ? panelWidget : this);
      menu.popup(w->mapToGlobal(event->pos())); }
    break;
  case COMMAND:
    runCommand(index);
    break;
  }
}

// Start the given command
void KTimeMon::runCommand(int index)
{
				// just in case it still hangs around
  if (bgProcess != 0) delete bgProcess;

  bgProcess = new KShellProcess;
  *bgProcess << mouseActionCommand[index];
  connect(bgProcess, SIGNAL(receivedStderr(KProcess *, char *, int)), 
	  this, SLOT(commandStderr(KProcess *, char *, int)));
  bgProcess->start(KProcess::DontCare, KProcess::Stderr);
}

// Check if there is any diagnostic output (command not found or such)
void KTimeMon::commandStderr(KProcess *proc, char *buffer, int length)
{
  char msgbuf[1024];
  strcpy(msgbuf, i18n("Got diagnostic output from child command:\n\n"));

  int l = sizeof(msgbuf) - strlen(msgbuf);
  if (l > length) l = length;

  strncat(msgbuf, buffer, l);
  KMsgBox::message(this, i18n("KTimeMon notice"), msgbuf,KMsgBox::EXCLAMATION);
}
