/**********************************************************************/
/*   TimeMon (c)  1994  Helmut Maierhofer			      */
/*   KDE-ified M. Maierhofer 1998                                     */
/**********************************************************************/

/*
 * sample.cc
 *
 * Definitions for the system dependent sampling class.
 */

#ifdef TEST_SAMPLE		// define this for a text based test version
# define HAVE_FCNTL_H
//# define HAVE_SNPRINTF
# define i18n(X) X
#else
# include "../config.h"
#endif

#include <unistd.h>
#include <stdlib.h>
#include <errno.h>
#include <string.h>
#ifdef HAVE_FCNTL_H
#include <fcntl.h>
#endif
#include <fstream.h>
#include <stdio.h>

#ifdef __osf__
#include <sys/table.h>
#elif defined(__sun__)
#include <kstat.h>
#include <sys/sysinfo.h>
#include <sys/stat.h>
#include <sys/swap.h>
#endif

#ifndef TEST_SAMPLE
# include <qwidget.h>

# include <kmsgbox.h>
# include <kapp.h>

# include "timemon.h"
#endif

#include "sample.h"

// -- fake definitions ---------------------------------------------------

#ifdef TEST_SAMPLE
class KTimeMon { public: KTimeMon() {} void stop() {} void cont() {} };

class KMsgBox { public: enum Code { STOP };
  static void message(KTimeMon *,char *,char *b,Code,char *){cerr<<b<<endl;} };
#endif

// -- global definitions -------------------------------------------------

#ifndef HAVE_SNPRINTF
#include <stdarg.h>
static int snprintf(char *b, int, char *fmt, ...)
{ va_list ap; va_start(ap, fmt); 
  int r = vsprintf(b, fmt, ap); va_end(ap); return r; }
#endif

#ifdef linux
// -- global constants ---------------------------------------------------
static const char *STAT_NAME = "stat";
static const char *MEMINFO_NAME = "meminfo";
static const char *MTAB_NAME = "/etc/mtab";
#endif

// -- KSample definition -------------------------------------------------

// Initialise the member variables and try to open the standard files in
// the proc filesystem
KSample::KSample(KTimeMon *t, bool a, unsigned p, unsigned s, unsigned c) :
  timemon(t), 
#ifdef linux
  memFD(-1), statFD(-1),
#elif defined (__sun__)
  kc(0), warned(false),
#endif
  pageScale(p), swapScale(s), cxScale(c), autoscale(a)
{
#ifdef linux

  parseMtab(proc);

  char msg[1024];
  char file[512];
  snprintf(file, sizeof(file), "%s/%s", proc, MEMINFO_NAME);

  if ((memFD = open(file, O_RDONLY)) == -1) {
    snprintf(msg, sizeof(msg), i18n("Sorry, I cannot open the file `%s'.\nThe diagnostics are: %s.\n\nI need this file to determine current memory usage.\nMaybe your proc filesystem is non Linux-standard?"), file, strerror(errno));
				    
    KMsgBox::message(timemon, i18n("KTimeMon notice"), msg, KMsgBox::STOP, 
		     i18n("Quit"));
    exit(1);
  }

  snprintf(file, sizeof(file), "%s/%s", proc, STAT_NAME);
  if ((statFD = open(file, O_RDONLY)) == -1) {
    snprintf(msg, sizeof(msg), i18n("Sorry, I cannot open the file `%s'.\nThe diagnostics are: %s.\n\nI need this file to determine current system info.\nMaybe your proc filesystem is non Linux-standard?"), file, strerror(errno));

    KMsgBox::message(timemon, i18n("KTimeMon notice"), msg, KMsgBox::STOP, 
		     i18n("Quit"));
    exit(1);
  }

#elif defined (__sun__)
  char msg[1024];

  if ((kc = kstat_open()) == 0) {
    snprintf(msg, sizeof(msg), i18n("Sorry, I cannot initialise the `kstat' library.\nThis library is used for accessing kernel information.\nThe diagnostics are: `%s'.\n\nAre you running Solaris at all?\nYou may want to mail me at m.maierhofer@tees.ac.uk,\nI will try to figure out what went wrong."), 
					  strerror(errno));
    KMsgBox::message(timemon, i18n("KTimeMon notice"), msg, KMsgBox::STOP,
		     i18n("Quit"));
    exit(1);
  }
#endif

  readSample();
  updateSample();
}

// Close the files.
KSample::~KSample()
{
#ifdef linux
  close(memFD);
  close(statFD);
#elif defined (__sun__)
  if (kc != 0) kstat_close(kc);
#endif
}

#ifdef linux
// Parse /etc/mtab to determine the proc filesystem mount point.
void KSample::parseMtab(char *dest)
{
  char line[1024];

  ifstream *mtab = new ifstream(MTAB_NAME);
  if (!mtab->good()) {
    snprintf(line, sizeof(line), i18n("I cannot open file `%s' to determine where the proc\nfilesystem is mounted. The diagnostics are:\n %s\n\nAre you running UNIX at all?!?"), MTAB_NAME, strerror(errno));
    KMsgBox::message(timemon, i18n("KTimeMon notice"), line, KMsgBox::STOP, 
		     i18n("Quit"));
    exit(1);
  }
    
  unsigned lineno = 0;

  for (;;) {
    lineno++;
    mtab->getline(line, sizeof(line));

    if (mtab->bad()) {
      snprintf(line, sizeof(line), i18n("I cannot read file `%s' to determine where the proc\nfilesystem is mounted. The diagnostics are:\n %s"), 
	       MTAB_NAME, strerror(errno));
      break;
    }

    if (mtab->eof()) {
      snprintf(line, sizeof(line), i18n("Hmm, I could not figure out where the proc filesystem\nis mounted (there is no entry in `%s').\n\nI need information from the proc filesystem to determine\ncurrent system usage. Maybe you are not running Linux (I'm\nafraid the proc filesystem is Linux specific)?\n\nIf you can provide help with porting KTimeMon to your\nplatform, please contact me at <m.maierhofer@tees.ac.uk>"), MTAB_NAME);
      break;
    }

    if (mtab->fail()) {
      snprintf(line, sizeof(line), i18n("Hmm, I encountered a very long line while reading\ninformation in `%s' (where \"very long\" is defined\nas > %u). This happened at line %u.\n\nIs %s the mount table on your platform?\n\n"),
	       MTAB_NAME, sizeof(line), lineno, MTAB_NAME);
      break;
    }

    char *p, *m;

    if ((p = m = strchr(line, ' ')) != 0) p = strchr(m+1, ' ');
    if (p == 0 || strncmp(p+1, "proc ", 5) != 0) continue;

    *p = '\0';
    strncpy(dest, m+1, 256);
    mtab->close();
    delete mtab;
    return;
  }

  KMsgBox::message(timemon, i18n("KTimeMon notice"), line, KMsgBox::STOP, 
		   i18n("Quit"));
  exit(1);
}
#endif

// Set the appropriate scaling parameters
void KSample::setScaling(bool a, unsigned p, unsigned s, unsigned c)
{
  autoscale = a;
  pageScale = p;
  swapScale = s;
  cxScale = c;
}


// Read a new sample from the files.
void KSample::readSample()
{
  char buffer[4096];
  sample.cpus = 0;		// just to make sure...

#ifdef linux
  int l;

  lseek(memFD, 0, 0);
  if ((l = read(memFD, buffer, sizeof(buffer) - 1)) < 0) {
    timemon->stop();
    snprintf(buffer, sizeof(buffer), i18n("Sorry, I cannot read the memory usage file `%s/%s'.\nThe diagnostics are: %s"), proc, MEMINFO_NAME, 
	     strerror(errno));
    KMsgBox::message(timemon, i18n("KTimeMon notice"), buffer, KMsgBox::STOP,
		     i18n("Quit"));
    exit(1);
  }
  buffer[l] = '\0';

  bool ok = true;

				// use the new interface to /proc/meminfo
				// code courtesy of Dirk A. Mueller
  char *p = strstr(buffer, "MemTotal:");
  if (p == 0 || sscanf(p, "MemTotal: %lu kB\nMemFree: %lu kB\n"
		       "MemShared: %*u kB\nBuffers: %lu kB\nCached: %lu kB\n"
                       "SwapTotal: %lu kB\n SwapFree: %lu kB",
                       &sample.mtotal, &sample.free, &sample.buffers,
		       &sample.cached, &sample.stotal, &sample.sfree) < 6) {
    ok = false;
  }

  if (!ok) {
    timemon->stop();
    snprintf(buffer, sizeof(buffer), i18n("Sorry, the memory usage file `%s/%s'\nseems to use a different file format than I expect.\n\nMaybe your version of the proc filesystem is\nincompatible with mine. Mail me at\nm.maierhofer@tees.ac.uk and I will try to sort this out."), proc, MEMINFO_NAME);

    KMsgBox::message(timemon, i18n("KTimeMon notice"), buffer, KMsgBox::STOP,
		     i18n("Quit"));
    exit(1);
  }

  lseek(statFD, 0, 0);
  if ((l = read(statFD, buffer, sizeof(buffer)-1)) < 0) {
    timemon->stop();
    snprintf(buffer, sizeof(buffer), i18n("Sorry, I cannot read the system usage file `%s/%s'.\nThe diagnostics are: %s"), proc, STAT_NAME, strerror(errno));
	     
    KMsgBox::message(timemon, i18n("KTimeMon notice"), buffer, KMsgBox::STOP,
		     i18n("Quit"));
    exit(1);
  }
  buffer[l] = '\0';

  ok = (sscanf(buffer, "cpu %lu %lu %lu %lu\n", &sample.user, &sample.nice, 
	       &sample.kernel, &sample.idle) == 4);

  if (ok) {
    for (l = 0; l < MAX_CPU; l++) { // get individual stat for SMP machines
      char cpuname[10];
      sprintf(cpuname, "cpu%d", l);

      if ((p = strstr(buffer, cpuname)) == NULL) break;

      unsigned long u, n, k, i;
      ok = sscanf(p, "cpu%*d %lu %lu %lu %lu\n", &u, &n, &k, &i);
      if (!ok) break;

      sample.smptotal[l] = u+n+k+i;
      sample.smpbusy[l] = sample.smptotal[l] - i;
    }
  }
  sample.cpus = l;

  if (ok) {
    if ((p = strstr(buffer, "page")) == NULL) ok = false;
    else ok = (sscanf(p, "page %lu %lu\n", &sample.pin, &sample.pout) == 2);
  }

  if (ok) {
    if ((p = strstr(buffer, "swap")) == NULL) ok = false;
    else ok = (sscanf(p, "swap %lu %lu\n", &sample.swin, &sample.swout) == 2);
  }

  if (ok) {
    if ((p = strstr(buffer, "ctxt")) == NULL) ok = false;
    else ok = (sscanf(p, "ctxt %lu\n", &sample.cswitches) == 1);
  }

  if (!ok) {
    timemon->stop();
    snprintf(buffer, sizeof(buffer), i18n("Sorry, the system usage file `%s/%s'\nseems to use a different file format than I expect.\n\nMaybe your version of the proc filesystem is\nincompatible with mine. Mail me at\nm.maierhofer@tees.ac.uk and I will try to sort this out."), proc, STAT_NAME);
	     
    KMsgBox::message(timemon, i18n("KTimeMon notice"), buffer, KMsgBox::STOP,
		     i18n("Quit"));
    exit(1);
  }

#elif defined(__osf__)

#define MAKE_ERROR(n) { \
  timemon->stop(); \
  snprintf(buffer, sizeof(buffer), i18n("Sorry, I cannot get system " \
    "information.\n\nThe table(2) system call returned error\n `%s' for " \
    "table %s.\n\nYou may want to mail me at m.maierhofer@tees.ac.uk, " \
    "I will try to figure out what went wrong."), strerror(errno), n); \
  KMsgBox::message(timemon, i18n("KTimeMon notice"), buffer, KMsgBox::STOP, \
		   i18n("Quit")); \
  exit(1); }

  struct tbl_sysinfo sysinfo;
  if (table(TBL_SYSINFO, 0, &sysinfo, 1, sizeof(sysinfo)) != 1)
    MAKE_ERROR("TBL_SYSINFO");

  sample.user = sysinfo.si_user;
  sample.nice = sysinfo.si_nice;
  sample.kernel = sysinfo.si_sys;
  sample.idle = sysinfo.si_idle + sysinfo.wait;

  struct tbl_vmstats vmstats;
  if (table(TBL_VMSTATS, 0, &vmstats, 1, sizeof(vmstats)) != 1)
    MAKE_ERROR("TBL_VMSTATS");

  sample.mtotal = vmstats.free_count + vmstats.active_count +
    vmstats.inactive_count + vmstats.wire_count;
  sample.free = vmstats.free_count;
  sample.buffers = vmstats.inactive_count; // pages not used for some time
  sample.cached = vmstats.wire_count; // kernel/driver memory

  struct tbl_swapinfo swapinfo;
  if (table(TBL_SWAPINFO, -1, &swapinfo, 1, sizeof(swapinfo)) != 1)
    MAKE_ERROR("TBL_SWAPINFO");

  sample.stotal = swapinfo.size;
  sample.sfree = swapinfo.free;

  struct tbl_paging paging;
  if (table(TBL_PAGING, 0, &paging, 1, sizeof(paging)) != 1)
    MAKE_ERROR("TBL_PAGING");

  sample.pin = sample.swin = paging.v_pgpgin; // MM no separate swap
  sample.pout = sample.swout = paging.v_pgout;

  struct tbl_intr intr;
  if (table(TBL_INTR, 0, &intr, 1, sizeof(intr)) != 1)
    MAKE_ERROR("TBL_INTR");

  sample.cswitches = intr.in_context;
#undef MAKE_ERROR

#elif defined(__sun__)
  kstat_t *ksp;

  sample.cpus = 0;
  for (ksp = kc->kc_chain; ksp != 0; ksp = ksp->ks_next) {
    if (strncmp(ksp->ks_name, "cpu_stat", 8) != 0) continue;
    sample.cpus++;
  }

  if (sample.cpus == 0) {
    timemon->stop();
    snprintf(buffer, sizeof(buffer), i18n("Sorry, I cannot find any entries for CPU statistics\nin the `kstat' library. Are you running a non-standard\nversion of Solaris?\n\nPlease contact me at m.maierhofer@tees.ac.uk and I will try to sort this out."));
    KMsgBox::message(timemon, i18n("KTimeMon notice"), buffer, KMsgBox::STOP,
		     i18n("Quit"));
    exit(1);
  }

  sample.user = sample.nice = sample.kernel = sample.idle = 0;
  sample.pin = sample.pout = sample.swin = sample.swout = 0;
  sample.cswitches = 0;
  sample.stotal = sample.sfree = 0;

  int cpus = 0;
  for (ksp = kc->kc_chain; ksp != 0; ksp = ksp->ks_next) {
    if (strncmp(ksp->ks_name, "cpu_stat", 8) != 0) continue;
    cpus++;

    cpu_stat_t cstat;
    if (kstat_read(kc, ksp, 0) == -1 || // update from kernel
	kstat_read(kc, ksp, &cstat) == -1) { // and read into buffer
      timemon->stop();
      snprintf(buffer, sizeof(buffer), i18n("Sorry, I cannot read the CPU statistics entry\nfrom the `kstat' library. The diagnostics are `%s'.\n\nYou might want to contact me at\nm.maierhofer@tees.ac.uk and I will try to sort this out."), strerror(errno));
      KMsgBox::message(timemon, i18n("KTimeMon notice"), buffer, KMsgBox::STOP,
		       i18n("Quit"));
      exit(1);
    }

  // fields are: idle user kernel iowait
    sample.user += cstat.cpu_sysinfo.cpu[1] / sample.cpus;
    sample.nice += cstat.cpu_sysinfo.cpu[3] / sample.cpus;
    sample.kernel += cstat.cpu_sysinfo.cpu[2] / sample.cpus;
    sample.idle += cstat.cpu_sysinfo.cpu[0] / sample.cpus;

    sample.pin += cstat.cpu_vminfo.pgin / sample.cpus;
    sample.pout += cstat.cpu_vminfo.pgout / sample.cpus;
    sample.swin += cstat.cpu_vminfo.swapin / sample.cpus;
    sample.swout += cstat.cpu_vminfo.swapout / sample.cpus;

    sample.cswitches += cstat.cpu_sysinfo.pswitch / sample.cpus;
  }
  
  if (cpus != sample.cpus) { 
    timemon->stop();
    snprintf(buffer, sizeof(buffer), i18n("Huhh, now this is very strange.\nThe number of CPUs seems to have changed at\nvery short notice. Or the `kstat' library returns\ninconsistent results (%d vs. %d CPUs).\nPlease contact me at m.maierhofer@tees.ac.uk and I will try to sort this out."), sample.cpus, cpus);
      KMsgBox::message(timemon, i18n("KTimeMon notice"), buffer, KMsgBox::STOP,
		       i18n("Quit"));
      exit(1);
  }

  // availrmem = pages of core for user-proc ( == physmem - kernelmem)
  // freemem = no of free pages
  // physmem == total mem in 4KB blocks

  errno = 0;
  if ((ksp = kstat_lookup(kc, "unix", -1, "system_pages")) == 0 ||
      kstat_read(kc, ksp, 0) == -1) {
    snprintf(buffer, sizeof(buffer), i18n("Sorry, I cannot read the memory statistics entry\nfrom the `kstat' library. The diagnostics are `%s' and ksp is 0x%x.\n\nYou might want to contact me at\nm.maierhofer@tees.ac.uk and I will try to sort this out."), strerror(errno), ksp);
      KMsgBox::message(timemon, i18n("KTimeMon notice"), buffer, KMsgBox::STOP,
		       i18n("Quit"));
      exit(1);
  }

  int i;
  unsigned long physmem = 0, freemem = 0, availrmem = 0;

  kstat_named_t *kn = (kstat_named_t *)ksp->ks_data;
  for (i = 0; i < (int) ksp->ks_ndata; i++) {
    if (strcmp(kn->name, "physmem") == 0) physmem = kn->value.ul;
    else if (strcmp(kn->name, "freemem") == 0) freemem = kn->value.ul;
    else if (strcmp(kn->name, "availrmem") == 0) availrmem = kn->value.ul;
    kn++;
  }

  if (physmem == 0) {		// sanity check, this should always be > 0
    timemon->stop();
    snprintf(buffer, sizeof(buffer), i18n("Uh uh, there seems to be a problem with my handling\nof the `kstat' library: I determined 0 physical memory!\n(Free memory is %u, available memory is %u.\n\nPlease contact me at m.maierhofer@tees.ac.uk and I will try to sort this out."), freemem, availrmem);
      KMsgBox::message(timemon, i18n("KTimeMon notice"), buffer, KMsgBox::STOP,
		       i18n("Quit"));
      exit(1);
  }

  sample.mtotal = physmem;
  sample.free = freemem;
  sample.buffers = 0;
  sample.cached = physmem - availrmem; // memory used by the kernel

  int swapentries;
  if ((swapentries = swapctl(SC_GETNSWP, 0)) == -1) {
    timemon->stop();
    snprintf(buffer, sizeof(buffer), i18n("Sorry, cannot determine the number of\nswap spaces. The diagnostics are `%s'.\n\nPlease contact me at m.maierhofer@tees.ac.uk and I will try to sort this out."), strerror(errno));
      KMsgBox::message(timemon, i18n("KTimeMon notice"), buffer, KMsgBox::STOP,
		       i18n("Quit"));
      exit(1);
  }

  if (swapentries != 0) {
				// 2* to get some space for padding??
    swaptbl_t *stbl = (swaptbl_t *) malloc(2*sizeof(int) + swapentries * 
    					   sizeof(struct swapent));
    if (stbl == 0) {
      timemon->stop();
      snprintf(buffer, sizeof(buffer), i18n("Sorry, I ran out of memory while\ntrying to determine the swap usage.\nI tried to allocate %d bytes of memory (2 * %d + %d * %d).\n\nPlease contact me at m.maierhofer@tees.ac.uk and I will try to sort this out."), 2 * sizeof(int) + swapentries * sizeof(struct swapent),
	       sizeof(int), swapentries, sizeof(struct swapent));
      KMsgBox::message(timemon, i18n("KTimeMon notice"), buffer, KMsgBox::STOP,
		       i18n("Quit"));
      exit(1);
    }

    char path[1024];
    stbl->swt_n = swapentries;
    for (i = 0; i < swapentries; i++) stbl->swt_ent[i].ste_path = path;

    if ((swapentries = swapctl(SC_LIST, stbl)) == -1) {
      timemon->stop();
      snprintf(buffer, sizeof(buffer), i18n("Sorry, I cannot determine the swap usage.\nThe diagnostics are `%s'.\n\nPlease contact me at m.maierhofer@tees.ac.uk and I will try to sort this out."), strerror(errno));
      KMsgBox::message(timemon, i18n("KTimeMon notice"), buffer, KMsgBox::STOP,
		       i18n("Quit"));
      exit(1);
    }

    if (!warned && swapentries != stbl->swt_n) {
      warned = true;
      timemon->stop();
      snprintf(buffer, sizeof(buffer), i18n("Strange, I requested information for\n%d swap spaces, but only got %d swap entries back.\nI am a bit confused now, but I will try to continue.\n\nPlease contact me at m.maierhofer@tees.ac.uk and I will try to sort this out."), stbl->swt_n, swapentries);
      KMsgBox::message(timemon, i18n("KTimeMon notice"), buffer, KMsgBox::STOP,
		       i18n("Quit"));
      timemon->cont();
    }

    for (i = 0; i < swapentries; i++) {
      sample.stotal += stbl->swt_ent[i].ste_pages;
      sample.sfree += stbl->swt_ent[i].ste_free;
    }

    free(stbl);
  }

#endif

  sample.cputotal = sample.user + sample.nice + sample.kernel + sample.idle;
  sample.used = sample.mtotal - sample.free - sample.buffers - sample.cached;
  sample.sused = sample.stotal - sample.sfree;
}

// Read a new sample after copying the old one.
void KSample::updateSample()
{
  oldSample = sample;
  readSample();
}

// Better scaling, round according to first decimal
inline unsigned long KSample::doScale(unsigned long value, unsigned scale1,
				      unsigned long scale2)
{
  if (scale2 == 0) scale2 = (unsigned long)~0; // avoid SEGVs

  unsigned long v = value * scale1 * 10;
  v /= scale2;
  unsigned r = v % 10;
  v /= 10;
  if (r > 4) v++;
  return v;
}

// Provide the difference from the last to the current sample, scale it
// and return it.
KSample::Sample KSample::getSample(unsigned scale)
{
  Sample diff = sample;

  diff.cputotal -= oldSample.cputotal;

  diff.user -= oldSample.user; 
  diff.nice -= oldSample.nice; 
  diff.kernel -= oldSample.kernel; 

  diff.user = doScale(diff.user, scale, diff.cputotal);
  diff.nice = doScale(diff.nice, scale, diff.cputotal);
  diff.kernel = doScale(diff.kernel, scale, diff.cputotal);

  int i;
  for (i = 0; i < diff.cpus; i++) {
    diff.smptotal[i] -= oldSample.smptotal[i];
    diff.smpbusy[i] -= oldSample.smpbusy[i]; 
    diff.smpbusy[i] = doScale(diff.smpbusy[i], scale, diff.smptotal[i]);
  }

  diff.cached = doScale(diff.cached, scale, diff.mtotal);
  diff.buffers = doScale(diff.buffers, scale, diff.mtotal);
  diff.used = doScale(diff.used, scale, diff.mtotal);

  diff.sused = doScale(diff.sused, scale, diff.stotal);

  diff.pin -= oldSample.pin; diff.pin *= scale;
  diff.pout -= oldSample.pout; diff.pout *= scale;
  unsigned page = (diff.pin + diff.pout) / 2;

  if (autoscale && page > 0 &&		// autoscale
      (page / pageScale > scale / 2 || page / pageScale < 1)) {
    pageScale = page / (scale/4);
  }
  diff.pin = doScale(diff.pin, 1, pageScale);
  diff.pout = doScale(diff.pout, 1, pageScale);

  diff.swin -= oldSample.swin; diff.swin *= scale; 
  diff.swout -= oldSample.swout; diff.swout *= scale; 
  unsigned swap = (diff.swin + diff.swout) / 2;

  if (autoscale && swap > 0 &&		// autoscale
      (swap / swapScale > scale / 2 || swap / swapScale < 1)) {
    swapScale = swap / (scale/4);
  }
  diff.swin = doScale(diff.swin, 1, swapScale);
  diff.swout = doScale(diff.swout, 1, swapScale);

  diff.cswitches -= oldSample.cswitches; diff.cswitches *= scale;
  if (autoscale && diff.cswitches > 0 &&	// auto scale
      (diff.cswitches / cxScale > scale || diff.cswitches / cxScale < 2)) {
    cxScale = diff.cswitches/(scale/2);
  }
  diff.cswitches = doScale(diff.cswitches, 1, cxScale);

  return diff;
}

// -- fake main ----------------------------------------------------------

#ifdef TEST_SAMPLE

int main()
{
  KTimeMon t;
  KSample ks(&t, true, 1, 1, 1);

  cout << "scale 100, press Ctrl-C to terminate..." << endl;
  for (;;) {
    ks.updateSample();
    KSample::Sample s = ks.getSample(100);

    cout << "cpu: kernel " << s.kernel << " user " << s.user << " nice " <<
      s.nice << "; sum: " << s.kernel+s.user+s.nice << endl;
    cout << "mem: used " << s.used << " buffers " << s.buffers << 
      " cached " << s.cached << "; sum: " << s.used+s.buffers+s.cached << endl;
    cout << "swap: used " << s.sused << endl;

    cout << "page: in " << s.pin << " out " << s.pout << "; sum: " <<
      s.pin+s.pout << endl;
    cout << "swap: in " << s.swin << " out " << s.swout << "; sum: " <<
      s.swin+s.swout << endl;
    cout << "cxsw: " << s.cswitches << endl;
    
    cout << "---------------------------------------------------------" <<endl;

    sleep(1);
  }

  return 0;
}

#endif

