/****************************************************************
**
** Implementation of LPanel class
**
****************************************************************/

#include "lpanel.h"

extern "C" rand();


const int LPanel::QSECOND   = 1000;  // 1000 milliseconds make one second
const int LPanel::MIN_WIDTH  = 10;
const int LPanel::MIN_HEIGHT = 6;

// Constructor
LPanel::LPanel(int n, char *d,QWidget *parent, const char *name )
        : QWidget( parent, name )
{
    numPoints = 0;
    numGraphs = n;
    description = d;
    yScale = 1.0;
    xScale = 1;  // FIXME: this is not yet correctly implemented!
    redrawInterval = 1 * QSECOND; // in milliseconds
    
    incremental = true;
    filled = true;
    
    setMinimumSize(MIN_WIDTH,MIN_HEIGHT);
    
    colors = new QColor[numGraphs];
    setBackgroundColor(QColor(gray));
    
    scaleValue = -1; // DEFAULT: Do not draw scale lines
    scaleColor = QColor(black);
 
    data = new FArray[numGraphs];
    for (int i = 0;i < numGraphs;i++) {
        data[i] = FArray(width());
    }
    
    QToolTip::add(this,description);
}


// Destructor 
LPanel::~LPanel() 
{
    delete[] data;
    delete[] colors;
}


// This is the slot hit by external events to regain data and to redraw
void LPanel::redraw()
{
    resample();
    repaint();
}


void LPanel::timerEvent(QTimerEvent *)
{
    redraw();
}


// Do the hard part: draw panel using actual geometry, colors, scale and data
void LPanel::paintEvent(QPaintEvent *)
{
    QPixmap  pix(this->size() );  
    QPainter p, panelPainter(this);
      
    int w = width(), h = height();
    int x1,y1,x2,y2;
    float yMax;
    
    // Compute max. y value
    yMax = -1;
    for (int i = 0;i < numGraphs;i++) {
        for (int j = 0;j < numPoints;j++) {
            if (data[i][j] > yMax)  
                 yMax = data[i][j];
        }
    }
    
    
    if (scaleValue > 0.0) {
        yScale = int(yMax + 1.0);
    }
    // else: yScale remains unchanged
        
    pix.fill(this,0,0);
    p.begin(&pix);

    p.translate(0,height());
    p.scale(1,-1.0 / yScale);
   
    for (int i = 0;i < numGraphs;i++) {
        QPen pen(colors[i]);
        p.setPen(pen);
        
        if (filled) {          // always incremental
            for (int j = 1;j < numPoints;j++) {
                
               
                if (i == 0) {
                    p.drawLine(j, 0, j, int(h * data[i][j] + 0.5) ); 
                }
                else {
                    p.drawLine(j, int(h * data[i-1][j] + 0.5), 
                               j, int(h * data[i][j]   + 0.5));
                }
            }
        }
        else {
            x1 = 0;
            y1 = int(h * data[i][0]);
        
            p.moveTo(x1,y1);
            for (int j = 1;j < numPoints;j++) {
                x2 = j * xScale;
                y2 = int(h * data[i][j]);
                p.lineTo(x2,y2);
                x1 = x2;
                y1 = y2;
            }
        }
    }
    
    if (scaleValue > 0) { // Draw scale lines!
        p.setPen(QPen(scaleColor));
        
        for (float i = scaleValue;i < yScale;i+= scaleValue) {
            p.drawLine(0, int(i * h) , w, int(i * h));
       }
    }
    
    p.end();
    panelPainter.drawPixmap(0,0,pix);
}

// Reallocate internal data structures 
// No need to deal with changes in height
void LPanel::resizeEvent(QResizeEvent *re)
{
    int old_w = re->oldSize().width();
    int new_w = re->size().width();
    
    if (new_w < old_w) {
        if (new_w < numPoints) {
            lshift(numPoints - new_w);
        }
    }
    
    if (new_w != old_w) {
        for (int i = 0;i < numGraphs;i++) {
            data[i].resize(new_w);
        }
    }
}


// left shift data by <num> points
void LPanel::lshift(int num)
{
    for (int i = 0;i < numGraphs;i++) {
        for (int j = 0;j < (numPoints - num);j++) {
            data[i][j] = data[i][j+num];
        }
    }
    numPoints -= num;
}


// Prepare for Insertion of next Column, i.e. shift left by one if
// necessary
void LPanel::nextColumn()
{
    if (numPoints > width()) {
        cerr << "LPanel::nextColumn(): numPoints > width()" << endl;
        exit(-1);
    }
    else if (numPoints == width()) {
        lshift(1);
    }
}


// This is the very function for derived classes to append new data!
void LPanel::append(FArray &r)
{
    if (width() == 0)
        return;
    
    nextColumn();
    
    ;
    float sum = 0.0;
    for (int i = 0;i < numGraphs;i++) {
        data[i][numPoints] = sum + r[i];
        if (incremental) {
            sum += r[i];
        }
    }
    numPoints++;
}



