/*
 * Configurable ps-like program.
 * Useful utility routines.
 *
 * Copyright (c) 1995 David I. Bell
 * Permission is granted to use, distribute, or modify this source,
 * provided that this copyright notice remains intact.
 */

#include <pwd.h>
#include <grp.h>
#include <sys/stat.h>
#include <dirent.h>

#include "ips.h"


/*
 * Definitions for collecting and looking up user names, group names,
 * and device names.
 */
#define	MAX_NAME_LEN	8	/* length of user or group name */
#define	MAX_DEV_LEN	16	/* length of device name */
#define	USERNAME_ALLOC	100	/* reallocation size for user names */
#define	GROUPNAME_ALLOC	20	/* reallocation size for group names */
#define	DEVNAME_ALLOC	100	/* reallocation size for device names */


/*
 * Structure holding login or group names corresponding to numeric ids.
 */
typedef	struct
{
	int	id;				/* user or group id */
	char	name[MAX_NAME_LEN + 2];		/* user or group name */
} NAME;


/*
 * Structure to hold information about devices.
 * This is only used for character devices.
 */
typedef	struct	{
	int	id;			/* device id */
	dev_t	dev;			/* dev device is on */
	ino_t	inode;			/* inode device is on */
	char	name[MAX_DEV_LEN + 2];	/* name of device */
} DEVICE;


/*
 * Table of user names.
 */
static	int		username_count;
static	int		username_avail;
static	NAME *		username_table;


/*
 * Table of group names.
 */
static	int		groupname_count;
static	int		groupname_avail;
static	NAME *		groupname_table;


/*
 * Table of device names.
 */
static	int		devname_count;
static	int		devname_avail;
static	DEVICE *	devname_table;


/*
 * Definitions for allocation of temporary string values.
 * These buffers are linked together, and freed all at once.
 */
#define	TEMPSTR_ALLOC_SIZE	(1024 * 16)
#define	TEMPSTR_MAX_SIZE	512


typedef	struct	TEMPSTR	TEMPSTR;

struct	TEMPSTR	 {
	TEMPSTR *	next;
	char		buf[TEMPSTR_ALLOC_SIZE];
};


/*
 * Amount of space left in the most recently allocated buffer.
 */
static	int	tempstr_avail;


/*
 * Linked list of string buffers.
 * Only the first one in the list has space for new allocation.
 */
static	TEMPSTR *	tempstr_list;


/*
 * Definitions for shared hashed strings which are individually allocated and
 * freed.  These strings are used for holding environment variable strings
 * since they are very large and many processes would share the same strings.
 * They are linked into hash tables for quick lookup.
 */
#define	SHARED_HASH_SIZE	101
#define	SHARED_MAGIC		749712759


typedef	struct	SHARED_STR	SHARED_STR;

struct	SHARED_STR	{
	long		magic;		/* magic number */
	int		tableindex;	/* index into table of the entry */
	int		count;		/* reference counter */
	int		len;		/* length of string */
	SHARED_STR *	next;		/* next string in list */
	char		buf[4];		/* string buffer (variable sized) */
};


static	SHARED_STR *	shared_table[SHARED_HASH_SIZE];


/*
 * The offset into the entry of the string buffer.
 */
#define	SHARED_BUF_OFFSET	((int) (((SHARED_STR *) 0)->buf))


/*
 * Allocate a shared string holding the specified text of the given length.
 * If the string is already in the shared string table, then it is found and
 * its use count is simply incremented.  Otherwise a new string is inserted.
 * The terminating NULL character is not included in the supplied length.
 * Returns NULL if the string could not be allocated.
 */
char *
AllocateSharedString(char *str, int len)
{
	int		begch;
	int		midch;
	int		endch;
	int		tableindex;
	SHARED_STR *	entry;

	if (len < 0)
		return NULL;

	if (len == 0)
		return empty_string;

	/*
	 * Get the first, middle, and last characters for comparison.
	 */
	begch = str[0];
	midch = str[len / 2];
	endch = str[len - 1];

	/*
	 * Hash the characters along with the string length to select
	 * an offset into the hash table to use for this string.
	 */
	tableindex = len + (begch << 16) + (midch << 20) + (endch << 24);
	tableindex = ((unsigned int) tableindex) % SHARED_HASH_SIZE;

	/*
	 * Search the selected string list in the table for the string.
	 * Test the string length and the first, middle, and last characters.
	 * If they all match then compare the whole string explicitly.
	 */
	for (entry = shared_table[tableindex]; entry; entry = entry->next)
	{
		if ((entry->len != len) || (entry->buf[0] != begch))
			continue;

		if (entry->buf[len / 2] != midch)
			continue;

		if (entry->buf[len - 1] != endch)
			continue;

		if (memcmp(entry->buf, str, len) != 0)
			continue;

		/*
		 * The string was found.
		 * Increment the use counter and return it.
		 */
		entry->count++;

		return entry->buf;
	}

	/*
	 * The string is not in the table, so we have to allocate a new one
	 * and link it in at the front of the appropriate shared table list.
	 * Note: the buffer in the structure has room for the terminating NULL.
	 */
	entry = (SHARED_STR *) malloc(sizeof(SHARED_STR) + len);

	if (entry == NULL)
		return NULL;

	entry->next = shared_table[tableindex];
	shared_table[tableindex] = entry;

	entry->magic = SHARED_MAGIC;
	entry->tableindex = tableindex;
	entry->count = 1;
	entry->len = len;

	memcpy(entry->buf, str, len);

	entry->buf[len] = '\0';

	return entry->buf;
}


/*
 * Free a string that had been allocated as a shared string.
 * This just decrements the usage count, and if it goes to zero,
 * then frees the string.
 */
void
FreeSharedString(char *str)
{
	SHARED_STR *	entry;
	SHARED_STR *	prev;
	int		tableindex;

	if ((str == NULL) || (str == empty_string))
		return;

	/*
	 * Back up to the entry header of the string and make sure that it
	 * looks reasonable.
	 */
	entry = (SHARED_STR *) (str - SHARED_BUF_OFFSET);

	tableindex = entry->tableindex;

	if ((entry->magic != SHARED_MAGIC) || (tableindex < 0) ||
		(tableindex >= SHARED_HASH_SIZE) || (entry->len < 0))
	{
		fprintf(stderr, "Freeing bad shared string header\n");

		exit(1);
	}

	/*
	 * Decrement the use counter, and if it is still positive,
	 * then the string is still in use.
	 */
	entry->count--;

	if (entry->count > 0)
		return;

	/*
	 * The string needs freeing.
	 * See if it is the first string in the hash table.
	 * If so, then removing it is easy.
	 */
	prev = shared_table[tableindex];

	if (prev == entry) {
		shared_table[tableindex] = entry->next;

		free((char *) entry);

		return;
	}

	/*
	 * We have to search the list for it.
	 * It is a fatal error if the string is not found.
	 */
	while (prev) {
		if (prev->next != entry) {
			prev = prev->next;

			continue;
		}

		prev->next = entry->next;

		free((char *) entry);

		return;
	}

	fprintf(stderr, "Freeing unknown shared string\n");
	exit(1);
}


/*
 * Allocate a string which can later be freed along with all other such
 * allocated strings.  The string cannot be individually freed.
 * To avoid wastage, there is a maximum size that can be allocated.
 * Prints an error message and exits on failure.
 */
char *
AllocTempString(int len)
{
	TEMPSTR *	head;
	char *		cp;

	if ((len <= 0) || (len > TEMPSTR_MAX_SIZE)) {
		fprintf(stderr, "Allocating bad length %d\n", len);

		exit(1);
	}

	if (len > tempstr_avail) {
		head = (TEMPSTR *) malloc(sizeof(TEMPSTR));

		if (head == NULL) {
			fprintf(stderr, "Failed to allocate %d bytes\n",
				sizeof(TEMPSTR));

			exit(1);
		}

		head->next = tempstr_list;
		tempstr_list = head;

		tempstr_avail = TEMPSTR_ALLOC_SIZE;
	}

	cp = &tempstr_list->buf[TEMPSTR_ALLOC_SIZE - tempstr_avail];
	tempstr_avail -= len;

	return cp;
}


/*
 * Copy a null-terminated string into a temporary string.
 * The new string cannot be individually freed.
 * Prints an error message and exits on failure.
 */
char *
CopyTempString(char *oldcp)
{
	char *	cp;
	int	len;

	len = strlen(oldcp) + 1;

	cp = AllocTempString(len);

	memcpy(cp, oldcp, len);

	return cp;
}


/*
 * Free all temporary strings.
 */
void
FreeTempStrings(void)
{
	TEMPSTR *	head;

	while (tempstr_list) {
		head = tempstr_list;
		tempstr_list = head->next;

		free((char *) head);
	}

	tempstr_avail = 0;
}


/*
 * Replace null characters in the specified memory buffer with spaces
 * and replace other unprintable characters with question marks.
 */
void
MakePrintable(char *cp, int len)
{
	int	ch;

	while (len-- > 0) {
		ch = *cp;

		if (ch == '\0')
			*cp = ' ';
		else if ((ch < ' ') || (ch >= 0x7f))
			*cp = '?';

		cp++;
	}
}


/*
 * Routine to see if a text string is matched by a wildcard pattern.
 * Returns TRUE if the text is matched, or FALSE if it is not matched
 * or if the pattern is invalid.
 *  *		matches zero or more characters
 *  ?		matches a single character
 *  [abc]	matches 'a', 'b' or 'c'
 *  \c		quotes character c
 *  Adapted from code written by Ingo Wilken.
 */
BOOL
PatternMatch(text, pattern)
	char *	text;
	char *	pattern;
{
	char *	retrypat;
	char *	retrytxt;
	int	ch;
	BOOL	found;

	retrypat = NULL;
	retrytxt = NULL;

	while (*text || *pattern) {
		ch = *pattern++;

		switch (ch) {
			case '*':  
				retrypat = pattern;
				retrytxt = text;
				break;

			case '[':  
				found = FALSE;

				while ((ch = *pattern++) != ']') {
					if (ch == '\\')
						ch = *pattern++;

					if (ch == '\0')
						return FALSE;

					if (*text == ch)
						found = TRUE;
				}

				if (!found) {
					pattern = retrypat;
					text = ++retrytxt;
				}

				/*
				 * Fall into next case
				 */

			case '?':  
				if (*text++ == '\0')
					return FALSE;

				break;

			case '\\':  
				ch = *pattern++;

				if (ch == '\0')
					return FALSE;

				/*
				 * Fall into next case
				 */

			default:        
				if (*text == ch) {
					if (*text)
						text++;

					break;
				}

				if (*text) {
					pattern = retrypat;
					text = ++retrytxt;
					break;
				}

				return FALSE;
		}

		if (pattern == NULL)
			return FALSE;
	}

	return TRUE;
}


/*
 * Collect all of the user names in the system.
 * This is only done once per run.
 */
void
CollectUserNames(void)
{
	struct passwd *	pwd;
	NAME *		name;

	if (username_count > 0)
		return;

	while ((pwd = getpwent()) != NULL)
	{
		if (username_count >= username_avail) {
			username_avail += USERNAME_ALLOC;

			username_table = (NAME *) realloc(username_table,
				(username_avail * sizeof(NAME)));

			if (username_table == NULL) {
				fprintf(stderr, "Cannot allocate memory\n");
				exit(1);
			}
		}

		name = &username_table[username_count++];

		name->id = pwd->pw_uid;
		strncpy(name->name, pwd->pw_name, MAX_NAME_LEN);
		name->name[MAX_NAME_LEN] = '\0';
	}

	endpwent();
}


/*
 * Find the user name for the specified user id.
 * Returns NULL if the user name is not known.
 */
char *
FindUserName(int uid)
{
	NAME *	user;
	int	count;

	user = username_table;
	count = username_count;

	while (count-- > 0) {
		if (user->id == uid)
			return user->name;

		user++;
	}

	return NULL;
}


/*
 * Find the user id associated with a user name.
 * Returns -1 if the user name is not known.
 */
int
FindUserId(char *name)
{
	NAME *	user;
	int	count;

	user = username_table;
	count = username_count;

	while (count-- > 0) {
		if (strcmp(user->name, name) == 0)
			return user->id;

		user++;
	}

	return -1;
}


/*
 * Collect all of the group names in the system.
 * This is only done once per run.
 */
void
CollectGroupNames(void)
{
	struct group *	grp;
	NAME *		name;

	if (groupname_count > 0)
		return;

	while ((grp = getgrent()) != NULL)
	{
		if (groupname_count >= groupname_avail) {
			groupname_avail += GROUPNAME_ALLOC;

			groupname_table = (NAME *) realloc(groupname_table,
				(groupname_avail * sizeof(NAME)));

			if (groupname_table == NULL) {
				fprintf(stderr, "Cannot allocate memory\n");
				exit(1);
			}
		}

		name = &groupname_table[groupname_count++];

		name->id = grp->gr_gid;
		strncpy(name->name, grp->gr_name, MAX_NAME_LEN);
		name->name[MAX_NAME_LEN] = '\0';
	}

	endgrent();
}


/*
 * Find the group name for the specified group id.
 * Returns NULL if the group name is not known.
 */
char *
FindGroupName(int gid)
{
	NAME *	group;
	int	count;

	group = groupname_table;
	count = groupname_count;

	while (count-- > 0) {
		if (group->id == gid)
			return group->name;

		group++;
	}

	return NULL;
}


/*
 * Find the group id associated with a group name.
 * Returns -1 if the group name is not known.
 */
int
FindGroupId(char *name)
{
	NAME *	group;
	int	count;

	group = groupname_table;
	count = groupname_count;

	while (count-- > 0) {
		if (strcmp(group->name, name) == 0)
			return group->id;

		group++;
	}

	return -1;
}


/*
 * Collect all device names that might be terminals or other character devices.
 * This is done only once per run.
 */
void
CollectDeviceNames(void)
{
	DIR *		dir;
	struct dirent *	dp;
	DEVICE *	device;
	struct stat	statbuf;
	char		fullname[512];

	if (devname_count > 0)
		return;

	dir = opendir("/dev");

	if (dir == NULL)
		return;

	while ((dp = readdir(dir)) != NULL)
	{
		strcpy(fullname, "/dev/");
		strcat(fullname, dp->d_name);

		if (lstat(fullname, &statbuf) < 0)
			continue;

		if (!S_ISCHR(statbuf.st_mode))
			continue;

		if (devname_count >= devname_avail) {
			devname_avail += DEVNAME_ALLOC;

			devname_table = (DEVICE *) realloc(devname_table,
				(devname_avail * sizeof(DEVICE)));

			if (devname_table == NULL) {
				fprintf(stderr, "Cannot allocate memory\n");
				exit(1);
			}
		}

		device = &devname_table[devname_count++];

		device->id = statbuf.st_rdev;
		device->dev = statbuf.st_dev;
		device->inode = statbuf.st_ino;
		strncpy(device->name, dp->d_name, MAX_DEV_LEN);
		device->name[MAX_DEV_LEN] = '\0';

		/*
		 * Look for /dev/null for other uses.
		 */
		if (strcmp(device->name, "null") == 0) {
			null_dev = device->dev;
			null_inode = device->inode;
		}
	}

	closedir(dir);
}


/*
 * Find the device name for the specified device id.
 * Returns NULL if the device name is not known.
 */
char *
FindDeviceName(int devid)
{
	DEVICE *device;
	int	count;

	if (devid <= 0)
		return "-";

	device = devname_table;
	count = devname_count;

	while (count-- > 0) {
		if (device->id == devid)
			return device->name;

		device++;
	}

	return NULL;
}


/*
 * Find the device name for a device and inode pair.
 * These arguments are for the inode which contains the device.
 */
char *
FindDeviceFromInode(dev_t dev, ino_t inode)
{
	DEVICE *device;
	int	count;

	device = devname_table;
	count = devname_count;

	while (count-- > 0) {
		if ((device->dev == dev) && (device->inode == inode))
			return device->name;

		device++;
	}

	return NULL;
}


/*
 * Parse a decimal number from a string and return it's value.
 * The supplied string pointer is updated past the number read.
 * Leading spaces or tabs are ignored.
 * An invalid character stops the parse.
 */
long
GetDecimalNumber(char **cpp)
{
	long	value;
	BOOL	neg;
	char *	cp;

	cp = *cpp;

	neg = FALSE;

	while (isblank(*cp))
		cp++;

	if (*cp == '-') {
		neg = TRUE;
		cp++;
	}

	value = 0;

	while (isdigit(*cp))
		value = value * 10 + *cp++ - '0';

	if (neg)
		value = -value;

	*cpp = cp;

	return value;
}

/* END CODE */
