/*
 * Configurable ps-like program.
 * Sorting support routines.
 *
 * Copyright (c) 1995 David I. Bell
 * Permission is granted to use, distribute, or modify this source,
 * provided that this copyright notice remains intact.
 */

#include "ips.h"
#include "expr.h"


#define	MAX_SORT	50	/* maximum items to allow sorting by */
#define	SORT_ALLOC_SIZE	100	/* reallocation size for sorting */


/*
 * Types of values to sort on.
 */
typedef	int	SORT_TYPE;

#define	SORT_NORMAL		1	/* normal sort on column value */
#define	SORT_REVERSE		2	/* reverse sort on column value */
#define	SORT_NORMAL_EXPR	3	/* sort on expression evaluation */
#define	SORT_REVERSE_EXPR	4	/* reverse sort on expression */


/*
 * One item of sorting.
 */
typedef	struct	{
	SORT_TYPE	type;		/* type of sort */
	COLUMN *	column;		/* column to sort by */
	TREE *		tree;		/* expression tree to sort by */
} SORT_ITEM;


/*
 * Data to specify sorting conditions.
 */
static	int		sort_count;		/* number of items in table */
static	SORT_ITEM	sort_table[MAX_SORT];	/* table of sorting items */


/*
 * Data used during the actual sorting.
 */
static	int		proc_sort_size;		/* entries in sort table */
static	PROC **		proc_sort_table;	/* proc sorting table */


/*
 * Local procedures.
 */
static	int	SortFunction(const void *e1, const void *e2);
static	int	CompareSortingItems(PROC *proc1, PROC *proc2);


/*
 * Get the use flags for all the columns that we sort by.
 * This just means find all references to columns, and OR together
 * the use flags for each column.
 */
USEFLAG
GetSortingUseFlags(void)
{
	SORT_ITEM *	item;
	USEFLAG		flags;

	flags = USE_NONE;

	for (item = sort_table; item < &sort_table[sort_count]; item++) {
		switch (item->type) {
			case SORT_NORMAL:
			case SORT_REVERSE:
				flags |= item->column->useflag;

				break;

			case SORT_NORMAL_EXPR:
			case SORT_REVERSE_EXPR:
				flags |= GetNodeUseFlags(item->tree->root);

				break;

			default:
				break;
		}
	}

	return flags;
}


/*
 * Clear all sorting.
 */
void
ClearSorting(void)
{
	sort_count = 0;
}


/*
 * Append items to do sorting according to the indicated column names.
 * The sorting can be the normal forwards sort, or else a reverse sort.
 * The arguments are column names, no expressions are allowed here.
 */
BOOL
AppendColumnSort(ARGS *ap, BOOL reverse)
{
	SORT_TYPE	type;
	SORT_ITEM *	item;
	char *		name;
	COLUMN *	column;
	int		count;
	int		i;
	char *		table[MAX_WORDS];

	type = (reverse ? SORT_REVERSE : SORT_NORMAL);

	count = ExpandArguments(ap, table, MAX_WORDS);

	if (count < 0)
		return FALSE;

	for (i = 0; i < count; i++) {
		if (sort_count >= MAX_SORT) {
			fprintf(stderr, "Too many sort items\n");

			return FALSE;
		}

		name = table[i];

		column = FindColumn(name);

		if (column == NULL) {
			fprintf(stderr, "Bad sorting column name %s\n", name);

			return FALSE;
		}

		item = &sort_table[sort_count];

		item->type = type;
		item->column = column;
		item->tree = NULL;

		sort_count++;
	}

	return TRUE;
}


/*
 * Routine to append a sort expression to the sorting list.
 * The sorting can be the normal forwards sort, or else a reverse sort.
 */
BOOL
AppendExpressionSort(ARGS *ap, BOOL reverse)
{
	SORT_TYPE	type;
	SORT_ITEM *	item;
	char *		str;

	type = (reverse ? SORT_REVERSE_EXPR : SORT_NORMAL_EXPR);

	if (ap->count <= 0) {
		fprintf(stderr, "Missing sort expression\n");

		return FALSE;
	}

	ap->count--;
	str = *ap->table++;

	if (sort_count >= MAX_SORT) {
		fprintf(stderr, "Too many sort items\n");

		return FALSE;
	}

	item = &sort_table[sort_count];

	item->type = type;
	item->column = NULL;
	item->tree = (TREE *) malloc(sizeof(TREE));

	if (item->tree == NULL) {
		fprintf(stderr, "Cannot allocate memory\n");

		return FALSE;
	}

	if (!ParseTree(item->tree, str, 0))
		return FALSE;

	sort_count++;

	return TRUE;
}


/*
 * Sort the process list.
 */
void
SortProcesses(void)
{
	int	count;
	PROC *	proc;
	PROC **	procptr;

	/*
	 * If we need to grow the sorting table to include new processes,
	 * then do so.
	 */
	if (proc_alloc_count >= proc_sort_size) {
		proc_sort_size = proc_alloc_count + SORT_ALLOC_SIZE;

		proc_sort_table = (PROC **) realloc(proc_sort_table,
			(sizeof(PROC *) * (proc_sort_size + 1)));

		if (proc_sort_table == NULL) {
			fprintf(stderr, "Cannot allocate memory\n");
			exit(1);
		}
	}

	/*
	 * Put the process entries into the sort table.
	 */
	count = 0;

	for (proc = process_list; proc; proc = proc->next)
		proc_sort_table[count++] = proc;

	proc_sort_table[count] = NULL_PROC;

	/*
	 * Sort the entries in the table.
	 */
	qsort((void *) proc_sort_table, count, sizeof(PROC *), SortFunction);

	/*
	 * Relink the processes into a new list according to the sorted
	 * table pointers.
	 */
	procptr = proc_sort_table;

	process_list = *procptr;

	while (*procptr) {
		procptr[0]->next = procptr[1];
		procptr++;
	}
}


/*
 * Function called by qsort to compare two processes for the correct
 * sorting order.  Returns -1 if the first process is first, 1 if the
 * second process is first, or 0 if they are equal.  (But there should
 * be no case where two different processes actually sort as equal.)
 */
static int
SortFunction(const void *e1, const void *e2)
{
	PROC **	procptr1;
	PROC **	procptr2;
	PROC *	proc1;
	PROC *	proc2;
	int	status;

	procptr1 = (PROC **) e1;
	procptr2 = (PROC **) e2;

	proc1 = *procptr1;
	proc2 = *procptr2;

	/*
	 * Check for processes which are not to be shown so as to
	 * avoid any hard work comparing them to sort them.
	 */
	if (!proc1->isshown || !proc2->isshown) {
		if (proc1->isshown)
			return -1;

		if (proc2->isshown)
			return 1;

		return 0;
	}

	/*
	 * Do sorting based on any specified sorting items,
	 * which can be column names or arbitrary expressions.
	 */
	status = CompareSortingItems(proc1, proc2);

	if (status != 0)
		return status;

	/*
	 * They appear equal for all sorting items, so do a final sort on
	 * their process ids which should always distinguish them.
	 */
	if (proc1->pid < proc2->pid)
		return -1;

	if (proc1->pid > proc2->pid)
		return 1;

	return 0;
}


/*
 * Compare two processes using the defined list of sorting items.
 * The comparison can be on column values or on arbitrary expressions.
 * Returns -1 if the first process is less, 1 if the first process is
 * more, or 0 if they are identical.
 */
static int
CompareSortingItems(PROC *proc1, PROC *proc2)
{
	SORT_ITEM *	item;
	TREE *		tree;
	int		status;
	VALUE		value1;
	VALUE		value2;

	if (proc1 == proc2)
		return 0;

	for (item = sort_table; item < &sort_table[sort_count]; item++) {
		switch (item->type) {
			case SORT_NORMAL:
				status = (*item->column->sortfunc)
					(proc1, proc2);

				break;

			case SORT_REVERSE:
				status = -(*item->column->sortfunc)
					(proc1, proc2);

				break;


			case SORT_NORMAL_EXPR:
			case SORT_REVERSE_EXPR:
				tree = item->tree;
				tree->proc = proc1;
				value1 = EvaluateNode(tree, tree->root);

				tree->proc = proc2;
				value2 = EvaluateNode(tree, tree->root);

				if (!CompareValues(value1, value2, &status))
					return 0;

				if (item->type == SORT_REVERSE_EXPR)
					status = -status;

				break;

			default:
				status = 0;
				break;
		}

		if (status != 0)
			return status;
	}

	return 0;
}

/* END CODE */
