/*
 * Configurable ps-like program.
 * Routines to display the results.
 *
 * Copyright (c) 1995 David I. Bell
 * Permission is granted to use, distribute, or modify this source,
 * provided that this copyright notice remains intact.
 */

#include "ips.h"


/*
 * The ANSI escape sequence for clearing the terminal screen.
 */
#define	CLEAR_SCREEN	"\033[H\033[2J"


/*
 * Static variables.
 */
static	BOOL	shown;		/* whether anything has been shown */
static	int	headerwidth;	/* width of longest header string to show */


/*
 * Local procedures.
 */
static	void	ShowVerticalProcessStatus(PROC *proc);
static	char *	ReturnHeaderString(void);
static	char *	ReturnProcessString(PROC *proc);


/*
 * Initialize for displaying the process status.
 * All we do here is calculate the maximum width of the column headers
 * for use in vertical format.  Returns TRUE if successful.
 */
BOOL
InitializeDisplay(void)
{
	COLUMN **	columnptr;
	COLUMN *	column;
	int		len;

	headerwidth = 0;

	if (no_header || !vertical)
		return TRUE;

	/*
	 * In vertical mode, set buffering for block mode for efficiency.
	 */
	setvbuf(stdout, NULL, _IOFBF, BUFSIZ);

	/*
	 * Iterate over all of the desired columns to be displayed
	 * and find the longest column header string.
	 */
	for (columnptr = show_list; *columnptr; columnptr++)
	{
		column = *columnptr;

		len = strlen(column->heading);

		if (len > headerwidth)
			headerwidth = len;
	}

	return TRUE;
}


/*
 * Show all of the selected processes for one iteration of the program.
 * This clears the screen or prints a blank line, prints the header string
 * if required, then prints the process status for each selected process.
 * The status can be printed horizontally or vertically.
 */
void
ShowSelectedProcesses(void)
{
	PROC *	proc;
	int	seencount;

	/*
	 * If we need to clear the screen, then do that.
	 * This is done whether or not any processes are displayed.
	 */
	if (clear_screen)
		fputs(CLEAR_SCREEN, stdout);

	/*
	 * Loop over all processes in the list.
	 */
	seencount = 0;

	for (proc = process_list; proc; proc = proc->next) {
		/*
		 * If we are restricting the output to the top few process,
		 * then stop when we reach the limit.
		 */
		if (top_count && (seencount >= top_count))
			return;

		/*
		 * If the process is not to be shown, then skip it.
		 */
		if (!proc->isshown)
			continue;

		/*
		 * If we haven't displayed any processes yet this iteration,
		 * then print a blank line if required and then print the
		 * header string if required.
		 */
		if (seencount == 0) {
			if (!clear_screen && shown)
				fputc('\n', stdout);

			if (!no_header && !vertical)
				puts(ReturnHeaderString());
		}

		/*
		 * If we want a vertical display of the columns,
		 * then do that, otherwise do the normal horizontal output.
		 */
		if (vertical)
			ShowVerticalProcessStatus(proc);
		else
			puts(ReturnProcessString(proc));

		seencount++;
		shown = TRUE;
	}
}


/*
 * Return the header line as determined by the currently defined columns.
 * This returns a pointer to a static buffer with the stored information.
 * The column headers are justified appropriately.  This is only used for
 * horizontal column displays.
 */
static char *
ReturnHeaderString(void)
{
	COLUMN **	columnptr;
	COLUMN *	column;
	int		col;
	int		len;
	int		padding;
	int		tmp;
	char *		cp;
	char *		bufcp;
	static	char	buf[MAX_WIDTH + 1];

	bufcp = buf;
	col = 0;
	padding = 0;
	columnptr = show_list;
	column = *columnptr++;

	while (column && (col + column->width <= width)) {
		while (padding-- > 0)
			*bufcp++ = ' ';

		cp = column->heading;

		len = strlen(cp);

		if (len > column->width)
			len = column->width;

		padding = column->width - len;

		if (column->justify == RIGHT) {
			while (padding-- > 0)
				*bufcp++ = ' ';
		}

		if (column->justify == CENTER) {
			tmp = padding / 2;
			padding -= tmp;

			while (tmp-- > 0)
				*bufcp++ = ' ';
		}

		memcpy(bufcp, cp, len);
		bufcp += len;

		col += column->width + separation;

		if (padding < 0)
			padding = 0;

		padding += separation;
		column = *columnptr++;
	}

	*bufcp = '\0';

	return buf;
}


/*
 * Return a line of information about the specified process based on the
 * currently defined columns.  This returns a pointer to a static buffer
 * with the stored information.  The columns are justified appropriately.
 * This is only used for horizontal column displays.
 */
static char *
ReturnProcessString(PROC *proc)
{
	COLUMN **	columnptr;
	COLUMN *	column;
	BOOL		overflow;
	int		len;
	int		padding;
	int		tmp;
	int		col;
	int		goal_col;
	char *		cp;
	char *		bufcp;
	static	char	buf[MAX_WIDTH + 1];

	bufcp = buf;
	col = 0;
	goal_col = 0;
	columnptr = show_list;
	column = *columnptr++;

	while (column && (goal_col + column->width <= width))
	{
		cp = (*column->showfunc)(proc);

		len = strlen(cp);

		/*
		 * Limit the output to the column width, or if it is the
		 * last column, the rest of the line.
		 */
		padding = column->width - len;

		overflow = FALSE;

		if (*columnptr) {
			if (len > column->width) {
				len = column->width - 1;
				overflow = TRUE;
				padding = 0;
			}
		} else {
			if (len > (width - goal_col)) {
				len = width - goal_col - 1;
				overflow = TRUE;
				padding = 0;
			}
		}

		if (padding > 0) {
			if (column->justify == RIGHT) {
				goal_col += padding;
				padding = 0;
			}

			if (column->justify == CENTER) {
				tmp = padding / 2;
				padding -= tmp;
				goal_col += tmp;
			}
		}

		while (col < goal_col) {
			col++;
			*bufcp++ = ' ';
		}

		memcpy(bufcp, cp, len);

		bufcp += len;
		goal_col += len;
		col += len;

		if (overflow) {
			*bufcp++ = '|';
			goal_col++;
			col++;
		}

		if (padding > 0)
			goal_col += padding;

		goal_col += separation;

		column = *columnptr++;
	}

	*bufcp = '\0';

	return buf;
}


/*
 * Print the status of a process in a vertical format.
 * In this format, each process status requires multiple lines, one per
 * item to be displayed.  The beginning of each line can give the item name,
 * and the rest of each line contains the value.  No column justification
 * is done here so that all values are left justified.
 */
static void
ShowVerticalProcessStatus(PROC *proc)
{
	COLUMN **	columnptr;
	COLUMN *	column;
	int		columnwidth;
	int		len;
	char *		cp;
	char		buf[80];

	/*
	 * Separate the process status from the previous process status.
	 * There is also a blank line between iterations, so a double blank
	 * line indicates the beginning of an iteration's output.
	 */
	if (shown)
		fputc('\n', stdout);

	/*
	 * Calculate how many columns are available for each value.
	 * Make sure there is at least one column available, even if
	 * that means we will overflow the line.
	 */
	if (no_header)
		columnwidth = width;
	else
		columnwidth = width - headerwidth - separation - 1;

	if (columnwidth <= 0)
		columnwidth = 1;

	/*
	 * Iterate over all of the desired columns to be displayed.
	 */
	for (columnptr = show_list; *columnptr; columnptr++)
	{
		column = *columnptr;

		/*
		 * Construct and output the header string for the line.
		 * The string contains the column header and a colon,
		 * followed by padding out to the maximum width of the
		 * header strings plus the column separation value.
		 */
		if (!no_header) {
			cp = buf;

			len = strlen(column->heading);
			memcpy(cp, column->heading, len);
			cp += len;

			*cp++ = ':';

			len = headerwidth + separation - len;

			if (len > 0) {
				memset(cp, ' ', len);
				cp += len;
			}

			*cp = '\0';

			fputs(buf, stdout);
		}

		/*
		 * Now get the value for the data item and get its length.
		 */
		cp = (*column->showfunc)(proc);

		len = strlen(cp);

		/*
		 * If the column is too wide, then truncate it and change
		 * the last character into a vertical bar.  Otherwise print
		 * the whole value.
		 */
		if (len > columnwidth)
			printf("%.*s|\n", columnwidth - 1, cp);
		else
			puts(cp);

		fflush(stdout);
	}
}

/* END CODE */
