/*
 * Configurable ps-like program.
 * Process information utility routines.
 *
 * Copyright (c) 1995 David I. Bell
 * Permission is granted to use, distribute, or modify this source,
 * provided that this copyright notice remains intact.
 */

#include "ips.h"


/*
 * Static variables.
 */
static	struct	timeval	current_timeval;	/* current accurate time */
static	struct	timeval	previous_timeval;	/* previous accurate time */
static	long		elapsed_milliseconds;	/* time between samples */


/*
 * Perform the initial process scan if required, and sleep after it.
 * This is required if some columns are selected which require more than
 * one sample to calculate the value (e.g., cpu percentage), or if the
 * user wanted to only show active processes.
 */
void
InitialProcessScan(void)
{
	if (init_time <= 0)
		return;

	if (active_only || use_init) {
		ScanProcesses();

		sleep(init_time);
	}
}


/*
 * Find the process structure with the specified pid.
 * If the process is new, then it is malloc'd and flagged as such.
 * A new process structure has mostly rubbish values.
 * The process structure is linked into the process list.
 */
PROC *
FindProcess(int pid)
{
	PROC *	proc;

	/*
	 * See if the process is already in the process list.
	 * If so, return the found structure.
	 */
	for (proc = process_list; proc; proc = proc->next) {
		if (proc->pid == pid)
			return proc;
	}

	/*
	 * Nope, so allocate a new structure either from the free list
	 * or else using malloc.
	 */
	proc = free_process_list;

	if (proc)
		free_process_list = proc->next;
	else {
		proc = malloc(sizeof(PROC));

		if (proc == NULL) {
			fprintf(stderr, "Cannot allocate memory\n");
			exit(1);
		}

		proc_alloc_count++;
	}

	/*
	 * Initialise some of its columns.
	 */
	proc->next = NULL;
	proc->pid = pid;
	proc->isnew = TRUE;
	proc->isvalid = FALSE;
	proc->isactive = FALSE;
	proc->isshown = FALSE;
	proc->hascommand = FALSE;
	proc->changed = TRUE;
	proc->isancient = ancient_flag;
	proc->state = ' ';
	proc->start_time_ticks = 0;
	proc->start_time_clock = 0;
	proc->last_saved_time = 0;
	proc->last_active_time = 0;
	proc->last_sync_time = 0;
	proc->live_counter = 0;
	proc->percent_cpu = 0;
	proc->percent_memory = 0;
	proc->run_order = 0;
	proc->openfiles = 0;
	proc->cwd_dev = 0;
	proc->cwd_inode = 0;
	proc->end_code = 0;
	proc->old_stime = 0;
	proc->old_utime = 0;
	proc->command_length = 0;
	proc->environment_length = 0;
	proc->command = proc->command_buffer;
	proc->environment = empty_string;

	/*
	 * Add it to the process list.
	 */
	proc->next = process_list;
	process_list = proc;

	return proc;
}


/*
 * Remove all dead processes from the process list.
 */
void
RemoveDeadProcesses(void)
{
	PROC *	proc;
	PROC *	prevproc;
	PROC *	nextproc;

	prevproc = NULL;

	for (proc = process_list; proc; proc = nextproc) {
		nextproc = proc->next;

		/*
		 * If the process is still alive skip onwards to the next one.
		 */
		if (proc->live_counter == live_counter) {
			prevproc = proc;

			continue;
		}

		/*
		 * The process is dead.
		 * Free any dynamic storage used by the process.
		 */
		if (proc->command != proc->command_buffer) {
			free(proc->command);
			proc->command = proc->command_buffer;
		}

		proc->command_length = 0;

		if (proc->environment != empty_string) {
			FreeSharedString(proc->environment);
			proc->environment = empty_string;
		}

		proc->environment_length = 0;

		/*
		 * Remove it from the process list by linking the previous
		 * process to the next one and leaving prevproc alone.
		 */
		if (prevproc)
			prevproc->next = nextproc;
		else
			process_list = nextproc;

		/*
		 * Add the dead process to the free process list.
		 */
		proc->next = free_process_list;
		free_process_list = proc;

		proc->pid = 0;
		proc->state = ' ';
	}
}


/*
 * Calculate how long it has been since the process has been active.
 * Part of this calculation compares previously gathered state with
 * the newest state.  In this way, even transiently running processes
 * can still be detected as not being idle.
 */
void
CheckActiveProcess(PROC *proc)
{
	BOOL	active;
	long	ticks_used;

	/*
	 * The process is definitely active if it is currently runnable
	 * or is in a short term wait like disk I/O.
	 */
	active = ((proc->state == 'R') || (proc->state == 'D'));

	if (active) {
		proc->last_active_time = current_time;
		proc->run_order = live_counter;
	}

	/*
	 * Calculate the percent of the cpu the process has used since
	 * the previous check.
	 */
	ticks_used = (proc->utime + proc->stime) -
		(proc->old_utime + proc->old_stime);

	if (elapsed_milliseconds > 0)
		proc->percent_cpu =
			(ticks_used * ((1000 * CPU_SCALE) / TICKS)) /
				elapsed_milliseconds;

	if (proc->percent_cpu >= CPU_SCALE)
		proc->percent_cpu = CPU_SCALE - 1;

	if (total_memory_clicks > 0) {
		proc->percent_memory =
			(proc->rss * MEMORY_SCALE) / total_memory_clicks;
	}

	proc->changed = FALSE;

	/*
	 * If some of its state has changed since the last check,
	 * then it is also considered active.  Save the new values
	 * of that state for future checks.
	 */
	if (proc->isnew || (ticks_used > 0) ||
		(proc->state != proc->old_state) ||
		(proc->flags != proc->old_flags) ||
		(proc->min_flt != proc->old_min_flt) ||
		(proc->maj_flt != proc->old_maj_flt) ||
		(proc->start_time_ticks != proc->old_start_time_ticks) ||
		(proc->end_code != proc->old_end_code) ||
		(proc->esp != proc->old_esp) ||
		(proc->eip != proc->old_eip) ||
		(proc->wchan != proc->old_wchan)
	) {
		proc->changed = TRUE;
		proc->isnew = FALSE;
		proc->isancient = ancient_flag;
		proc->last_active_time = current_time;
		proc->last_saved_time = current_time;
		proc->run_order = live_counter;
		proc->old_state = proc->state;
		proc->old_flags = proc->flags;
		proc->old_min_flt = proc->min_flt;
		proc->old_maj_flt = proc->maj_flt;
		proc->old_utime = proc->utime;
		proc->old_stime = proc->stime;
		proc->old_start_time_ticks = proc->start_time_ticks;
		proc->old_end_code = proc->end_code;
		proc->old_esp = proc->esp;
		proc->old_eip = proc->eip;
		proc->old_wchan = proc->wchan;
	}

	/*
	 * If the state changed recently, then the process is still active.
	 * Don't do this check for ancient processes that were there
	 * before we knew about their idleness.
	 */
	if ((!proc->isancient) &&
		(current_time <= (proc->last_saved_time + active_time)))
	{
		active = TRUE;
	}

 	if (active)
		proc->isancient = FALSE;

	proc->isactive = active;
}


/*
 * Update the current time and calculate the elapsed time interval since
 * the last call.
 */
void
UpdateTimes(void)
{
	long		secdiff;
	long		usecdiff;
	struct timezone	tz;

	live_counter++;

	time(&current_time);

	previous_timeval = current_timeval;

	gettimeofday(&current_timeval, &tz);

	secdiff = current_timeval.tv_sec - previous_timeval.tv_sec;
	usecdiff = current_timeval.tv_usec - previous_timeval.tv_usec;

	if (usecdiff < 0) {
		usecdiff += 1000000;
		secdiff--;
	}

	if (secdiff < 0) {
		secdiff = 0;
		usecdiff = 0;
	}

	elapsed_milliseconds = (secdiff * 1000) + (usecdiff / 1000);
}

/* END CODE */
