/*
 * Configurable ps-like program.
 * Main program.
 *
 * Copyright (c) 1995 David I. Bell
 * Permission is granted to use, distribute, or modify this source,
 * provided that this copyright notice remains intact.
 */

#include <termios.h>
#include <signal.h>
#include <sys/ioctl.h>

#include "ips.h"


#define	STDOUT		1


/*
 * List of columns being shown.
 */
int	show_count;
COLUMN *show_list[MAX_COLUMNS];


/*
 * Other global variables.
 */
ULONG		start_uptime;		/* uptime jiffies at start */
time_t		start_time;		/* clock time at start */
time_t		current_time;		/* current clock time */
long		total_memory_clicks;	/* amount of total memory */
ULONG		live_counter;		/* counter for live procs */
BOOL		ancient_flag;		/* seeing pre-existing procs */
BOOL		no_self;		/* don't show myself */
BOOL		no_root;		/* don't show root procs */
BOOL		no_header;		/* don't show column header */
BOOL		my_procs;		/* only show my procs */
BOOL		active_only;		/* only show active procs */
BOOL		clear_screen;		/* clear screen each loop */
BOOL		loop;			/* loop showing status */
BOOL		vertical;		/* vertical output format */
BOOL		top_auto;		/* autosize height for top */
BOOL		use_open_files;		/* using open file info */
BOOL		use_current_directory;	/* using current dir info */
BOOL		use_root_directory;	/* using root dir info */
BOOL		use_exec_inode;		/* using executable info */
BOOL		use_device_names;	/* using device name info */
BOOL		use_init;		/* using initial sleep */
BOOL		use_command;		/* using command line info */
BOOL		use_self;		/* using my own proc info */
BOOL		use_environment;	/* using environment info */
BOOL		use_stdio[3];		/* using various stdio info */
dev_t		null_dev;		/* device of /dev/null */
ino_t		null_inode;		/* inode of /dev/null */
int		proc_alloc_count;	/* allocated proc structures */
int		active_time;		/* seconds for active procs */
int		pid_count;		/* pids in pid_list */
int		user_count;		/* users in user_list */
int		group_count;		/* groups in group_list */
int		my_pid;			/* my pid */
int		my_uid;			/* my real user id */
int		my_gid;			/* my real group id */
int		width;			/* width of output */
int		height;			/* height of output */
int		separation;		/* blanks between columns */
int		sleep_time;		/* seconds between loops */
int		sync_time;		/* seconds between syncs */
int		init_time;		/* seconds for initial sleep */
int		top_count;		/* number of procs for top */
char		empty_string[4];	/* empty string */
PROC *		process_list;		/* list of existent procs */
PROC *		free_process_list;	/* free proc structure list */
int		pid_list[MAX_PIDS];	/* pids to be shown */
int		user_list[MAX_USERS];	/* user ids to be shown */
int		group_list[MAX_GROUPS];	/* group ids to be shown */


/*
 * Static variables.
 */
static	BOOL	need_resize;		/* terminal size has changed */


/*
 * Static procedures.
 */
static	void	HandleResize(int arg);
static	void	GetTerminalSize(void);
static	void	SetUseFlags(void);


int
main(int argc, char **argv)
{
	ARGS *	ap;
	ARGS	args;

	ap = &args;
	ap->table = ++argv;
	ap->count = --argc;

	my_pid = getpid();
	my_uid = getuid();
	my_gid = getgid();

	DefaultAllOptions();

	/*
	 * If output is to a terminal, then get its current size and
	 * set up to handle resize signals.
	 */
	if (isatty(STDOUT)) {
		signal(SIGWINCH, HandleResize);

		GetTerminalSize();
	}

	/*
	 * Parse the system definition file.
	 */
	if (!ParseSystemInitFile())
		return 1;

	/*
	 * Check specially for the -noinit option before reading the user's
	 * option file.  This option MUST be immediately after the program
	 * name.  This check is a bit of a hack.
	 */
	if ((ap->count > 0) && (strcmp(ap->table[0], "-noinit") == 0)) {
		ap->table++;
		ap->count--;
	} else if (!ParseUserInitFile())
		return 1;

	/*
	 * Look up and execute the initial macros if they exist.
	 */
	if (MacroExists(MACRO_TYPE_OPTION, SYSTEM_INIT_MACRO) &&
		!ExpandOptionName(SYSTEM_INIT_MACRO, 0))
	{
		fprintf(stderr,
			"Problem expanding system initial macro \"%s\"\n",
			SYSTEM_INIT_MACRO);

		return 1;
	}

	if (MacroExists(MACRO_TYPE_OPTION, USER_INIT_MACRO) &&
		!ExpandOptionName(USER_INIT_MACRO, 0))
	{
		fprintf(stderr,
			"Problem expanding system initial macro \"%s\"\n",
			USER_INIT_MACRO);

		return 1;
	}

	/*
	 * Parse the command line options possibly expanding them.
	 * Indicate that we are level zero so that bare macro names on the
	 * command line will be recognized even if they are lower case.
	 */
	if (!ParseOptions(ap, 0))
		return 1;

	/*
	 * Set the flags for what items we need collecting based on
	 * the columns that have been referenced.
	 */
	SetUseFlags();

	/*
	 * Initialize for collecting of process data.
	 */
	if (!InitializeProcessData())
		return 1;

	/*
	 * Initialize for displaying of process status.
	 */
	if (!InitializeDisplay())
		return 1;

	/*
	 * If we require a time interval before displaying results
	 * (such as for calculating percentage cpu), then collect
	 * the initial process status and sleep a while.
	 */
	InitialProcessScan();

	/*
	 * Here is the main loop.  It terminates after one iteration
	 * if we just want a snapshot.
	 */
	for (;;) {
		/*
		 * Release any string storage used during the last loop.
		 */
		FreeTempStrings();

		/*
		 * Collect the new process status.
		 */
		ScanProcesses();

		/*
		 * If the window was resized, then get the new window sizes.
		 */
		if (need_resize)
			GetTerminalSize();

		/*
		 * Show the selected processes.
		 */
		ShowSelectedProcesses();

		/*
		 * If we don't want to loop, then we are done.
		 * Otherwise, sleep a little while before looping.
		 */
		if (!loop)
			return 0;

		if (sleep_time > 0)
			sleep(sleep_time);
	}
}


/*
 * Set the use variables according to the items required by the columns
 * that have been referenced.  These variables are used to avoid
 * collecting expensive-to-obtain data when it is not required.
 */
static void
SetUseFlags(void)
{
	int	i;
	USEFLAG	flags;

	flags = USE_NONE;

	/*
	 * Scan the columns that are being shown.
	 */
	for (i = 0; i < show_count; i++)
		flags |= show_list[i]->useflag;

	/*
	 * Add in the columns that are being sorted by.
	 */
	flags |= GetSortingUseFlags();

	/*
	 * Add in the columns referenced by conditions.
	 */
	flags |= GetConditionUseFlags();

	/*
	 * Now set the boolean variables according to the flags.
	 */
	if (flags & USE_INIT)
		use_init = TRUE;

	if (flags & USE_DEV_NAME)
		use_device_names = TRUE;

	if (flags & USE_OPEN_FILE)
		use_open_files = TRUE;

	if (flags & USE_CURR_DIR)
		use_current_directory = TRUE;

	if (flags & USE_ROOT_DIR)
		use_root_directory = TRUE;

	if (flags & USE_EXEC_INODE)
		use_exec_inode = TRUE;

	if (flags & USE_COMMAND)
		use_command = TRUE;

	if (flags & USE_SELF)
		use_self = TRUE;

	if (flags & USE_ENVIRON)
		use_environment = TRUE;

	if (flags & USE_STDIN)
		use_stdio[0] = TRUE;

	if (flags & USE_STDOUT)
		use_stdio[1] = TRUE;

	if (flags & USE_STDERR)
		use_stdio[2] = TRUE;
}


/*
 * Signal handler for resizing of window.
 * This only sets a flag so that we can handle the resize later.
 * (Changing the size at unpredictable times would be dangerous.)
 */
static void
HandleResize(int arg)
{
	need_resize = TRUE;

	signal(SIGWINCH, HandleResize);
}


/*
 * Routine called to get the new terminal size from the kernel.
 */
static void
GetTerminalSize(void)
{
	struct	winsize	size;

	need_resize = FALSE;

	if (ioctl(STDOUT, TIOCGWINSZ, &size) < 0)
		return;

	width = size.ws_col;
	height = size.ws_row;

	if (width <= 0)
		width = 1;

	if (height <= 0)
		height = 1;

	if (top_auto) {
		top_count = height - 1;

		if (!no_header)
			top_count--;

		if (top_count <= 0)
			top_count = 1;
	}
}

/* END CODE */
