/*
 * Configurable ps-like program.
 * Global definitions.
 *
 * Copyright (c) 1995 David I. Bell
 * Permission is granted to use, distribute, or modify this source,
 * provided that this copyright notice remains intact.
 */

#ifndef	IPS_H
#define	IPS_H

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <string.h>
#include <time.h>
#include <sys/time.h>
#include <sys/types.h>


#ifndef	SYSTEM_INIT_FILE
#define	SYSTEM_INIT_FILE	"/usr/local/lib/ips.init"
#endif

#ifndef	USER_INIT_FILE
#define	USER_INIT_FILE		".ipsrc"
#endif

#ifndef	SYSTEM_INIT_MACRO
#define	SYSTEM_INIT_MACRO	"SysInit"
#endif

#ifndef	USER_INIT_MACRO
#define	USER_INIT_MACRO		"UserInit"
#endif

#define	FIRST_LINE		"#ips#"		/* first line of init files */


/*
 * Some constants related to the system.
 */
#define	TICKS		100		/* ticks in a second */
#define	CPU_SCALE	10000		/* scaling factor for cpu usage */
#define	MEMORY_SCALE	1000		/* scaling factor for memory usage */
#define	CLICK_SIZE	4096		/* size of memory allocation chunk */
#define	PZERO		15		/* normal priority */
#define	OLDPIPEDEV	0		/* old device number for pipes */
#define	NEWPIPEDEV	0xffff		/* new device number for pipes */
#define	BASE_USER_UID	100		/* first "user" uid */


/*
 * Some limits on the program.
 */
#define	MAX_PIDS	100
#define	MAX_USERS	100
#define	MAX_GROUPS	100
#define	MAX_WORDS	1000
#define	MAX_COLUMNS	50
#define	MAX_SEPARATION	20
#define	MAX_MACRO_LEN	16
#define	MAX_OPTION_DEPTH 20
#define	MAX_EXPR_DEPTH	20
#define	MAX_WIDTH	(1024 * 31)
#define	MAX_COMMAND_LEN	(1024 * 10)
#define	MAX_ENVIRON_LEN	(1024 * 20)
#define	BUF_COMMAND_LEN	128
#define	MAX_PROGRAM_LEN	32


/*
 * Macros to help parse strings.
 */
#define	isblank(ch)	(((ch) == ' ') || ((ch) == '\t'))
#define	isdigit(ch)	(((ch) >= '0') && ((ch) <= '9'))
#define	isupper(ch)	(((ch) >= 'A') && ((ch) <= 'Z'))
#define	islower(ch)	(((ch) >= 'a') && ((ch) <= 'z'))
#define	ismacro(ch)	isupper(ch)


/*
 * Boolean values.
 */
typedef	int	BOOL;

#define	FALSE	((BOOL) 0)
#define	TRUE	((BOOL) 1)


/*
 * Some other typedefs.
 */
typedef	unsigned long	ULONG;

typedef	struct	COLUMN	COLUMN;


/*
 * Argument structure.
 * This is for parsing command line arguments.
 */
typedef	struct
{
	int	count;			/* number of arguments */
	char **	table;			/* table of arguments */
} ARGS;


/*
 * Structure which holds information about a process.
 */
typedef	struct	PROC	PROC;

struct	PROC
{
	PROC *	next;			/* next process in list */
	BOOL	isvalid;		/* the process is existent */
	BOOL	isancient;		/* the process is before we started */
	BOOL	isnew;			/* the process is just created */
	BOOL	isactive;		/* the process is active */
	BOOL	changed;		/* the process state has changed */
	BOOL	isshown;		/* the process is to be shown */
	BOOL	hascommand;		/* there is a real command line */
	ULONG	live_counter;		/* counter to mark existant procs */
	ULONG	run_order;		/* counter value when last active */
	time_t	last_saved_time;	/* time status was copied to old */
	time_t	last_active_time;	/* time process last was active */
	time_t	last_sync_time;		/* time status was last synchonized */
	int	pid;			/* process id */
	int	uid;			/* effective user id */
	int	gid;			/* effective group id */
	int	state;			/* process state character */
	int	parent_pid;		/* parent pid */
	int	process_group;		/* process group */
	int	session_id;		/* session id */
	int	tty_dev;		/* controlling terminal's device */
	int	tty_pgrp;		/* process group of terminal */
	ULONG	flags;			/* kernel flags */
	long	min_flt;		/* minor page faults */
	long	maj_flt;		/* major page faults */
	long	cmin_flt;		/* child minor page faults */
	long	cmaj_flt;		/* child major page faults */
	long	utime;			/* user runtime in jiffies */
	long	stime;			/* system runtime in jiffies */
	long	cutime;			/* child user runtime */
	long	cstime;			/* child system runtime */
	long	counter;		/* scheduling counter */
	long	priority;		/* run priority */
	long	timeout;		/* timeout */
	long	it_real_value;		/* jiffies sleeping for */
	long	vsize;			/* virtual size of process in bytes */
	long	rss;			/* resident size in clicks */
	long	rss_limit;		/* resident size limit */
	ULONG	start_time_ticks;	/* jiffies uptime when started */
	ULONG	start_time_clock;	/* time when started */
	ULONG	start_code;		/* beginning address of code */
	ULONG	end_code;		/* ending address of code */
	ULONG	start_stack;		/* starting address of stack */
	ULONG	esp;			/* spack pointer */
	ULONG	eip;			/* instruction pointer */
	ULONG	signal;			/* current signal */
	ULONG	sigblock;		/* signals to block */
	ULONG	sigignore;		/* signals to ignore */
	ULONG	sigcatch;		/* signals to catch */
	ULONG	wchan;			/* address sleeping on */
	int	percent_cpu;		/* percentage of cpu used */
	int	percent_memory;		/* percentage of memory used */
	int	openfiles;		/* number of open files */
	dev_t	cwd_dev;		/* device of current directory */
	ino_t	cwd_inode;		/* inode of current directory */
	dev_t	root_dev;		/* device of root directory */
	ino_t	root_inode;		/* inode of root directory */
	dev_t	exec_dev;		/* device of executable */
	ino_t	exec_inode;		/* inode of executable */
	dev_t	stdio_dev[3];		/* devices of stdin, stdout, stderr */
	ino_t	stdio_inode[3];		/* inodes of stdin, stdout, stderr */
	int	command_length;		/* length of command line */
	int	environment_length;	/* length of environment */
	char *	command;		/* command line */
	char *	environment;		/* environment */
	char	program[MAX_PROGRAM_LEN + 2];	/* program name */
	char	command_buffer[BUF_COMMAND_LEN + 2];	/* command buffer */

	/*
	 * Status which is saved in order to determine active processes
	 * even if they are currently sleeping for this snapshot.
	 * A process is active if any of this differs from the current
	 * value.
	 */
	int	old_state;
	int	old_flags;
	long	old_min_flt;
	long	old_maj_flt;
	long	old_utime;
	long	old_stime;
	long	old_start_time_ticks;
	ULONG	old_end_code;	
	ULONG	old_esp;
	ULONG	old_eip;
	ULONG	old_wchan;
};

#define	NULL_PROC	((PROC *) 0)


/*
 * Column justification definitions.
 */
typedef	int	JUSTIFY;

#define	LEFT	((JUSTIFY) 0)
#define	RIGHT	((JUSTIFY) 1)
#define	CENTER	((JUSTIFY) 2)


/*
 * Values returned by expressions.
 */
typedef	struct	{
	int	type;		/* type of value */
	long	intval;		/* integer value */
	char *	strval;		/* string value */
	COLUMN *column;		/* column value */
} VALUE;


#define	VALUE_BAD	0
#define	VALUE_NONE	1
#define	VALUE_NUMBER	2
#define	VALUE_STRING	3
#define	VALUE_COLUMN	4
#define	VALUE_BOOLEAN	5

#define	TWOVAL(first, second)	(((first) * 10) + (second))


/*
 * Flags for what a column requires to be collected.
 * These flags are OR'd together.
 */
typedef	unsigned int	USEFLAG;

#define	USE_NONE	((USEFLAG) 0x000)
#define	USE_INIT	((USEFLAG) 0x001)
#define	USE_DEV_NAME	((USEFLAG) 0x002)
#define	USE_OPEN_FILE	((USEFLAG) 0x004)
#define	USE_CURR_DIR	((USEFLAG) 0x008)
#define	USE_COMMAND	((USEFLAG) 0x010)
#define	USE_SELF	((USEFLAG) 0x020)
#define	USE_STDIN	((USEFLAG) 0x040)
#define	USE_STDOUT	((USEFLAG) 0x080)
#define	USE_STDERR	((USEFLAG) 0x100)
#define	USE_ENVIRON	((USEFLAG) 0x200)
#define	USE_ROOT_DIR	((USEFLAG) 0x400)
#define	USE_EXEC_INODE	((USEFLAG) 0x800)


/*
 * Structure for one column that can be displayed.
 */
struct	COLUMN	{
	char *	name;		/* column name for commands */
	char *	heading;	/* heading string for column */
	int	initwidth;	/* initial minimum field width of column */
	int	width;		/* actual minimum field width of column */
	JUSTIFY	justify;	/* how value is justified in column */
	USEFLAG	useflag;	/* flags indicating what data column uses */
	char *	(*showfunc)(PROC *proc);
	int	(*sortfunc)(PROC *proc1, PROC *proc2);
	void	(*evalfunc)(PROC *proc, VALUE *retval);
	BOOL	(*testfunc)(PROC *proc);
};


/*
 * Identifiers for the different types of macros.
 * Some features of these macro types are built-in.
 * These definitions cannot be changed in isolation.
 */
typedef	int	MACRO_TYPE;

#define	MACRO_TYPE_NONE		((MACRO_TYPE) -1)
#define	MACRO_TYPE_OPTION	((MACRO_TYPE) 0)
#define	MACRO_TYPE_COLUMN	((MACRO_TYPE) 1)
#define	MACRO_TYPE_EXPRESSION	((MACRO_TYPE) 2)


/*
 * List of columns being shown.
 */
extern	int	show_count;
extern	COLUMN *show_list[MAX_COLUMNS];


/*
 * Other global variables.
 */
extern	ULONG		start_uptime;		/* uptime jiffies at start */
extern	time_t		start_time;		/* clock time at start */
extern	time_t		current_time;		/* current clock time */
extern	long		total_memory_clicks;	/* amount of total memory */
extern	ULONG		live_counter;		/* counter for live procs */
extern	BOOL		ancient_flag;		/* seeing pre-existing procs */
extern	BOOL		no_self;		/* don't show myself */
extern	BOOL		no_root;		/* don't show root procs */
extern	BOOL		no_header;		/* don't show column header */
extern	BOOL		my_procs;		/* only show my procs */
extern	BOOL		active_only;		/* only show active procs */
extern	BOOL		clear_screen;		/* clear screen each loop */
extern	BOOL		loop;			/* loop showing status */
extern	BOOL		vertical;		/* vertical output format */
extern	BOOL		top_auto;		/* autosize height for top */
extern	BOOL		use_open_files;		/* using open file info */
extern	BOOL		use_current_directory;	/* using current dir info */
extern	BOOL		use_root_directory;	/* using root dir info */
extern	BOOL		use_exec_inode;		/* using executable info */
extern	BOOL		use_device_names;	/* using device name info */
extern	BOOL		use_init;		/* using initial sleep */
extern	BOOL		use_command;		/* using command line info */
extern	BOOL		use_self;		/* using my own proc info */
extern	BOOL		use_environment;	/* using environment info */
extern	BOOL		use_stdio[3];		/* using various stdio info */
extern	dev_t		null_dev;		/* device of /dev/null */
extern	ino_t		null_inode;		/* inode of /dev/null */
extern	int		proc_alloc_count;	/* allocated proc structures */
extern	int		active_time;		/* seconds for active procs */
extern	int		pid_count;		/* pids in pid_list */
extern	int		user_count;		/* users in user_list */
extern	int		group_count;		/* groups in group_list */
extern	int		my_pid;			/* my pid */
extern	int		my_uid;			/* my real user id */
extern	int		my_gid;			/* my real group id */
extern	int		width;			/* width of output */
extern	int		height;			/* height of output */
extern	int		separation;		/* blanks between columns */
extern	int		sleep_time;		/* seconds between loops */
extern	int		sync_time;		/* seconds between syncs */
extern	int		init_time;		/* seconds for initial sleep */
extern	int		top_count;		/* number of procs for top */
extern	PROC *		process_list;		/* list of existent procs */
extern	PROC *		free_process_list;	/* free proc structure list */
extern	char		empty_string[4];	/* empty string */
extern	int		pid_list[MAX_PIDS];	/* pids to be shown */
extern	int		user_list[MAX_USERS];	/* user ids to be shown */
extern	int		group_list[MAX_GROUPS];	/* group ids to be shown */


/*
 * Global procedures.
 */
extern	BOOL	PatternMatch(char *text, char *pattern);
extern	char *	AllocateSharedString(char *str, int len);
extern	void	FreeSharedString(char *str);
extern	char *	AllocTempString(int len);
extern	char *	CopyTempString(char *oldcp);
extern	void	FreeTempStrings(void);
extern	void	DefaultAllOptions(void);
extern	void	DefaultColumns(void);
extern	void	DefaultColumnWidths(void);
extern	ULONG	GetUptime(void);
extern	BOOL	ParseSystemInitFile(void);
extern	BOOL	ParseUserInitFile(void);
extern	BOOL	ParseFile(char *name);
extern	BOOL	ParseOptions(ARGS *ap, int depth);
extern	BOOL	ExpandOptionName(char *name, int depth);
extern	long	GetDecimalNumber(char **cpp);
extern	void	ClearCondition(void);
extern	BOOL	ParseCondition(char *str);
extern	USEFLAG	GetConditionUseFlags(void);
extern	USEFLAG	GetSortingUseFlags(void);
extern	void	ClearSorting(void);
extern	void	SortProcesses(void);
extern	BOOL	AppendColumnSort(ARGS *ap, BOOL reverse);
extern	BOOL	AppendExpressionSort(ARGS *ap, BOOL reverse);
extern	PROC *	FindProcess(int pid);
extern	BOOL	InitializeProcessData(void);
extern	BOOL	InitializeDisplay(void);
extern	void	ScanProcesses(void);
extern	void	CheckActiveProcess(PROC *proc);
extern	void	RemoveDeadProcesses(void);
extern	void	InitialProcessScan(void);
extern	BOOL	IsShownProcess(PROC *proc);
extern	void	ShowSelectedProcesses(void);
extern	void	ListMacros(void);
extern	COLUMN *FindColumn(char *name);
extern	void	ListColumns(void);
extern	void	CollectUserNames(void);
extern	void	CollectGroupNames(void);
extern	void	CollectDeviceNames(void);
extern	void	CollectStaticSystemInfo(void);
extern	int	FindUserId(char *name);
extern	int	FindGroupId(char *name);
extern	char *	FindUserName(int uid);
extern	char *	FindGroupName(int gid);
extern	char *	FindDeviceName(int devid);
extern	char *	FindDeviceFromInode(dev_t dev, ino_t inode);
extern	void	UpdateTimes(void);
extern	int	ExpandArguments(ARGS *ap, char **table, int tablelen);
extern	BOOL	MacroExists(MACRO_TYPE id, char *name);
extern	BOOL	DefineMacro(MACRO_TYPE id, char *name, char *str);
extern	BOOL	ExpandMacro(MACRO_TYPE id, char *name, ARGS *retargs);
extern	BOOL	CompareValues(VALUE leftval, VALUE rightval, int *result);
extern	void	MakePrintable(char *cp, int len);

#endif
