/*
 * Configurable ps-like program.
 * Column data and display routines.
 *
 * Copyright (c) 1995 David I. Bell
 * Permission is granted to use, distribute, or modify this source,
 * provided that this copyright notice remains intact.
 */

#include <signal.h>
#include <time.h>
#include "ips.h"


/*
 * Macro to define the column functions and to fill in the COLUMN
 * structure with values.
 */
#define	DEFCOL(symbol, name, header, width, justify, flags)	\
	static	char *	Show##symbol(PROC *proc);		\
	static	int	Sort##symbol(PROC *proc1, PROC *proc2);	\
	static	void	Eval##symbol(PROC *proc, VALUE *val);	\
	static	BOOL	Test##symbol(PROC *proc);		\
								\
	static	COLUMN	Column##symbol = {			\
		#name, #header, width, width, justify, flags,	\
		Show##symbol,	Sort##symbol,			\
		Eval##symbol,	Test##symbol			\
	};


/*
 * The column definitions.
 */
DEFCOL(Pid,		pid,		Pid,
	5,	RIGHT,	USE_NONE)

DEFCOL(State,		state,		Stat,
	4,	CENTER,	USE_NONE)

DEFCOL(ParentPid,	parentpid,	PPid,
	5,	RIGHT,	USE_NONE)

DEFCOL(SystemTime,	systemtime,	System Time,
	10,	RIGHT,	USE_NONE)

DEFCOL(UserTime,	usertime,	User Time,
	10,	RIGHT,	USE_NONE)

DEFCOL(RunTime,		runtime,	Runtime,
	10,	RIGHT,	USE_NONE)

DEFCOL(ChildRunTime,	childruntime,	ChildRuntime,
	10,	RIGHT,	USE_NONE)

DEFCOL(Eip,		eip,		EIP,
	8,	RIGHT,	USE_NONE)

DEFCOL(Esp,		esp,		ESP,
	8,	RIGHT,	USE_NONE)

DEFCOL(WaitChannel,	waitchannel,	WaitChan,
	8,	RIGHT,	USE_NONE)

DEFCOL(Program,		program,	Program,
	12,	LEFT,	USE_NONE)

DEFCOL(Command,		command,	Command,
	20,	LEFT,	USE_COMMAND)

DEFCOL(Environment,	environment,	Environment,
	25,	LEFT,	USE_ENVIRON)

DEFCOL(ProcessGroup,	processgroup,	PGrp,
	5,	RIGHT,	USE_NONE)

DEFCOL(TtyProcessGroup,	ttyprocessgroup,TtyPG,
	5,	RIGHT,	USE_NONE)

DEFCOL(IdleTime,	idletime,	Idle,
	8,	RIGHT,	USE_INIT)

DEFCOL(TtyName,		ttyname,	TtyName,
	7,	LEFT,	USE_DEV_NAME)

DEFCOL(TtyDevice,	ttydevice,	TtyD,
	4,	RIGHT,	USE_NONE)

DEFCOL(UserName,	user,		User,
	8,	LEFT,	USE_NONE)

DEFCOL(UserId,		uid,		Uid,
	5,	RIGHT,	USE_NONE)

DEFCOL(GroupName,	group,		Group,
	8,	LEFT,	USE_NONE)

DEFCOL(GroupId,		gid,		Gid,
	5,	RIGHT,	USE_NONE)

DEFCOL(PercentCpu,	percentcpu,	%Cpu,
	5,	RIGHT,	USE_INIT)

DEFCOL(PercentMemory,	percentmemory,	%Mem,
	4,	RIGHT,	USE_NONE)

DEFCOL(Rss,		residentsetsize,Rss,
	5,	RIGHT,	USE_NONE)

DEFCOL(StartTime,	starttime,	Start Time,
	10,	LEFT,	USE_NONE)

DEFCOL(Age,		age,		Age,
	8,	RIGHT,	USE_NONE)

DEFCOL(Flags,		flags,		Flags,
	8,	RIGHT,	USE_NONE)

DEFCOL(PageFaults,	pagefaults,	Faults,
	10,	RIGHT,	USE_NONE)

DEFCOL(MinorPageFaults,	minorpagefaults,MinorFault,
	10,	RIGHT,	USE_NONE)

DEFCOL(MajorPageFaults,	majorpagefaults,MajorFault,
	10,	RIGHT,	USE_NONE)

DEFCOL(SignalCatch,	signalcatch,	SigCatch,
	8,	RIGHT,	USE_NONE)

DEFCOL(SignalIgnore,	signalignore,	SigIgnor,
	8,	RIGHT,	USE_NONE)

DEFCOL(SignalBlock,	signalblock,	SigBlock,
	8,	RIGHT,	USE_NONE)

DEFCOL(OpenFileCount,	openfiles,	Files,
	5,	RIGHT,	USE_OPEN_FILE)

DEFCOL(RunOrder,	runorder,	RunOrder,
	10,	RIGHT,	USE_NONE)

DEFCOL(CurrentDirectory,currentdirectory,Current Dir,
	11,	LEFT,	USE_CURR_DIR)

DEFCOL(RootDirectory,	rootdirectory,	Root Dir,
	11,	LEFT,	USE_ROOT_DIR)

DEFCOL(Executable,	executable,	Executable,
	11,	LEFT,	USE_EXEC_INODE)

DEFCOL(Summary,		summary,	Summary,
	14,	LEFT,	USE_NONE)

DEFCOL(Priority,	priority,	Pri,
	3,	RIGHT,	USE_NONE)

DEFCOL(Size,		size,		Size,
	5,	RIGHT,	USE_NONE)

DEFCOL(RealTimer,	realtimer,	RealTimer,
	10,	RIGHT,	USE_NONE)

DEFCOL(Stdin,		stdin,		Stdin,
	11,	LEFT,	USE_STDIN | USE_DEV_NAME)

DEFCOL(Stdout,		stdout,		Stdout,
	11,	LEFT,	USE_STDOUT | USE_DEV_NAME)

DEFCOL(Stderr,		stderr,		Stderr,
	11,	LEFT,	USE_STDERR | USE_DEV_NAME)

DEFCOL(Stdio,		stdio,		Stdio,
	5,	CENTER,	USE_STDIN | USE_STDOUT | USE_STDERR | USE_DEV_NAME)

DEFCOL(Active,		active,		Active,
	5,	CENTER,	USE_INIT)


/*
 * Table of all columns.
 * The table is NULL terminated.
 */
static	COLUMN *	column_table[] = {
	&ColumnPid,
	&ColumnState,
	&ColumnParentPid,
	&ColumnSystemTime,
	&ColumnUserTime,
	&ColumnRunTime,
	&ColumnChildRunTime,
	&ColumnEip,
	&ColumnEsp,
	&ColumnWaitChannel,
	&ColumnProgram,
	&ColumnCommand,
	&ColumnEnvironment,
	&ColumnProcessGroup,
	&ColumnTtyProcessGroup,
	&ColumnIdleTime,
	&ColumnTtyName,
	&ColumnTtyDevice,
	&ColumnUserName,
	&ColumnGroupName,
	&ColumnUserId,
	&ColumnGroupId,
	&ColumnPercentCpu,
	&ColumnPercentMemory,
	&ColumnRss,
	&ColumnStartTime,
	&ColumnAge,
	&ColumnFlags,
	&ColumnPageFaults,
	&ColumnMinorPageFaults,
	&ColumnMajorPageFaults,
	&ColumnSignalCatch,
	&ColumnSignalIgnore,
	&ColumnSignalBlock,
	&ColumnOpenFileCount,
	&ColumnRunOrder,
	&ColumnCurrentDirectory,
	&ColumnRootDirectory,
	&ColumnExecutable,
	&ColumnSummary,
	&ColumnPriority,
	&ColumnSize,
	&ColumnRealTimer,
	&ColumnStdin,
	&ColumnStdout,
	&ColumnStderr,
	&ColumnStdio,
	&ColumnActive,
	NULL
};


/*
 * Default columns when nothing else has been set.
 */
static	COLUMN *	default_columns[] = {
	&ColumnPid,
	&ColumnParentPid,
	&ColumnTtyDevice,
	&ColumnUserName,
	&ColumnSummary,
	&ColumnRunTime,
	&ColumnCommand,
	NULL
};


/*
 * Temporary buffer to contain constructed column strings.
 */
static	char	show_buffer[80];


/*
 * Other static routines.
 */
static	void	TicksToString(char *buf, long ticks);
static	char *	GetStdioString(dev_t dev, ino_t inode);


/*
 * Default the widths of all the columns.
 */
void
DefaultColumnWidths(void)
{
	COLUMN **	columnptr;
	COLUMN *	column;

	for (columnptr = column_table; *columnptr; columnptr++) {
		column = *columnptr;

		column->width = column->initwidth;
	}
}


/*
 * List the available columns to stdout.
 */
void
ListColumns(void)
{
	COLUMN **	columnptr;
	COLUMN *	column;

	printf("Column Name          Displayed As    Width\n");
	printf("-----------          ------------    -----\n");

	for (columnptr = column_table; *columnptr; columnptr++) {
		column = *columnptr;

		printf("%-20s %-15s %3d\n", column->name, column->heading,
			column->width);
	}

	printf("\n");
	printf("Note: Column names may be abbreviated.  They are used in\n");
	printf("several ways: displaying, sorting, and for the cond option.\n");
}


/*
 * Set the default columns.
 */
void
DefaultColumns(void)
{
	COLUMN **	src;
	COLUMN **	dest;

	dest = show_list;
	src = default_columns;
	show_count = 0;

	while (*src) {
		*dest++ = *src++;
		show_count++;
	}
}


/*
 * Find the column with the given name.
 * Abbreviations are allowed.
 * Returns NULL if there was no match or it was ambiguous.
 */
COLUMN *
FindColumn(char *name)
{
	COLUMN **	columnptr;
	COLUMN *	column;
	COLUMN *	match;
	int		count;
	int		len;
	int		namelen;

	match = NULL;
	count = 0;

	namelen = strlen(name);

	for (columnptr = column_table; *columnptr; columnptr++) {
		column = *columnptr;

		len = strlen(column->name);

		if ((len < namelen) ||
			((memcmp(name, column->name, namelen) != 0)))
				continue;

		if (len == namelen)
			return column;

		match = column;
		count++;
	}

	if (count == 1)
		return match;

	return NULL;
}


/*
 * Functions to sort various columns based on comparing two processes.
 * This returns a negative value if the item for the first process is less
 * than the second process, a positive value if the item is greater, or
 * zero if the item is the same for both processes.
 */
static int
SortPid(PROC *proc1, PROC *proc2)
{
	return proc1->pid - proc2->pid;
}


static int
SortState(PROC *proc1, PROC *proc2)
{
	return proc1->state - proc2->state;
}


static int
SortParentPid(PROC *proc1, PROC *proc2)
{
	return proc1->parent_pid - proc2->parent_pid;
}


static int
SortSystemTime(PROC *proc1, PROC *proc2)
{
	return proc1->stime - proc2->stime;
}


static int
SortUserTime(PROC *proc1, PROC *proc2)
{
	return proc1->utime - proc2->utime;
}


static int
SortRunTime(PROC *proc1, PROC *proc2)
{
	return (proc1->stime + proc1->utime) - (proc2->stime + proc2->utime);
}


static int
SortChildRunTime(PROC *proc1, PROC *proc2)
{
	return (proc1->cstime + proc1->cutime) -
		(proc2->cstime + proc2->cutime);
}


static int
SortEip(PROC *proc1, PROC *proc2)
{
	return proc1->eip - proc2->eip;
}


static int
SortEsp(PROC *proc1, PROC *proc2)
{
	return proc1->esp - proc2->esp;
}


static int
SortWaitChannel(PROC *proc1, PROC *proc2)
{
	return proc1->wchan - proc2->wchan;
}


static int
SortProgram(PROC *proc1, PROC *proc2)
{
	return strcmp(proc1->program, proc2->program);
}


static int
SortCommand(PROC *proc1, PROC *proc2)
{
	return strcmp(proc1->command, proc2->command);
}


static int
SortEnvironment(PROC *proc1, PROC *proc2)
{
	return strcmp(proc1->environment, proc2->environment);
}


static int
SortProcessGroup(PROC *proc1, PROC *proc2)
{
	return proc1->process_group - proc2->process_group;
}


static int
SortTtyProcessGroup(PROC *proc1, PROC *proc2)
{
	return proc1->tty_pgrp - proc2->tty_pgrp;
}


static int
SortIdleTime(PROC *proc1, PROC *proc2)
{
	return proc2->last_active_time - proc1->last_active_time;
}


static int
SortTtyName(PROC *proc1, PROC *proc2)
{
	/*
	 * TODO: Should be alpha Sort
	 */
	return proc1->tty_dev - proc2->tty_dev;
}


static int
SortTtyDevice(PROC *proc1, PROC *proc2)
{
	return proc1->tty_dev - proc2->tty_dev;
}


static int
SortUserName(PROC *proc1, PROC *proc2)
{
	char *	name1;
	char *	name2;

	if (proc1->uid == proc2->uid)
		return 0;

	name1 = FindUserName(proc1->uid);
	name2 = FindUserName(proc2->uid);

	if (name1 && name2)
		return strcmp(name1, name2);

	if (!name1 && name2)
		return -1;

	if (name1 && !name2)
		return 1;

	return proc1->uid - proc2->uid;
}


static int
SortUserId(PROC *proc1, PROC *proc2)
{
	return proc1->uid - proc2->uid;
}


static int
SortGroupName(PROC *proc1, PROC *proc2)
{
	char *	name1;
	char *	name2;

	if (proc1->gid == proc2->gid)
		return 0;

	name1 = FindGroupName(proc1->gid);
	name2 = FindGroupName(proc2->gid);

	if (name1 && name2)
		return strcmp(name1, name2);

	if (!name1 && name2)
		return -1;

	if (name1 && !name2)
		return 1;

	return proc1->gid - proc2->gid;
}


static int
SortGroupId(PROC *proc1, PROC *proc2)
{
	return proc1->gid - proc2->gid;
}


static int
SortPercentCpu(PROC *proc1, PROC *proc2)
{
	return proc1->percent_cpu - proc2->percent_cpu;
}


static int
SortPercentMemory(PROC *proc1, PROC *proc2)
{
	return proc1->percent_memory - proc2->percent_memory;
}


static int
SortRss(PROC *proc1, PROC *proc2)
{
	return proc1->rss - proc2->rss;
}


static int
SortStartTime(PROC *proc1, PROC *proc2)
{
	return proc1->start_time_ticks - proc2->start_time_ticks;
}


static int
SortAge(PROC *proc1, PROC *proc2)
{
	return proc2->start_time_ticks - proc1->start_time_ticks;
}


static int
SortFlags(PROC *proc1, PROC *proc2)
{
	return proc1->flags - proc2->flags;
}


static int
SortPageFaults(PROC *proc1, PROC *proc2)
{
	return (proc1->min_flt + proc1->maj_flt) -
		(proc2->min_flt + proc2->maj_flt);
}


static int
SortMinorPageFaults(PROC *proc1, PROC *proc2)
{
	return proc1->min_flt - proc2->min_flt;
}


static int
SortMajorPageFaults(PROC *proc1, PROC *proc2)
{
	return proc1->maj_flt - proc2->maj_flt;
}


static int
SortSignalCatch(PROC *proc1, PROC *proc2)
{
	return proc1->sigcatch - proc2->sigcatch;
}


static int
SortSignalIgnore(PROC *proc1, PROC *proc2)
{
	return proc1->sigignore - proc2->sigignore;
}


static int
SortSignalBlock(PROC *proc1, PROC *proc2)
{
	return proc1->sigblock - proc2->sigblock;
}


static int
SortOpenFileCount(PROC *proc1, PROC *proc2)
{
	if (proc1->openfiles < proc2->openfiles)
		return -1;

	if (proc1->openfiles > proc2->openfiles)
		return 1;

	return 0;
}


static int
SortRunOrder(PROC *proc1, PROC *proc2)
{
	if (proc1->run_order < proc2->run_order)
		return -1;

	if (proc1->run_order > proc2->run_order)
		return 1;

	return 0;
}


static int
SortCurrentDirectory(PROC *proc1, PROC *proc2)
{
	if (proc1->cwd_dev < proc2->cwd_dev)
		return -1;

	if (proc1->cwd_dev > proc2->cwd_dev)
		return 1;

	if (proc1->cwd_inode < proc2->cwd_inode)
		return -1;

	if (proc1->cwd_inode > proc2->cwd_inode)
		return 1;

	return 0;
}


static int
SortRootDirectory(PROC *proc1, PROC *proc2)
{
	if (proc1->root_dev < proc2->root_dev)
		return -1;

	if (proc1->root_dev > proc2->root_dev)
		return 1;

	if (proc1->root_inode < proc2->root_inode)
		return -1;

	if (proc1->root_inode > proc2->root_inode)
		return 1;

	return 0;
}


static int
SortExecutable(PROC *proc1, PROC *proc2)
{
	if (proc1->exec_dev < proc2->exec_dev)
		return -1;

	if (proc1->exec_dev > proc2->exec_dev)
		return 1;

	if (proc1->exec_inode < proc2->exec_inode)
		return -1;

	if (proc1->exec_inode > proc2->exec_inode)
		return 1;

	return 0;
}


static int
SortStdin(PROC *proc1, PROC *proc2)
{
	if (proc1->stdio_dev[0] < proc2->stdio_dev[0])
		return -1;

	if (proc1->stdio_dev[0] > proc2->stdio_dev[0])
		return 1;

	if (proc1->stdio_inode[0] < proc2->stdio_inode[0])
		return -1;

	if (proc1->stdio_inode[0] > proc2->stdio_inode[0])
		return 1;

	return 0;
}


static int
SortStdout(PROC *proc1, PROC *proc2)
{
	if (proc1->stdio_dev[1] < proc2->stdio_dev[1])
		return -1;

	if (proc1->stdio_dev[1] > proc2->stdio_dev[1])
		return 1;

	if (proc1->stdio_inode[1] < proc2->stdio_inode[1])
		return -1;

	if (proc1->stdio_inode[1] > proc2->stdio_inode[1])
		return 1;

	return 0;
}


static int
SortStderr(PROC *proc1, PROC *proc2)
{
	if (proc1->stdio_dev[2] < proc2->stdio_dev[2])
		return -1;

	if (proc1->stdio_dev[2] > proc2->stdio_dev[2])
		return 1;

	if (proc1->stdio_inode[2] < proc2->stdio_inode[2])
		return -1;

	if (proc1->stdio_inode[2] > proc2->stdio_inode[2])
		return 1;

	return 0;
}


static int
SortStdio(PROC *proc1, PROC *proc2)
{
	char	buf[8];

	strcpy(buf, ShowStdio(proc1));

	return strcmp(buf, ShowStdio(proc2));
}


static int
SortSummary(PROC *proc1, PROC *proc2)
{
	char	buf[30];

	strcpy(buf, ShowSummary(proc1));

	return strcmp(buf, ShowSummary(proc2));
}


static int
SortPriority(PROC *proc1, PROC *proc2)
{
	if (proc1->priority < proc2->priority)
		return -1;

	if (proc1->priority > proc2->priority)
		return 1;

	return 0;
}


static int
SortSize(PROC *proc1, PROC *proc2)
{
	if (proc1->vsize < proc2->vsize)
		return -1;

	if (proc1->vsize > proc2->vsize)
		return 1;

	return 0;
}


static int
SortRealTimer(PROC *proc1, PROC *proc2)
{
	if (proc1->it_real_value < proc2->it_real_value)
		return -1;

	if (proc1->it_real_value > proc2->it_real_value)
		return 1;

	return 0;
}


static int
SortActive(PROC *proc1, PROC *proc2)
{
	if (!proc1->isactive && proc2->isactive)
		return -1;

	if (proc1->isactive && !proc2->isactive)
		return 1;

	return 0;
}


/*
 * Routines to evaluate a column, and return it with the specified type.
 * (This is either a string or a number.)
 */
static void
EvalPid(PROC *proc, VALUE *val)
{
	val->type = VALUE_NUMBER;
	val->intval = proc->pid;
}


static void
EvalState(PROC *proc, VALUE *val)
{
	val->type = VALUE_STRING;
	val->strval = AllocTempString(2);
	val->strval[0] = proc->state;
	val->strval[1] = '\0';
}


static void
EvalParentPid(PROC *proc, VALUE *val)
{
	val->type = VALUE_NUMBER;
	val->intval = proc->parent_pid;
}


static void
EvalSystemTime(PROC *proc, VALUE *val)
{
	val->type = VALUE_NUMBER;
	val->intval = proc->stime;
}


static void
EvalUserTime(PROC *proc, VALUE *val)
{
	val->type = VALUE_NUMBER;
	val->intval = proc->utime;
}


static void
EvalRunTime(PROC *proc, VALUE *val)
{
	val->type = VALUE_NUMBER;
	val->intval = proc->stime + proc->utime;
}


static void
EvalChildRunTime(PROC *proc, VALUE *val)
{
	val->type = VALUE_NUMBER;
	val->intval = proc->cstime + proc->cutime;
}


static void
EvalEip(PROC *proc, VALUE *val)
{
	val->type = VALUE_NUMBER;
	val->intval = proc->eip;
}


static void
EvalEsp(PROC *proc, VALUE *val)
{
	val->type = VALUE_NUMBER;
	val->intval = proc->eip;
}


static void
EvalWaitChannel(PROC *proc, VALUE *val)
{
	val->type = VALUE_NUMBER;
	val->intval = proc->wchan;
}


static void
EvalProgram(PROC *proc, VALUE *val)
{
	val->type = VALUE_STRING;
	val->strval = proc->program;
}


static void
EvalCommand(PROC *proc, VALUE *val)
{
	val->type = VALUE_STRING;
	val->strval = proc->command;
}


static void
EvalEnvironment(PROC *proc, VALUE *val)
{
	val->type = VALUE_STRING;
	val->strval = proc->environment;
}


static void
EvalProcessGroup(PROC *proc, VALUE *val)
{
	val->type = VALUE_NUMBER;
	val->intval = proc->process_group;
}


static void
EvalTtyProcessGroup(PROC *proc, VALUE *val)
{
	val->type = VALUE_NUMBER;
	val->intval = proc->tty_pgrp;
}


static void
EvalIdleTime(PROC *proc, VALUE *val)
{
	val->type = VALUE_NUMBER;
	val->intval = (current_time - proc->last_active_time) / 60;
}


static void
EvalTtyName(PROC *proc, VALUE *val)
{
	val->type = VALUE_STRING;
	val->strval = FindDeviceName(proc->tty_dev);

	if (val->strval == NULL)
		val->strval = "";
}


static void
EvalTtyDevice(PROC *proc, VALUE *val)
{
	val->type = VALUE_NUMBER;
	val->intval = proc->tty_dev;
}


static void
EvalUserName(PROC *proc, VALUE *val)
{
	val->type = VALUE_STRING;
	val->strval = FindUserName(proc->uid);

	if (val->strval == NULL)
		val->strval = "";
}


static void
EvalGroupName(PROC *proc, VALUE *val)
{
	val->type = VALUE_STRING;
	val->strval = FindGroupName(proc->uid);

	if (val->strval == NULL)
		val->strval = "";
}


static void
EvalUserId(PROC *proc, VALUE *val)
{
	val->type = VALUE_NUMBER;
	val->intval = proc->uid;
}


static void
EvalGroupId(PROC *proc, VALUE *val)
{
	val->type = VALUE_NUMBER;
	val->intval = proc->gid;
}


static void
EvalPercentCpu(PROC *proc, VALUE *val)
{
	val->type = VALUE_NUMBER;
	val->intval = proc->percent_cpu;
}


static void
EvalPercentMemory(PROC *proc, VALUE *val)
{
	val->type = VALUE_NUMBER;
	val->intval = proc->percent_memory;
}


static void
EvalRss(PROC *proc, VALUE *val)
{
	val->type = VALUE_NUMBER;
	val->intval = proc->rss * 4;
}


static void
EvalStartTime(PROC *proc, VALUE *val)
{
	val->type = VALUE_NUMBER;
	val->intval = proc->start_time_clock;
}


static void
EvalAge(PROC *proc, VALUE *val)
{
	val->type = VALUE_NUMBER;
	val->intval = current_time - proc->start_time_clock;

	if (val->intval < 0)
		val->intval = 0;
}


static void
EvalFlags(PROC *proc, VALUE *val)
{
	val->type = VALUE_NUMBER;
	val->intval = proc->flags;
}


static void
EvalPageFaults(PROC *proc, VALUE *val)
{
	val->type = VALUE_NUMBER;
	val->intval = proc->min_flt + proc->maj_flt;
}


static void
EvalMinorPageFaults(PROC *proc, VALUE *val)
{
	val->type = VALUE_NUMBER;
	val->intval = proc->min_flt;
}


static void
EvalMajorPageFaults(PROC *proc, VALUE *val)
{
	val->type = VALUE_NUMBER;
	val->intval = proc->maj_flt;
}


static void
EvalSignalCatch(PROC *proc, VALUE *val)
{
	val->type = VALUE_NUMBER;
	val->intval = proc->sigcatch;
}


static void
EvalSignalIgnore(PROC *proc, VALUE *val)
{
	val->type = VALUE_NUMBER;
	val->intval = proc->sigignore;
}


static void
EvalSignalBlock(PROC *proc, VALUE *val)
{
	val->type = VALUE_NUMBER;
	val->intval = proc->sigblock;
}


static void
EvalOpenFileCount(PROC *proc, VALUE *val)
{
	val->type = VALUE_NUMBER;
	val->intval = proc->openfiles;
}


static void
EvalRunOrder(PROC *proc, VALUE *val)
{
	val->type = VALUE_NUMBER;
	val->intval = proc->run_order;
}


static void
EvalCurrentDirectory(PROC *proc, VALUE *val)
{
	val->type = VALUE_STRING;
	val->strval = "";

	if ((proc->cwd_dev == 0) && (proc->cwd_inode == 0))
		return;

	val->strval = AllocTempString(20);

	sprintf(val->strval, "%04x:%u", proc->cwd_dev,
		(unsigned) proc->cwd_inode);
}


static void
EvalRootDirectory(PROC *proc, VALUE *val)
{
	val->type = VALUE_STRING;
	val->strval = "";

	if ((proc->root_dev == 0) && (proc->root_inode == 0))
		return;

	val->strval = AllocTempString(20);

	sprintf(val->strval, "%04x:%u", proc->root_dev,
		(unsigned) proc->root_inode);
}


static void
EvalExecutable(PROC *proc, VALUE *val)
{
	val->type = VALUE_STRING;
	val->strval = "";

	if ((proc->exec_dev == 0) && (proc->exec_inode == 0))
		return;

	val->strval = AllocTempString(20);

	sprintf(val->strval, "%04x:%u", proc->exec_dev,
		(unsigned) proc->exec_inode);
}


static void
EvalStdin(PROC *proc, VALUE *val)
{
	val->type = VALUE_STRING;
	val->strval = "";

	if ((proc->stdio_dev[0] == 0) && (proc->stdio_inode[0] == 0))
		return;

	val->strval = AllocTempString(20);

	sprintf(val->strval, "%04x:%u", proc->stdio_dev[0],
		(unsigned) proc->stdio_inode[0]);
}


static void
EvalStdout(PROC *proc, VALUE *val)
{
	val->type = VALUE_STRING;
	val->strval = "";

	if ((proc->stdio_dev[1] == 0) && (proc->stdio_inode[1] == 0))
		return;

	val->strval = AllocTempString(20);

	sprintf(val->strval, "%04x:%u", proc->stdio_dev[1],
		(unsigned) proc->stdio_inode[1]);
}


static void
EvalStderr(PROC *proc, VALUE *val)
{
	val->type = VALUE_STRING;
	val->strval = "";

	if ((proc->stdio_dev[2] == 0) && (proc->stdio_inode[2] == 0))
		return;

	val->strval = AllocTempString(20);

	sprintf(val->strval, "%04x:%u", proc->stdio_dev[2],
		(unsigned) proc->stdio_inode[2]);
}


static void
EvalStdio(PROC *proc, VALUE *val)
{
	val->type = VALUE_STRING;
	val->strval = CopyTempString(ShowStdio(proc));
}


static void
EvalSummary(PROC *proc, VALUE *val)
{
	val->type = VALUE_STRING;
	val->strval = CopyTempString(ShowSummary(proc));
}


static void
EvalPriority(PROC *proc, VALUE *val)
{
	val->type = VALUE_NUMBER;
	val->intval = proc->priority;
}


static void
EvalSize(PROC *proc, VALUE *val)
{
	val->type = VALUE_NUMBER;
	val->intval = (proc->vsize + 1023) / 1024;
}


static void
EvalRealTimer(PROC *proc, VALUE *val)
{
	val->type = VALUE_NUMBER;
	val->intval = proc->it_real_value;
}


static void
EvalActive(PROC *proc, VALUE *val)
{
	val->type = VALUE_BOOLEAN;
	val->intval = proc->isactive;
}


/*
 * Routines to test whether a column value is "TRUE" in some sense.
 * The meaning differs from column to column, but generally means that
 * the column value is nonzero or non-null.  These functions are used when
 * the "test" attribute is used on a column name.
 */
static BOOL
TestPid(PROC *proc)
{
	return (proc->pid != 0);
}


static BOOL
TestState(PROC *proc)
{
	return (proc->state != 'Z');
}


static BOOL
TestParentPid(PROC *proc)
{
	return (proc->parent_pid && (proc->parent_pid != 1));
}


static BOOL
TestSystemTime(PROC *proc)
{
	return (proc->stime != 0);
}


static BOOL
TestUserTime(PROC *proc)
{
	return (proc->utime != 0);
}


static BOOL
TestRunTime(PROC *proc)
{
	return ((proc->stime != 0) || (proc->utime != 0));
}


static BOOL
TestChildRunTime(PROC *proc)
{
	return ((proc->cstime != 0) || (proc->cutime != 0));
}


static BOOL
TestEip(PROC *proc)
{
	return (proc->eip != 0);
}


static BOOL
TestEsp(PROC *proc)
{
	return (proc->esp != 0);
}


static BOOL
TestWaitChannel(PROC *proc)
{
	return (proc->wchan != 0);
}


static BOOL
TestProgram(PROC *proc)
{
	return (proc->program[0] != '\0');
}


static BOOL
TestCommand(PROC *proc)
{
	return proc->hascommand;
}


static BOOL
TestEnvironment(PROC *proc)
{
	return (proc->environment[0] != '\0');
}


static BOOL
TestProcessGroup(PROC *proc)
{
	return (proc->process_group != 0);
}


static BOOL
TestTtyProcessGroup(PROC *proc)
{
	return ((proc->tty_pgrp != 0) && (proc->tty_pgrp != -1));
}


static BOOL
TestIdleTime(PROC *proc)
{
	return (proc->last_active_time != current_time);
}


static BOOL
TestTtyName(PROC *proc)
{
	return (proc->tty_dev > 0);
}


static BOOL
TestTtyDevice(PROC *proc)
{
	return (proc->tty_dev > 0);
}


static BOOL
TestUserName(PROC *proc)
{
	return (FindUserName(proc->uid) != NULL);
}


static BOOL
TestGroupName(PROC *proc)
{
	return (FindGroupName(proc->gid) != NULL);
}


static BOOL
TestUserId(PROC *proc)
{
	return (proc->uid != 0);
}


static BOOL
TestGroupId(PROC *proc)
{
	return (proc->gid != 0);
}


static BOOL
TestPercentCpu(PROC *proc)
{
	return (proc->percent_cpu != 0);
}


static BOOL
TestPercentMemory(PROC *proc)
{
	return (proc->percent_memory != 0);
}


static BOOL
TestRss(PROC *proc)
{
	return (proc->rss != 0);
}


static BOOL
TestStartTime(PROC *proc)
{
	return (proc->start_time_ticks != 0);
}


static BOOL
TestAge(PROC *proc)
{
	return (proc->start_time_clock < current_time);
}


static BOOL
TestFlags(PROC *proc)
{
	return (proc->flags != 0);
}


static BOOL
TestPageFaults(PROC *proc)
{
	return ((proc->min_flt != 0) || (proc->maj_flt != 0));
}


static BOOL
TestMinorPageFaults(PROC *proc)
{
	return (proc->min_flt != 0);
}


static BOOL
TestMajorPageFaults(PROC *proc)
{
	return (proc->maj_flt != 0);
}


static BOOL
TestSignalCatch(PROC *proc)
{
	return (proc->sigcatch != 0);
}


static BOOL
TestSignalIgnore(PROC *proc)
{
	return (proc->sigignore != 0);
}


static BOOL
TestSignalBlock(PROC *proc)
{
	return (proc->sigblock != 0);
}


static BOOL
TestOpenFileCount(PROC *proc)
{
	return (proc->openfiles > 0);
}


static BOOL
TestRunOrder(PROC *proc)
{
	return (proc->run_order > 0);
}


static BOOL
TestCurrentDirectory(PROC *proc)
{
	return (proc->cwd_dev != 0) || (proc->cwd_inode != 0);
}


static BOOL
TestRootDirectory(PROC *proc)
{
	return (proc->root_dev != 0) || (proc->root_inode != 0);
}


static BOOL
TestExecutable(PROC *proc)
{
	return (proc->exec_dev != 0) || (proc->exec_inode != 0);
}


static BOOL
TestStdin(PROC *proc)
{
	return (proc->stdio_dev[0] != 0) || (proc->stdio_inode[0] != 0);
}


static BOOL
TestStdout(PROC *proc)
{
	return (proc->stdio_dev[1] != 0) || (proc->stdio_inode[1] != 0);
}


static BOOL
TestStderr(PROC *proc)
{
	return (proc->stdio_dev[2] != 0) || (proc->stdio_inode[2] != 0);
}


static BOOL
TestStdio(PROC *proc)
{
	return (((proc->stdio_dev[0] != 0) || (proc->stdio_inode[0] != 0))
		&& ((proc->stdio_dev[1] != 0) || (proc->stdio_inode[1] != 0))
		&& ((proc->stdio_dev[2] != 0) || (proc->stdio_inode[2] != 0)));
}


static BOOL
TestSummary(PROC *proc)
{
	return TRUE;
}


static BOOL
TestPriority(PROC *proc)
{
	return (proc->priority != PZERO);
}


static BOOL
TestSize(PROC *proc)
{
	return (proc->vsize != 0);
}


static BOOL
TestRealTimer(PROC *proc)
{
	return (proc->it_real_value != 0);
}


static BOOL
TestActive(PROC *proc)
{
	return proc->isactive;
}


/*
 * Functions to show various columns.
 * This just means return a pointer to a string containing the desired
 * information.  The string should be of minimal length, since left or
 * right space padding is done elsewhere.  The string only has to be valid
 * until the next ShowXXX function is called, and so a common buffer can
 * be used for many routines.
 */
static char *
ShowPid(PROC *proc)
{
	sprintf(show_buffer, "%d", proc->pid);

	return show_buffer;
}


static char *
ShowState(PROC *proc)
{
	show_buffer[0] = proc->state;
	show_buffer[1] = '\0';

	return show_buffer;
}


static char *
ShowParentPid(PROC *proc)
{
	sprintf(show_buffer, "%d", proc->parent_pid);

	return show_buffer;
}


static char *
ShowSystemTime(PROC *proc)
{
	TicksToString(show_buffer, proc->stime);

	return show_buffer;
}


static char *
ShowUserTime(PROC *proc)
{
	TicksToString(show_buffer, proc->utime);

	return show_buffer;
}


static char *
ShowRunTime(PROC *proc)
{
	TicksToString(show_buffer, proc->stime + proc->utime);

	return show_buffer;
}


static char *
ShowChildRunTime(PROC *proc)
{
	TicksToString(show_buffer, proc->cstime + proc->cutime);

	return show_buffer;
}


static char *
ShowEip(PROC *proc)
{
	sprintf(show_buffer, "%08lx", proc->eip);

	return show_buffer;
}


static char *
ShowEsp(PROC *proc)
{
	sprintf(show_buffer, "%08lx", proc->esp);

	return show_buffer;
}


static char *
ShowWaitChannel(PROC *proc)
{
	if (proc->wchan == 0)
		return "";

	sprintf(show_buffer, "%08lx", proc->wchan);

	return show_buffer;
}


static char *
ShowProgram(PROC *proc)
{
	return proc->program;
}


static char *
ShowCommand(PROC *proc)
{
	return proc->command;
}


static char *
ShowEnvironment(PROC *proc)
{
	return proc->environment;
}


static char *
ShowProcessGroup(PROC *proc)
{
	sprintf(show_buffer, "%d", proc->process_group);

	return show_buffer;
}


static char *
ShowTtyProcessGroup(PROC *proc)
{
	sprintf(show_buffer, "%d", proc->tty_pgrp);

	return show_buffer;
}


static char *
ShowIdleTime(PROC *proc)
{
	char *	cp;
	long	idle;

	cp = show_buffer;

	if (proc->isancient)
		*cp++ = '+';

	idle = (current_time - proc->last_active_time) / 60;

	if (idle < 60)
		sprintf(cp, "%ld", idle);
	else
		sprintf(cp, "%ld:%02ld", idle / 60, idle % 60);

	return show_buffer;
}


static char *
ShowTtyName(PROC *proc)
{
	char *	name;

	name = FindDeviceName(proc->tty_dev);

	if (name != NULL)
		return name;

	sprintf(show_buffer, "%x", proc->tty_dev);

	return show_buffer;
}


static char *
ShowTtyDevice(PROC *proc)
{
	sprintf(show_buffer, "%x", proc->tty_dev);

	return show_buffer;
}


static char *
ShowUserName(PROC *proc)
{
	char *	name;

	name = FindUserName(proc->uid);

	if (name)
		return name;

	sprintf(show_buffer, "%d", proc->uid);

	return show_buffer;
}


static char *
ShowGroupName(PROC *proc)
{
	char *	name;

	name = FindGroupName(proc->gid);

	if (name)
		return name;

	sprintf(show_buffer, "%d", proc->gid);

	return show_buffer;
}


static char *
ShowUserId(PROC *proc)
{
	sprintf(show_buffer, "%d", proc->uid);

	return show_buffer;
}


static char *
ShowGroupId(PROC *proc)
{
	sprintf(show_buffer, "%d", proc->gid);

	return show_buffer;
}


static char *
ShowPercentCpu(PROC *proc)
{
	sprintf(show_buffer, "%d.%02d", proc->percent_cpu / 100,
		proc->percent_cpu % 100);

	return show_buffer;
}


static char *
ShowPercentMemory(PROC *proc)
{
	sprintf(show_buffer, "%d.%d", proc->percent_memory / 10,
		proc->percent_memory % 10);

	return show_buffer;
}


static char *
ShowRss(PROC *proc)
{
	sprintf(show_buffer, "%ld", proc->rss * 4);

	return show_buffer;
}


static char *
ShowStartTime(PROC *proc)
{
	int	proc_age_days;
	char *	cp;

	/*
	 * Get the formatted string for the process's start time:
	 *	"Wed Jun 30 21:49:08 1993\n"
	 */
	cp = ctime(&proc->start_time_clock);

	/*
	 * Show the start time as just the hour and minute, unless the
	 * process started more than one day ago, in which case also give
	 * the number of days it has been around.
	 */
	memcpy(show_buffer, cp + 11, 5);
	show_buffer[5] = '\0';

	proc_age_days = (current_time - proc->start_time_clock)
		/ (60 * 60 * 24);

	if (proc_age_days > 0)
		sprintf(show_buffer + 5, "-%dd", proc_age_days);

	return show_buffer;
}


static char *
ShowAge(PROC *proc)
{
	long	minutes;
	long	hours;
	long	days;

	minutes = (current_time - proc->start_time_clock) / 60;

	if (minutes < 0)
		minutes = 0;

	hours = minutes / 60;
	minutes %= 60;

	days = hours / 24;
	hours %= 24;

	if (days)
		sprintf(show_buffer, "%ldd%02ld:%02ld", days, hours, minutes);
	else if (hours)
		sprintf(show_buffer, "%ld:%02ld", hours, minutes);
	else
		sprintf(show_buffer, "%ld", minutes);

	return show_buffer;
}


static char *
ShowFlags(PROC *proc)
{
	sprintf(show_buffer, "%lx", proc->flags);

	return show_buffer;
}


static char *
ShowPageFaults(PROC *proc)
{
	sprintf(show_buffer, "%ld", proc->min_flt + proc->maj_flt);

	return show_buffer;
}


static char *
ShowMinorPageFaults(PROC *proc)
{
	sprintf(show_buffer, "%ld", proc->min_flt);

	return show_buffer;
}


static char *
ShowMajorPageFaults(PROC *proc)
{
	sprintf(show_buffer, "%ld", proc->maj_flt);

	return show_buffer;
}


static char *
ShowSignalCatch(PROC *proc)
{
	sprintf(show_buffer, "%08lx", proc->sigcatch);

	return show_buffer;
}


static char *
ShowSignalIgnore(PROC *proc)
{
	sprintf(show_buffer, "%08lx", proc->sigignore);

	return show_buffer;
}


static char *
ShowSignalBlock(PROC *proc)
{
	sprintf(show_buffer, "%08lx", proc->sigblock);

	return show_buffer;
}


static char *
ShowOpenFileCount(PROC *proc)
{
	if (proc->openfiles < 0)
		return "-";

	sprintf(show_buffer, "%d", proc->openfiles);

	return show_buffer;
}


static char *
ShowRunOrder(PROC *proc)
{
	sprintf(show_buffer, "%lu", proc->run_order);

	return show_buffer;
}


static char *
ShowCurrentDirectory(PROC *proc)
{
	if ((proc->cwd_dev == 0) && (proc->cwd_inode == 0))
		return "-";

	sprintf(show_buffer, "%04x:%u", proc->cwd_dev,
		(unsigned) proc->cwd_inode);

	return show_buffer;
}


static char *
ShowRootDirectory(PROC *proc)
{
	if ((proc->root_dev == 0) && (proc->root_inode == 0))
		return "-";

	sprintf(show_buffer, "%04x:%u", proc->root_dev,
		(unsigned) proc->root_inode);

	return show_buffer;
}


static char *
ShowExecutable(PROC *proc)
{
	if ((proc->exec_dev == 0) && (proc->exec_inode == 0))
		return "-";

	sprintf(show_buffer, "%04x:%u", proc->exec_dev,
		(unsigned) proc->exec_inode);

	return show_buffer;
}


static char *
ShowStdin(PROC *proc)
{
	return GetStdioString(proc->stdio_dev[0], proc->stdio_inode[0]);
}


static char *
ShowStdout(PROC *proc)
{
	return GetStdioString(proc->stdio_dev[1], proc->stdio_inode[1]);
}


static char *
ShowStderr(PROC *proc)
{
	return GetStdioString(proc->stdio_dev[2], proc->stdio_inode[2]);
}


/*
 * Local routine to convert device and inode into displayable strings.
 * This uses the show_buffer.
 */
static char *
GetStdioString(dev_t dev, ino_t inode)
{
	char *	name;

	if ((dev == 0) && (inode == 0))
		return "-";

	name = FindDeviceFromInode(dev, inode);

	if (name)
		return name;

	if (dev == NEWPIPEDEV)
		sprintf(show_buffer, "pipe:%u", (unsigned) inode);
	else if ((dev == OLDPIPEDEV) && (inode == 0))
		return "pipe";
	else
		sprintf(show_buffer, "%04x:%u", dev, (unsigned) inode);

	return show_buffer;
}


static char *
ShowStdio(PROC *proc)
{
	char *	devname;
	char *	ttyname;
	dev_t	dev;
	ino_t	inode;
	int	fd;

	strcpy(show_buffer, "---");

	/*
	 * Loop over the stdio file descriptors.
	 */
	for (fd = 0; fd <= 2; fd++) {
		dev = proc->stdio_dev[fd];
		inode = proc->stdio_inode[fd];

		/*
		 * If the file descriptor isn't open, leave the dash.
		 */
		if ((dev == 0) && (inode == 0))
			continue;

		/*
		 * If the file descriptor is a pipe, then say so.
		 */
		if ((dev == NEWPIPEDEV) || (dev == OLDPIPEDEV)) {
			show_buffer[fd] = 'P';

			continue;
		}

		/*
		 * If the file descriptor is the null device, then say so.
		 */
		if ((dev == null_dev) && (inode == null_inode)) {
			show_buffer[fd] = 'N';

			continue;
		}

		/*
		 * See if the file is a device, and if that device is the
		 * terminal the process is on.  If so, then indicate that.
		 */
		devname = FindDeviceFromInode(dev, inode);
		ttyname = FindDeviceName(proc->tty_dev);

		if (devname && ttyname && (strcmp(devname, ttyname) == 0)) {
			show_buffer[fd] = 'T';

			continue;
		}

		/*
		 * Don't know what it is, say it is just some file.
		 */
		show_buffer[fd] = 'F';
	}

	return show_buffer;
}


static char *
ShowSummary(PROC *proc)
{
	long	age;
	char *	cp;

	cp = show_buffer;
	strcpy(cp, "--------------");

	if ((proc->state == 'R') || (proc->state == 'Z') ||
		(proc->state == 'T') || (proc->state == 'D'))
	{
		cp[0] = proc->state;	/* copy state */
	} else if (proc->isactive)
		cp[0] = 'A';		/* active */
	else
		cp[0] = 'I';		/* or idle */

	if ((proc->rss == 0) && (proc->state != 'Z'))
		cp[1] = 'W';

	if (proc->priority < PZERO)
		cp[2] = 'N';
	else if (proc->priority > PZERO)
		cp[2] = 'H';

	if (proc->session_id == proc->pid)
		cp[3] = 'S';		/* session id leader */

	if (proc->process_group == proc->pid)
		cp[4] = 'P';		/* process group leader */

	if (proc->tty_dev > 0)
		cp[5] = 'T';		/* on a terminal */

	if (proc->tty_pgrp == proc->process_group)
		cp[6] = 'F';		/* foreground process */

	if (proc->parent_pid == 1)
		cp[7] = 'I';		/* owned by init */

	if (proc->sigignore & (1 << (SIGHUP - 1)))
		cp[8] = 'H';		/* ignores SIGHUP */
	else if (proc->sigcatch & (1 << (SIGHUP - 1)))
		cp[8] = 'h';		/* catches SIGHUP */

	if (proc->sigignore & (1 << (SIGTERM - 1)))
		cp[9] = 'T';		/* ignores SIGTERM */
	else if (proc->sigcatch & (1 << (SIGTERM - 1)))
		cp[9] = 't';		/* catches SIGTERM */

	if (proc->uid == my_uid)
		cp[10] = 'U';		/* has my user id */

	if (proc->gid == my_gid)
		cp[11] = 'G';		/* has my group id */

	if (proc->uid == 0)
		cp[12] = 'R';		/* root process */
	else if (proc->uid < BASE_USER_UID)
		cp[12] = 'S';		/* server process */

	age = current_time - proc->start_time_clock;

	if (age >= (60 * 60 * 24 * 7))
		cp[13] = 'W';		/* week old */
	else if (age >= (60 * 60 * 24))
		cp[13] = 'D';		/* day old */
	else if (age >= (60 * 60))
		cp[13] = 'H';		/* hour old */
	else if (age >= (60 * 10))
		cp[13] = 'T';		/* ten minutes old */
	else if (age >= (60 * 5))
		cp[13] = 'F';		/* five minutes old */
	else if (age >= 60)
		cp[13] = 'M';		/* minute old */
	else
		cp[13] = 'N';		/* new process */

	return show_buffer;
}


static char *
ShowPriority(PROC *proc)
{
	sprintf(show_buffer, "%ld", proc->priority);

	return show_buffer;
}


static char *
ShowSize(PROC *proc)
{
	sprintf(show_buffer, "%ld", (proc->vsize + 1023) / 1024);

	return show_buffer;
}


static char *
ShowRealTimer(PROC *proc)
{
	long	intpart;
	long	fracpart;

	intpart = proc->it_real_value / TICKS;
	fracpart = proc->it_real_value % TICKS;

	if (fracpart) {
		sprintf(show_buffer, "%ld.%02ld", intpart,
			((fracpart * 100) / TICKS));
	} else
		sprintf(show_buffer, "%ld", intpart);

	return show_buffer;
}


static char *
ShowActive(PROC *proc)
{
	return (proc->isactive ? "active" : "idle");
}


/*
 * Convert a time in ticks into hours, minutes, seconds and hundreths of
 * seconds in the form:
 *	hhh:mm:ss.hh
 * and store that into the supplied buffer.
 */
static void
TicksToString(char *buf, long ticks)
{
	int		hundreths;
	int		seconds;
	int		minutes;
	long		hours;

	hundreths = ticks % TICKS;
	ticks /= TICKS;

	seconds = ticks % 60;
	ticks /= 60;

	minutes = ticks % 60;
	ticks /= 60;

	hours = ticks;

	if (hours > 0) {
		sprintf(buf, "%ld:%02d:%02d.%02d", hours, minutes, seconds,
			hundreths);
	} else if (minutes > 0)
		sprintf(buf, "%d:%02d.%02d", minutes, seconds, hundreths);
	else
		sprintf(buf, "%d.%02d", seconds, hundreths);
}

/* END CODE */
