/* ESM - An extensible system monitoring tool.
 * Copyright (C) 1999, 2000 Peter Todd <retep@penguinpowered.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

/* main.c - Parses the command line options. Starts the main loop. */

#include <common/common.h>

#include "main.h"

#include "loop.h"
#include "msg.h"
#include "version.h"
#include "getopt.h"

int notifylevel = ESM_LOG_EMERG;

int maxmsgs = 2048;

int savepri = ESM_LOG_ERR;

int ignorepri = ESM_LOG_NOTICE;

int forcecheck_interval = 1800; /* 30 minutes */

int forcecheck_wait_time = 60; 

void printhelp();
void printversion();

int main(int argc,char *argv[]){
#ifdef HAVE_REGCOMP
  int regexpri = ESM_LOG_ERR;
#endif
  int c;
  
  /* Parse the command line options. */
  while (1)
    {
      int option_index = 0;
      static struct option long_options[] =
      {
	{"notifylevel", 1, 0, 0},
	{"maxmsgs", 1, 0, 0},
	{"savepri", 1, 0, 0},
	{"help", 0, 0, 0},
	{"version", 0, 0, 0},
	{"fcint", 1, 0, 0},
	{"regex", 1, 0, 0},
	{"regexpri", 1, 0, 0},
	{"fc-wait-time", 1, 0, 0},
	{"ignorepri", 1, 0, 0},
	{0, 0, 0, 0}
      };
      
      c = getopt_long (argc, argv, "n:m:s:i:x:p:hv",
		       long_options, &option_index);
      if (c == -1)
	break;
      
      switch (c)
	{
	case 0:
	  if (option_index == 0){
	    notifylevel = str2loglevel(optarg);
	  }
	  else if (option_index == 1){
	    maxmsgs = atoi(optarg);
	  }
	  else if (option_index == 2){
	    savepri = str2loglevel(optarg);
	  }
	  else if (option_index == 3){
	    printhelp();
	    exit(EXIT_SUCCESS);
	  }
	  else if (option_index == 4){
	    printversion();
	    exit(EXIT_SUCCESS);
	  }
	  else if (option_index == 5){
	    forcecheck_interval = strtime2sec(optarg);
	    if (forcecheck_interval < 1){
	      fprintf(stderr,"Invalid force check interval!\n");
	      exit(EXIT_FAILURE);
	    }
	  }
	  else if (option_index == 6){
#ifdef HAVE_REGCOMP
	    if (addregex(optarg,regexpri) != 0){
	      fprintf(stderr,"Invalid regular expression %s\n",optarg);
	      exit(EXIT_FAILURE);
	    }
#else
	    fprintf(stderr,"Regular expressions are not supported on your machine.\n");
	    exit(EXIT_FAILURE);
#endif
	  }
	  else if (option_index == 7){
#ifdef HAVE_REGCOMP
	    regexpri = str2loglevel(optarg);
	    if (regexpri <= 0){
	      fprintf(stderr,"Invalid priority level %s\n",optarg);
	      exit(EXIT_FAILURE);
	    }
#else
	    fprintf(stderr,"Regular expressions are not supported on your machine.\n");
	    exit(EXIT_FAILURE);
#endif
	  }
	  else if (option_index == 8){
	    forcecheck_wait_time = strtime2sec(optarg);
	    if (forcecheck_wait_time < 1){
	      fprintf(stderr,"Invalid force check wait time!\n");
	      exit(EXIT_FAILURE);
	    }
	  }
	  else if (option_index == 9){
	    ignorepri = str2loglevel(optarg);
	  }
	  break;
	  
	case 'n':
	  notifylevel = str2loglevel(optarg);
	  break;

	case 'm':
	  maxmsgs = atoi(optarg);
	  break;

	case 's':
	  savepri = str2loglevel(optarg);
	  break;

	case 'i':
	  ignorepri = str2loglevel(optarg);
	  break;

	case 'h':
	  printhelp();
   	  exit(EXIT_SUCCESS);
	  break;
       
	case 'v':
	  printversion();
	  exit(EXIT_SUCCESS);
	  break;

	case 'x':
#ifdef HAVE_REGCOMP
	  if (addregex(optarg,regexpri) != 0){
	    fprintf(stderr,"Invalid regular expression %s\n",optarg);
	    exit(EXIT_FAILURE);
	  }
#else
	  fprintf(stderr,"Regular expressions are not supported on your machine.\n");
	  exit(EXIT_FAILURE);
#endif
	  break;

	case 'p':
#ifdef HAVE_REGCOMP
	  regexpri = str2loglevel(optarg);
	  if (regexpri <= 0){
	    fprintf(stderr,"Invalid priority level %s\n",optarg);
	    exit(EXIT_FAILURE);
	  }
#else
	  fprintf(stderr,"Regular expressions are not supported on your machine.\n");
	  exit(EXIT_FAILURE);
#endif
	  break;
	  
	case '?':
                 break;

	default:
                 fprintf (stderr,"?? getopt returned character code 0%o ??\n", c);

	}
    }

  if (optind < argc)
    {
      fprintf (stderr,"non-option ARGV-elements: ");
      while (optind < argc)
	fprintf (stderr,"%s ", argv[optind++]);
      fprintf (stderr,"\n");
    }

  mainloop();

  return 0;
}

void printhelp(){
  printf("Usage: esmsyslog [OPTION]...\n");
  printf("ESMSysLog Daemon Monitor.\n");
  printf("\n");
  printf("  -n x, --notifylevel x    Set the notifylevel to x.\n");
  printf("  -m x, --maxmsgs x        Set maxmsgs to x.\n");
  printf("  -s x, --savepri x        Set the save priority to x.\n");
  printf("  -i x, --ignorepri x      Set the ignore priority to x.\n");
  printf("  --fcint x                Set the force check interval to x.\n");
  printf("  --fc-wait-time x         Set the force check wait time to x.\n");
#ifdef HAVE_REGCOMP
  printf("  -x x, --regex x          Match regular expression x.\n");
  printf("  -p x, --regexpri x       Set match priority to x.\n");
  printf("                           See docs for more information.\n");
#endif
  printf("  -h, --help               Displays this screen.\n");
  printf("  -v, --version            Displays version information.\n");
  printf("\n");
  printf("Report bugs to " MAINTAINER_EMAIL_ADDRESS "\n");
  exit(EXIT_SUCCESS);
}

void printversion(){
  printf("ESMSysLog %s\n",version);
  printf("Copyright (C) 1999, 2000 Peter Todd\n");
  printf("\n");
  printf("esmsyslog comes with NO WARRANTY,\n");
  printf("to the extent permitted by law.\n");
  printf("You may redistribute copies of esmd\n");
  printf("under the terms of the GNU General Public License.\n");
  printf("For more information about these matters,\n");
  printf("see the files named COPYING.\n");
  exit(EXIT_SUCCESS);
}

static char *id="@(#) $Id: main.c,v 3.6 2000/04/30 02:30:56 pete Exp $";
