/* ESM - An extensible system monitoring tool.
 * Copyright (C) 1999, 2000 Peter Todd <retep@penguinpowered.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

/* The main loop. */

#include "main.h"

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

#ifdef HAVE_FCNTL_H
#include <fcntl.h>
#endif

#include <sys/time.h>
#include <sys/types.h>
#include <signal.h>
 
#include "loop.h"
#include "version.h"
#include "msg.h"

void checkstdin();
void checkfifo(int i);
int forcecheck();

/* The array of fd's for the FIFOs. */
FILE *fifo_fd[ESM_LOG_EMERG];
int fifo_fd2[ESM_LOG_EMERG];

int have_set_alarm = 0;
int forcecheck_num = 0;

/* Incremented for each fifo that is successfully opened. */
int fifosopened = 0;

void mainloop(){
  RETSIGTYPE signalhandler(int sig);
  struct timeval timeout;
 
  /* The list of files we need to open. */
  char *fifo_name[] = {"emerg","alert","crit","err","warning","notice","info","debug"};

  /* The read array for the select call. */
  fd_set rset;

  int i;
  char *tmpstr = NULL;

  /* Setup the signalhandler to catch SIGALRM */
  signal(SIGALRM, signalhandler);
 
  /* Open the FIFOs we use to communicate with syslogd.
     Note that fopen will block if syslogd isn't running.
     So instead of using fopen we use open with the 
     O_NONBLOCK option. Then we open the stream with fdopen
  */
  for (i = 0; i < ESM_LOG_EMERG; i++){
    tmpstr = strdup(FIFODIR);
    tmpstr = realloc(tmpstr,strlen(tmpstr) + strlen(fifo_name[i]) + 1);
    tmpstr = strcat(tmpstr,fifo_name[i]);

    fifo_fd2[i] = open(tmpstr,O_NONBLOCK | O_RDONLY);

    if (fifo_fd2[i] < 0){
      fifo_fd[i] = NULL;
      free(tmpstr);
      continue;
    }
    fifosopened++;
    
    fifo_fd[i] = fdopen(fifo_fd2[i],"r");

    free(tmpstr);
  }

  /* Now to the loop. */

  /* The loop is really simple. Since we are a daemon monitor
     we just read from the FIFO's and stdin and do what we need
     to. We use select to block untill a FIFO or stdin needs
     attention.
  */
  while(1){

    /* Setup the select call so we are watching each FIFO and stdin */
    FD_ZERO (&rset);
    FD_SET(fileno(stdin),&rset);
    for (i = 0; i < ESM_LOG_EMERG; i++){
      if (fifo_fd2[i] >= 0)
	FD_SET (fifo_fd2[i], &rset);
    }

    timeout.tv_sec = forcecheck_interval;
    timeout.tv_usec = 0;
    select (FD_SETSIZE,&rset, NULL, NULL,&timeout);

    if (forcecheck_num > 0)
      forcecheck_num--;

    /* Check stdin for input. */
    if (FD_ISSET (fileno(stdin), &rset))
      checkstdin();

    /* Check the FIFOs for input. */
    for (i = 0; i < ESM_LOG_EMERG; i++){
      if (fifo_fd2[i] < 0)
	continue;

      if (FD_ISSET (fifo_fd2[i], &rset)){
	checkfifo(i);
      }
    }
    fflush(NULL);
  }
}

/* Checks stdin for input and does whatever it needs to. */
void checkstdin(){
  char tmpline[6];
  char *tmpline2 = NULL;
  struct syslogmsg *curmsg = NULL,*prevmsg = NULL;

  tmpline2 = fgets(tmpline,5,stdin);

  if (!tmpline2)
    return;

  if (!strncmp(tmpline2,"STAT",4)){
    /* Any problems while opening FIFO's? If so set maxurgency to
       emerg */
    if (fifosopened < ESM_LOG_EMERG)
      maxurgency = ESM_LOG_EMERG;
    printf("BEGIN\n");
    printf("URGENCY %d\n",maxurgency);
    printf("REPORT\n");

    /* Any problems while opening FIFO's? If so tell the user. */
    if (fifosopened < ESM_LOG_EMERG){
      printf("Error opening FIFOs. Please check syslogd configuration.\n");
    }

    /* Are there any msg in the msg list? */
    if (syslogmsglist){

      /* Print all of the msgs in the msg list. */
      curmsg = syslogmsglist;
      while (1){
	printf("%d: %s",curmsg->urgency,curmsg->msg);
	prevmsg = curmsg;
	if (curmsg->next)
	  curmsg = curmsg->next;
	else{
	  free(prevmsg->msg);
	  free(prevmsg);
	  break;
	}
	free(prevmsg->msg);
	free(prevmsg);
      }

      maxurgency = 0;
      nummsgs = 0;
      syslogmsglist = NULL;
    }
    else
      printf("No messages.\n");

    printf("DONE\n");

  }
}

/* Checks FIFO i */
void checkfifo(int i){
  char tmpline[1024];
  char *tmpline2 = NULL;
  struct syslogmsg *newmsg;
  int i2;

  tmpline2 = fgets(tmpline,1024,fifo_fd[i]);

  if (!tmpline2)
    return;

  /* Set maxurgency. */
  if (maxurgency < ESM_LOG_EMERG - i)
    maxurgency = ESM_LOG_EMERG - i;

  /* Add the message to the message list. */
  newmsg = malloc(sizeof(struct syslogmsg));
  newmsg->msg = strdup(tmpline2);

  newmsg->urgency = ESM_LOG_EMERG - i; /* The urgency is reversed. Have to fix that. */

#ifdef HAVE_REGCOMP
  /* Is msg matched by any regular expressions? If so set the urgency
     again. */
  for (i2 = 0; i2 < regextblnum; i2++){
    if (!regexec(regextbl + i2,newmsg->msg,0,NULL,0)){
      newmsg->urgency = regpritbl[i2];
      break;
    }
  }
#endif

  /* If we should notify esmd do so in forcecheck_wait_time. Since we may
   end up never running forcecheck if we get repeated notifies we only
   set the alarm if we havn't already done so. */
  if (newmsg->urgency >= notifylevel){
    if (!have_set_alarm){
      have_set_alarm = 1;
      alarm(forcecheck_wait_time);
    }
  }

  if (newmsg->urgency > ignorepri){
    addmsg(newmsg);
  }
  else{
    free(newmsg->msg);
    free(newmsg);
  }
}

/* Sends SIGUSR1 to esmd. Returns 0 on success, non-zero on fail. */
int forcecheck(){
  FILE *pidfile = NULL;
  int pid,i;

  /* Have we already been run? */
  if (forcecheck_num > 0)
    return 0;

  pidfile = fopen("/var/run/esmd.pid","r");
  
  if (!pidfile)
    return 1;
  
  i = fscanf(pidfile,"%d",&pid);

  fclose(pidfile);

  if (i != 1)
    return 1;

  if (pid <= 0)
    return 1;

  return kill(pid,SIGUSR1);
}

/* signal handler. Only used by SIGALRM */
RETSIGTYPE signalhandler(int sig){

  forcecheck();

  have_set_alarm = 0;

  return;

}

static char *id="@(#) $Id: loop.c,v 3.4 2000/04/30 02:32:49 pete Exp $";
