/* ESM - An extensible system monitoring tool.
 * Copyright (C) 1999, 2000 Peter Todd <retep@penguinpowered.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
                                                                                                                           
/* Sender plugin functions. */

#include "main.h"

#include <sys/types.h>
#include <sys/wait.h>
#include <signal.h>
#include <unistd.h>

#include "modulefunc.h"
#include "report.h"
#include "sender.h"
#include "options.h"
#include "runcmd.h"
#include "version.h"
#include "logfunc.h"

SENDER *senderlist;
int numsenders;

int did_sender_succeed(int i);

/* Sends a system report. */
void sendsysreport(){
  int i,send_status;
  FILE *stdin_fd;

  for (i = 0; i < numsenders; i++){

    if (!daemon_mode)
      printf("Trying sender %s\n",senderlist[i].title);

    esmd_mktempfifo(&senderlist[i].stdin_file);
    fclose(esmd_mktempfile(&senderlist[i].stdout_file));

    senderlist[i].pid = re_system(senderlist[i].cmd,senderlist[i].stdin_file,senderlist[i].stdout_file);

    /* Give the sender the report via standard input. */
    stdin_fd = fopen(senderlist[i].stdin_file,"a");                                                                            addfile(stdin_fd,sysreport_file);
    fclose(stdin_fd);

    /* Make sure the sender exits. */
    wait_module(senderlist[i].pid,sendertimeout);

    send_status = did_sender_succeed(i);

    /* Did the sender fail? */
    if (send_status == -1)
      ; /* Do nothing. */
    else if (send_status == 1){
      snprintf(errmsg,ERRMSG_MAXLEN,"Clear failure of sender %s\n",senderlist[i].title);
      logmsg(errmsg,ESM_LOG_ERR);
    }
    else{
      snprintf(errmsg,ERRMSG_MAXLEN,"Unclear failure of sender %s - Sender did not return fail code.\n",senderlist[i].cmd);
      logmsg(errmsg,ESM_LOG_ERR);
    }

    if (!nodelete){
      /* Delete the stdin and stdout files. */
      esmd_rmtempfile(senderlist[i].stdin_file);
      esmd_rmtempfile(senderlist[i].stdout_file);
    }

    if (send_status == -1){
      /* Have we ran all of the senders in the priority group? */
      if (i == (numsenders - 1))
	break;
      else if (senderlist[i].pri != senderlist[i + 1].pri)
	break;
    }
  }

  if (!nodelete){
    /* Delete the report file. */
    esmd_rmtempfile(sysreport_file);
  }
}

/* Returns true if a sender succeeded. 1 if there is a clear failure, 0 if nothing was returned in
 standard output.
*/
int did_sender_succeed(int i){
  FILE *stdout_fd;
  char tmpbuff[1024];
  char *tmpbuff2;

  char strsucceed[] = "SUCCEED\n";
  char strfail[] = "FAIL\n";

  /* Open the standard output file. */
  stdout_fd = fopen(senderlist[i].stdout_file,"r");

  /* Read the first bit of the file and check for
     the string "SUCCEED\n" 
  */
  tmpbuff2 = fgets(tmpbuff,1024,stdout_fd);

  if (!daemon_mode && tmpbuff2)
    printf("Sender returned %s\n",tmpbuff);

  fclose(stdout_fd);

  if (strcmp(strsucceed,tmpbuff) == 0)
    return -1;
  else if (strcmp(strfail,tmpbuff) == 0)
    return 1;

  return 0;
}

/* Adds a sender of priority pri. Note that pri is currently ignored. */
void addsender(SENDER newsend,int pri){
  int i,newi;

  numsenders++;
  senderlist = realloc(senderlist,sizeof(SENDER) * numsenders);

  /* Only search for a spot if we have to. */
  if (numsenders > 2){

    /* Go through the sender list from the end to the start. If the priority of the sender is greater then
       pri move the sender +1 Otherwise take the above (+1) the spot we are looking at. Note that we
       start at the *second* last position. This way if the first position we check is less then out pri
       we can take the above (+1) position. 

       We also stop at -1 So when we find ourselves at the start we can take the first entry.
    */
    for (i = numsenders - 2; i >= -1; i--){
      if (i < 0){
	newi = 0;
	break;
      }
      else if (senderlist[i].pri > pri)
	senderlist[i + 1] = senderlist[i];
      else{
	newi = i + 1; /* Set newi to the position above this one. */
	break;
      }
    }
  }
  else if (numsenders == 2){
    if (pri > senderlist[0].pri){
      senderlist[1] = newsend;
      newi = 1;
    }
    else{
      senderlist[1] = senderlist[0];
      newi = 0;
    }
  }
  else
    newi = 0;

  /* Since the strings are just pointers to a string we have make a
     copy of the strings. Otherwise they might be overwritten later.
  */
  senderlist[newi].title = malloc(strlen(newsend.title) + 1);
  senderlist[newi].title[0] = '\000';
  senderlist[newi].title = strcpy(senderlist[newi].title,newsend.title);

  senderlist[newi].cmd = malloc(strlen(newsend.cmd) + 1);
  senderlist[newi].cmd[0] = '\000';
  senderlist[newi].cmd = strcpy(senderlist[newi].cmd,newsend.cmd);

  senderlist[newi].pri = pri;
}

static char *id="@(#) $Id: sender.c,v 5.3 2000/04/27 01:39:43 pete Exp $";

