/* ESM - An extensible system monitoring tool.
 * Copyright (C) 1999, 2000 Peter Todd <retep@penguinpowered.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

/* Module functions. */

#include "main.h"

#include <sys/types.h>
#include <sys/wait.h>
#include <signal.h>
#include <unistd.h>

#include "modulefunc.h"
#include "options.h"

/* A linked list of all the temp files. */
struct tmpfile *tmpfilelist = NULL;

/* Signals to try on a non-responding module. */
const int killsignal[] = {SIGINT,SIGHUP,SIGKILL};

/* Kills a module. */
void kill_module(pid_t mpid){
  int i,r,status;
  pid_t tmppid;
  
  if (!daemon_mode)
    printf("Killing module pid %d\n",mpid);

  for (i = 0; i < 3; i++){

    if (!daemon_mode)
      printf("Sending signal %d to module pid %d\n",killsignal[i],mpid);

    /* Send killsignal[i] to mpid. */
    r = kill(mpid,killsignal[i]);

    /* Wait for mpid to close. It should exit, but if not we can try another signal on it. */
    tmppid = waitpid(mpid,&status,WNOHANG);
    
    /* Did mpid exit? */
    if ((WIFEXITED(status) | WIFSIGNALED(status)) & (tmppid == mpid))
      break;

    /* If the child didn't exit wait a few seconds. */
    sleep(module_kill_signal_interval);
    
    /* Disabled. Why would a module ever be able to ignore kill?
       Except while debugging but that's not much of a real reason. */

    /* Keep repeating the kill signal untill the module is dead. */
    /*if (i == 2)
      i = 1;*/
  }
}

/* Waits for a module to exit. Kills the module after timeout seconds. 
   Returns -1 if the module exited normally. 0 if it had to be killed.
*/
int wait_module(pid_t pid,int timeout){
  int status,i;
  pid_t rpid;

  for (i = 0; i < timeout; i++){
    sleep(1);

    /* Is the module still alive? */
    rpid = waitpid(pid,&status,WNOHANG);
    if (rpid == pid)
      return -1;
  }

  /* Looks like the module will have to be killed. Call kill_module(pid) */
  kill_module(pid);

  return 0;
}

void esmd_mktempfifo(char **name){
  int r;
  char *n;
  struct tmpfile *tmptmpfile;

  /* name must never be NULL as there has to be a call to
     esmd_rmtmpfile eventually. */
  if (!name){
    fprintf(stderr,"Internal error! name is NULL in esmd_mktempfile");
    abort();
  }

  r = mktmpfifo(&n);

  if (r){
    fprintf(stderr,"Couldn't create temp file, exiting.");
    exit(EXIT_FAILURE);
  }

  *name = n;

  /* Adds the temp file to the temp file list. */
  tmptmpfile = malloc(sizeof(struct tmpfile));
  tmptmpfile->name = n;
  tmptmpfile->prev = NULL;

  if (tmpfilelist){
    tmptmpfile->next = tmpfilelist;
    tmpfilelist = tmptmpfile;
    tmpfilelist->next->prev = tmpfilelist;
  }
  else{
    tmptmpfile->next = NULL;
    tmpfilelist = tmptmpfile;
  }
}

FILE *esmd_mktempfile(char **name){
  FILE *r;
  char *n;
  struct tmpfile *tmptmpfile;

  /* name must never be NULL as there has to be a call to
     esmd_rmtmpfile eventually. */
  if (!name){
    fprintf(stderr,"Internal error! name is NULL in esmd_mktempfile");
    abort();
  }

  r = mktmpfile(&n);

  if (!r){
    fprintf(stderr,"Couldn't create temp file, exiting. %s",strerror(errno));
    exit(EXIT_FAILURE);
  }

  *name = n;

  /* Adds the temp file to the temp file list. */
  tmptmpfile = malloc(sizeof(struct tmpfile));
  tmptmpfile->name = n;
  tmptmpfile->prev = NULL;

  if (tmpfilelist){
    tmptmpfile->next = tmpfilelist;
    tmpfilelist = tmptmpfile;
    tmpfilelist->next->prev = tmpfilelist;
  }
  else{
    tmptmpfile->next = NULL;
    tmpfilelist = tmptmpfile;
  }

  return r;
}

/* Removes all temp files. */
void esmd_rmalltmpfiles(void){
  struct tmpfile *curfile;

  curfile = tmpfilelist;
  while (1){
    if (!curfile)
      return;

    remove(curfile->name);

    if (curfile->next)
      curfile = curfile->next;
    else
      return;
  }
}

/* Removes a temp file. */
void esmd_rmtempfile(char *file){
  struct tmpfile *rmtmp,*curfile;

  /* Find which tmpfilelist tmpfile is file. */
  curfile = tmpfilelist;
  while (1){
    if (!strcmp(curfile->name,file)){
      rmtmp = curfile;
      break;
    }
    if (curfile->next)
      curfile = curfile->next;
    else
      return;
  }

  remove(rmtmp->name);

  if (rmtmp->next){
    if (rmtmp->prev)
      rmtmp->next->prev = rmtmp->prev;
    else
      rmtmp->next->prev = NULL;
  }
  if (rmtmp->prev){
    if (rmtmp->next)
      rmtmp->prev->next = rmtmp->next;
    else
      rmtmp->prev->next = NULL;
  }
  
  if (rmtmp == tmpfilelist){
    if (rmtmp->next)
      tmpfilelist = rmtmp->next;
    else if (rmtmp->prev)
      tmpfilelist = rmtmp->prev;
    else
      tmpfilelist = NULL;
  }

  free(rmtmp->name);
  free(rmtmp);
}
 
static char *id="@(#) $Id: modulefunc.c,v 5.3 2000/04/27 01:39:41 pete Exp $";
