/* ESM - An extensible system monitoring tool.
 * Copyright (C) 1999, 2000 Peter Todd <retep@penguinpowered.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */                                              

/* main.c - Parses the command line options. Starts the main loop. */

#include "main.h"

#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>                                                                                                

#include "loop.h"
#include "logfunc.h"
#include "options.h"
#include "configload.h"
#include "getopt.h"
#include "version.h"

void printhelp();
void printversion();

int main(int argc,char *argv[]){
  pid_t tmppid;
  int c;
  FILE *pidfile;

  /* Copy defaultconfigdir to configdir. */
  configdir = malloc(strlen(defaultconfigdir) + 1);
  configdir[0] = '\000';
  configdir = strcpy(configdir,defaultconfigdir);
  configdir = addslash(configdir);

  /* Parse the command line options. */
  while (1)
    {
      int option_index = 0;
      static struct option long_options[] =
      {
	{"file", 1, 0, 0},
	{"forground", 0, 0, 0},
	{"help", 0, 0, 0},
	{"version", 0, 0, 0},
	{0, 0, 0, 0}
      };
      
      c = getopt_long (argc, argv, "f:hv",
		       long_options, &option_index);
      if (c == -1)
	break;

      switch (c)
	{
	case 0:
	  if (option_index == 0){
	    free(configdir);
	    configdir = malloc(strlen(optarg));
	    strcpy(configdir,optarg);
	  }
	  else if (option_index == 1){
	    daemon_mode = 0;
	  }
	  else if (option_index == 2){
	    printhelp();
	    exit(EXIT_SUCCESS);
	  }
	  else if (option_index == 3){
	    printversion();
	    exit(EXIT_SUCCESS);
	  }
	  break;
	  
	case 'f':
	  free(configdir);
	  configdir = malloc(strlen(optarg));
	  strcpy(configdir,optarg);
	  break;
	  
	case 'h':
	  printhelp();
   	  exit(EXIT_SUCCESS);
	  break;
       
	case 'v':
	  printversion();
	  exit(EXIT_SUCCESS);
	  break;

	case '?':
                 break;

	default:
                 fprintf (stderr,"?? getopt returned character code 0%o ??\n", c);

	}
    }

  if (optind < argc)
    {
      fprintf (stderr,"non-option ARGV-elements: ");
      while (optind < argc)
	fprintf (stderr,"%s ", argv[optind++]);
      fprintf (stderr,"\n");
    }

  /* Fork and become a daemon. */
  if (daemon_mode){
    tmppid = fork();
    if (tmppid > 0){
      exit(EXIT_SUCCESS);
    }
    else if (tmppid < 0){
      #if HAVE_STRERROR
      fprintf(stderr,"Fork failed. - %s\n",strerror(errno));
      #else
      fprintf(stderr,"Fork failed.\n");
      #endif
      exit(EXIT_FAILURE);
    }

    /* become group and session leader */
    setsid();

    /* fork again so the parent, group leader, can exit */
    if (fork())
      exit(EXIT_SUCCESS);
    
  }   

  chdir("/");
  /* Set initial permissions to no permissions for anyone but
     ourselves. */
  umask(S_IRWXG | S_IRWXO);

  /* Write our pid to /var/run/esmd.pid */
  pidfile = fopen("/var/run/esmd.pid","w");
  if (pidfile){
    fprintf(pidfile,"%d\n",(int)getpid());
    fclose(pidfile);
  }
  else{
    snprintf(errmsg,ERRMSG_MAXLEN,
	     "Could not open pid file, some plugins may not work.");
    logmsg(errmsg,ESM_LOG_ERR);
  }

  mainloop();

  return 0;
}

void printhelp(){
 printf("Usage: esmd [OPTION]...\n");
 printf("Extensible System Monitor (ESM) daemon.\n");
 printf("\n");
 printf("  -f DIR, --file DIR       Uses DIR as the configuration directory.\n");
 printf("  --forground              Don't run as a daemon.\n");
 printf("  -h, --help               Displays this screen.\n");
 printf("  -v, --version            Displays version information.\n");
 printf("\n");
 printf("Report bugs to " MAINTAINER_EMAIL_ADDRESS "\n");
}

void printversion(){
  printf("Extensible System Monitor Daemon (ESMD) %s\n",version);
  printf("Copyright (C) 1999, 2000 Peter Todd\n");
  printf("\n");
  printf("esmd comes with NO WARRANTY,\n");
  printf("to the extent permitted by law.\n");
  printf("You may redistribute copies of esmd\n");
  printf("under the terms of the GNU General Public License.\n");
  printf("For more information about these matters,\n");
  printf("see the files named COPYING.\n");
}

static char *id="@(#) $Id: main.c,v 5.5 2000/04/27 23:12:35 pete Exp $";
