/* ESM - An extensible system monitoring tool.
 * Copyright (C) 1999, 2000 Peter Todd <retep@penguinpowered.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

/* Daemon Monitor functions. */

#include "main.h"

#include <sys/types.h>
#include <sys/wait.h>
#include <sys/stat.h>
#include <signal.h>
#include <unistd.h>
#include <fcntl.h>
 
#include "monitor.h"
#include "dmonitor.h"
#include "modulefunc.h"
#include "options.h"
#include "runcmd.h"
#include "version.h"
#include "logfunc.h"

MONITOR *dmonitorlist;
int numdmonitors = 0;

/* Gets the report from a dmonitor and writes it to the report file. - Only used within dmonitor.c */
int getreport(int i);

/* Starts a dmonitor. - Only used within dmonitor.c */
void startdmonitor(int i);

/* Tells a dmonitor to make a report. - Only used within dmonitor.c */
void makereport(int i);

/* Runs all of the dmonitors. */
void rundmonitors(){
  int i,status;
  pid_t pid;

  for (i = 0; i < numdmonitors; i++){

    /* Has the dmonitor been started yet? */
    if (!dmonitorlist[i].pid){
      /* Start 'er up! */

      startdmonitor(i);

      continue;
    }

    /* Is the dmonitor still alive? */
    pid = waitpid(dmonitorlist[i].pid,&status,WNOHANG);
    if (pid){

      snprintf(errmsg,ERRMSG_MAXLEN,"dmonitor %s died.",dmonitorlist[i].title);
      logmsg(errmsg,ESM_LOG_ERR);

      /* Delete the temp files used by the dmonitor. */
      esmd_rmtempfile(dmonitorlist[i].stdin_file);
      esmd_rmtempfile(dmonitorlist[i].stdout_file);
      esmd_rmtempfile(dmonitorlist[i].report_file);

      /* Start the dmonitor up again. */
      startdmonitor(i);
    }

  }

  /* Tell each dmonitor to make a report. */
  for (i = 0; i < numdmonitors; i++){
    /* Get a new report file name. Remember that createsysreport() deletes the report file. */
    fclose(esmd_mktempfile(&(dmonitorlist[i].report_file)));

    /* And make the report. */
    makereport(i);
  }

  /* Now go through the dmonitors and get the results from each one. Note that getreport will timeout for us. */
  for (i = 0; i < numdmonitors; i++){

    /* Copy the report. */
    getreport(i);
  }
}

/* Kills all dmonitors. */
void killdmonitors(){
  int i;
  if (!daemon_mode)
    printf("Killing dmonitors...\n");
  for (i = 0; i < numdmonitors; i++){
    kill_module(dmonitorlist[i].pid);
  }
}

/* Gets the report from a dmonitor and writes it to the report file. i is the index of the dmonitor. 
   If the whole report was recived returns 1 Otherwise returns 0
*/
int getreport(int i){
  FILE *report_fd;
  FILE *stdout_fd;

  char tmpline[1024];
  char *tmpline2;

  int tmp,oldflags,reps = 0;

  /* Open stdout_file and report_file */
  report_fd = fopen(dmonitorlist[i].report_file,"w");
  stdout_fd = fopen(dmonitorlist[i].stdout_file,"r");

  /* Not needed, mktmpfile does that for us */

  /* Change the permissions of report_file to 600 */
  /*  chmod(dmonitorlist[i].report_file,S_IREAD | S_IWRITE); */

  /* Set stdout to non-blocking. If we don't do that a dmonitor that doesn't print a DONE will cause a crash. */
  tmp = fileno(stdout_fd);
  oldflags = fcntl (tmp, F_GETFL, 0);
  oldflags |= O_NONBLOCK;
  /* Store modified flag word in the descriptor. */
  fcntl (tmp, F_SETFL, oldflags);

  /* Read each line of stdout and write it to report_file. */
  while (1){

    tmpline2 = fgets(tmpline,1024,stdout_fd);

    if (!tmpline2){
      /* Timeout after monitortimeout seconds. */
      reps++;
      if (reps > monitortimeout){
	/* Looks like a dmonitor is dead... Kill it and move on. */
	kill_module(dmonitorlist[i].pid);
	dmonitorlist[i].pid = 0;

	snprintf(errmsg,ERRMSG_MAXLEN,"dmonitor %s timed out.",dmonitorlist[i].title);
	logmsg(errmsg,ESM_LOG_ERR);

	break;
      }
      sleep(1);
      continue;
    }

    fprintf(report_fd,"%s",tmpline);

    if (!strncmp(tmpline,"DONE",4)){
      fclose(stdout_fd);
      fclose(report_fd);

      return 1;
    }
  }

  fclose(stdout_fd);
  fclose(report_fd);

  return 0;
}

/* Starts a dmonitor. */
void startdmonitor(int i){
  if (!daemon_mode)
    printf("Starting dmonitor %s\n",dmonitorlist[i].title);

  esmd_mktempfifo(&(dmonitorlist[i].stdin_file));
  esmd_mktempfifo(&(dmonitorlist[i].stdout_file));

  dmonitorlist[i].pid = re_system2(dmonitorlist[i].cmd,dmonitorlist[i].stdin_file,dmonitorlist[i].stdout_file);
}

/* Tells a dmonitor to make a report. i is the index of the dmonitor. */
void makereport(int i){
  FILE *stdin_fd;

  if (!daemon_mode)
    printf("Getting report from dmonitor %s\n",dmonitorlist[i].title);

  /* Open stdin and fprintf STAT */
  stdin_fd = fopen(dmonitorlist[i].stdin_file,"w");

  fprintf(stdin_fd,"STAT\n");

  fclose(stdin_fd);
}

static char *id="@(#) $Id: dmonitor.c,v 5.4 2000/05/02 00:51:53 pete Exp $";
