/* ESM - An extensible system monitoring tool.
 * Copyright (C) 1999, 2000 Peter Todd <retep@penguinpowered.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
                                                                                             
/* Config file loading functions. */

#include "main.h"

#include <stdlib.h>
#include <sys/types.h>

#if HAVE_DIRENT_H
# include <dirent.h>
# define NAMLEN(dirent) strlen((dirent)->d_name)
#else
# define dirent direct
# define NAMLEN(dirent) (dirent)->d_namlen
# if HAVE_SYS_NDIR_H
#  include <sys/ndir.h>
# endif
# if HAVE_SYS_DIR_H
#  include <sys/dir.h>
# endif
# if HAVE_NDIR_H
#  include <ndir.h>
# endif
#endif

/* Needed by lineisempty function. */ 
#include <ctype.h>

#include "configload.h"
#include "monitor.h"
#include "sender.h"
#include "options.h"

void loadesmconfig(char *dir,char *file);
void loadmoduleconfig(char *dir,char *file);

/* Have we loaded the config dir? */
int haveconfig = 0;

/* Adds a slash to the end of a char if needed. */
char *addslash(char *slash){
  int slashlen;

  /* If needed add a / to the end of dir. */
  slashlen = strlen(slash);
  if (slash[slashlen - 1] != '/'){
    slash = realloc(slash,slashlen + 2);
    slash[slashlen] = '/';
    slash[slashlen + 1] = '\000';
  }

  return slash;
}

/* Loads the config directory. */
int loadconfig(){
  DIR *dir_fd = NULL;
  struct dirent *tmpdirent = NULL;
  struct dirent curdirent;

  /* If needed add a / to the end of dir. */
  configdir = addslash(configdir);

  /* Open the configdir directory. */
  dir_fd = opendir(configdir);


  /* Error? */
  if (!dir_fd){
    #if HAVE_STRERROR
    fprintf(stderr,"Couldn't open configdir %s - %s\n",configdir,strerror(errno));
    #else
    fprintf(stderr,"Couldn't open configdir %s\n",configdir);
    #endif
    exit(EXIT_FAILURE);
  }

  /* Read each entry in the dir. */
  while(1){

    tmpdirent = NULL;
    tmpdirent = readdir(dir_fd);

    if (!tmpdirent)
      break;

    /* The man page for readdir says that tmpdirent might get it overwritten. Better play it
       safe and copy it to another var. */
    curdirent = *tmpdirent;

    /* Figure out what to do with the file. */
    if (curdirent.d_name[0] == '.')
      continue;
    /* Ignore save files made with emacs. */
    else if (curdirent.d_name[strlen(curdirent.d_name) - 1] == '~')
      continue;
    /* Ignore backup files from emacs. */
    else if ((curdirent.d_name[strlen(curdirent.d_name) - 1] == '#') && (curdirent.d_name[0] == '#'))
      continue;
    else if (!strcmp(curdirent.d_name,"esm.conf"))
      loadesmconfig(configdir,curdirent.d_name);
    else 
      loadmoduleconfig(configdir,curdirent.d_name);
  }

  closedir(dir_fd);


  /* Check if any senders were defined. */
  if (numsenders < 1){
    fprintf(stderr,"No senders defined. Exiting \n");
    exit(EXIT_FAILURE);
  }

  haveconfig = 1;

  return 0;
}

/* A macro that checks if a line is a comment. */
#define iscomment(line) (line[0] == '#')

/* A macro that gets a number from a line in the format x # */
#define getnum(line,tag) atoi(&line[strlen(tag)])

/* A macro that checks if a tag is in a line. Replaced by below function.
#define hastag(line,tag) !strncmp(line,tag,strlen(tag))
*/

/* A function that checks if a tag is in a line. This function used to be a macro. */
int hastag(char *line, char *tag){
  char *tmp;
  int i = 0;
  
  i = sscanf(line,"%as",&tmp);

  if (i == 1){
    if (!strcmp(tmp,tag)){
      free(tmp);
      return 1;
    }
  }
  return 0;
}

/* A function that gets a time from a line. The time can be in minutes, hours, or days */
int gettime(char *line,char *tag){
  char *tmp;
  int i = 0;

  i = sscanf(line,"%*s %as",&tmp);
 
  if (i != 1)
    return 0;

  /* Got the tag. Now figure how if we are dealing with minutes, hours or days. */
  if (strchr(tmp,'m') != NULL)
    i = atoi(tmp) * 60;
  else if (strchr(tmp,'h') != NULL)
    i = atoi(tmp) * 3600;
  else if (strchr(tmp,'d') != NULL)
    i = atoi(tmp) * 86400;
  else
    i = atoi(tmp);

  return i;
}

/* Returns true if a line is empty. */
int lineisempty(char *s){
  int i;
  for (i = 0; i < strlen(s); i++){
    if (!isspace(s[i]))
      return 0;
  }
  return 1;
}

/* Loads a esm.conf file. */
void loadesmconfig(char *dir,char *file){
  FILE *file_fd;
  char *filename;
  char tmpline[1024];
  char *tmpline2;
  int tmp;
  
  /* The different options. */
  char strupdateinterval[] = "updateinterval";
  char strmonitortimeout[] = "monitortimeout";
  char strsendertimeout[] = "sendertimeout";
  char strmodule_kill_signal_interval[] = "module_kill_signal_interval";

  filename = malloc(strlen(dir) + strlen(file) + 1);
  filename[0] = '\000';
  filename = strcat(filename,dir);
  filename = strcat(filename,file);
  file_fd = fopen(filename,"r");

  /* Read each line and do whatever is needed. */
  while (1){

    tmpline2 = fgets(tmpline,1024,file_fd);

    /* Remove the \n at the end of tmpline. */
    if (tmpline[strlen(tmpline) - 1] == '\n')
      tmpline[strlen(tmpline) - 1] = '\000';

    if (!tmpline2)
      break;
    
    /* Is the line a comment? */
    if (iscomment(tmpline))
      continue;

    /* Parse the line. */
    if (hastag(tmpline,strupdateinterval)){
      tmp = gettime(tmpline,strupdateinterval);
      if (tmp >= 0)
	updateinterval = tmp;
      else{
	fprintf(stderr,"Invalid value of updateinterval in esm.conf!\n");
	exit(EXIT_FAILURE);
      }
    }
    else if (hastag(tmpline,strmonitortimeout)){
      tmp = gettime(tmpline,strmonitortimeout);
      if (tmp > 0)
        monitortimeout = tmp;
      else{
        fprintf(stderr,"Invalid value of monitortimeout in esm.conf!\n");
	exit(EXIT_FAILURE);
      }
    }
    else if (hastag(tmpline,strsendertimeout)){
      tmp = gettime(tmpline,strsendertimeout);
      if (tmp > 0)
        sendertimeout = tmp;
      else{
        fprintf(stderr,"Invalid value of sendertimeout in esm.conf!\n");
	exit(EXIT_FAILURE);
      }
    }
    else if (hastag(tmpline,strmodule_kill_signal_interval)){
      tmp = gettime(tmpline,strmodule_kill_signal_interval);
      if (tmp >= 0)
        module_kill_signal_interval = tmp;
      else{
        fprintf(stderr,"Invalid value of module_kill_signal_interval in esm.conf!\n");
	exit(EXIT_FAILURE);
      }
    }
    else if (lineisempty(tmpline)){
      ;
    }
    else{
      fprintf(stderr,"Unrecognized line %s in %s%s\n",tmpline,dir,file);
      exit(EXIT_FAILURE);
    }

  }

  fclose(file_fd);
}

/* A macro that gets a string from a line in the format x <string> The + 1 is needed to account for
 the space between the tag and the value. */
#define getstr(line,tag) (line + strlen(tag) + 1)

/* Loads a moduleconfig file. */
void loadmoduleconfig(char *dir,char *file){
  FILE *file_fd;
  char *filename;
  char tmpline[1024];
  char *tmpline2;

  /* The different options. */
  char strtitle[] = "title";
  char strcmd[] = "cmd";
  char strmonitor[] = "monitor";
  char strdmonitor[] = "dmonitor";
  char strsender[] = "sender";

  char *title = NULL;
  char *cmd = NULL;
  int type = 0;
  int pri;

  int flags = 0;

  #define TYPE_MONITOR 1
  #define TYPE_SENDER  2

  filename = malloc(strlen(dir) + strlen(file) + 1);
  filename[0] = '\000';
  filename = strcat(filename,dir);
  filename = strcat(filename,file);
  file_fd = fopen(filename,"r");

  /* Read each line and do whatever is needed. */
  while (1){

    tmpline2 = fgets(tmpline,1024,file_fd);

    /* Remove the \n at the end of tmpline. */
    if (tmpline[strlen(tmpline) - 1] == '\n')
      tmpline[strlen(tmpline) - 1] = '\000';

    if (!tmpline2)
      break;

    /* Is the line a comment? */
    if (iscomment(tmpline))
      continue;

    /* Parse the line. */
    if (hastag(tmpline,strtitle)){
      title = malloc(strlen(tmpline) + 1);
      title = strcpy(title,getstr(tmpline,strtitle));

    }
    else if (hastag(tmpline,strcmd)){
      cmd = malloc(strlen(tmpline) + 1);
      cmd = strcpy(cmd,getstr(tmpline,strcmd));

    }
    else if (hastag(tmpline,strmonitor)){
      type = TYPE_MONITOR;
    }
    else if (hastag(tmpline,strdmonitor)){
      type = TYPE_MONITOR;
      flags |= MONITOR_FLAG_DAEMON;
    }
    else if (hastag(tmpline,strsender)){
      type = TYPE_SENDER;
      pri = getnum(tmpline,strsender);
    }
    else if (lineisempty(tmpline)){
      ;
    }
    else{
      fprintf(stderr,"Unrecognized line %s in %s%s\n",tmpline,dir,file);
      exit(EXIT_FAILURE);
    }
  }

  fclose(file_fd);


  /* Save the new module in the right modulelist. */
  if ((type == TYPE_MONITOR) && title && cmd){
    MONITOR tmpmon;

    tmpmon.title = title;
    tmpmon.cmd = cmd;
    
    /* Setup the flags. */
    tmpmon.flags = flags;

    addmonitor(tmpmon);
  }
  else if ((type == TYPE_SENDER) && title && cmd){

    SENDER tmpsend;

    tmpsend.title = title;
    tmpsend.cmd = cmd;
    
    addsender(tmpsend,pri);
  }
  else{
    fprintf(stderr,"No sender or monitor ");

    if (!title)
      fprintf(stderr,"or title ");
    if (!cmd)
      fprintf(stderr,"or cmd ");

    fprintf(stderr,"tag in config file %s%s\n",dir,file);
    exit(EXIT_FAILURE);
  }
}

static char *id="@(#) $Id: configload.c,v 5.4 2000/05/01 19:41:08 pete Exp $";
