/*************************************************************************
** interpcom-3.1     (command interpreter)                               **
** interp.h      Command interpreter                                     **
**                                                                       **
** Copyright (C) 2003  Jean-Marc Drezet                                  **
**                                                                       **
**  This library is free software; you can redistribute it and/or        **
**  modify it under the terms of the GNU Library General Public          **
**  License as published by the Free Software Foundation; either         **
**  version 2 of the License, or (at your option) any later version.     **
**                                                                       **
**  This library is distributed in the hope that it will be useful,      **
**  but WITHOUT ANY WARRANTY; without even the implied warranty of       **
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU    **
**  Library General Public License for more details.                     **
**                                                                       **
**  You should have received a copy of the GNU Library General Public    **
**  License along with this library; if not, write to the Free           **
**  Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.   **
**                                                                       **
** Please mail any bug reports/fixes/enhancements to me at:              **
**      drezet@math.jussieu.fr                                           **
** or                                                                    **
**      Jean-Marc Drezet                                                 **
**      Institut de Mathematiques                                        **
**      UMR 7586 du CNRS                                                 **
**      173, rue du Chevaleret                                           **
**      75013 Paris                                                      **
**      France                                                           **
**                                                                       **
 *************************************************************************/

#ifndef _INTERPCOM
#define _INTERPCOM

#ifndef _WIN32
#include <unistd.h>
#else
#include <direct.h>
#include <windows.h>
#endif
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#ifdef _DMALLOC
#include <dmalloc.h>
#else
#include <malloc.h>
#endif
#include <ctype.h>
#include <time.h>
#include <stdarg.h>
#include <setjmp.h>

#ifdef _HAVE_THREADS
#include <pthread.h>
extern pthread_mutex_t  mutex_ev,
                        mutex_obj,
                        mutex_thread,
                        mutex_string,
                        mutex_cond,
                        mutex_file,
                        mutex_quest;
#endif


/*-------------------------------------------------------------------------
---------------------------------------------------------------------------
       This contains a part of the file "ee.h" of Mark Morley's Expression
       Evaluator with some modifications
---------------------------------------------------------------------------
-------------------------------------------------------------------------*/
#define XVARLEN         15             /* Max length of variable names */
#define MAXXVARS        500            /* Max user-defined variables   */
#define TOKLEN          300            /* Max token length             */
#define MAX_F_ARGS      10             /* Maximal number of arguments
                                          of functions                 */

#define XVAR            1
#define DEL             2
#define NUM             3

typedef struct
{
   char             *name;             /* Variable name */
   double            value;            /* Variable value */
} XVARIABLE;

typedef struct
{
   char*            name;              /* Function name                 */
   int              args;              /* Number of arguments to expect */
   double         (*func)(double *);   /* Pointer to function           */
} FUNCTION;

typedef struct
{
   char*            name;              /* Function name                 */
   int              args;              /* Number of arguments to expect */
   char*            form;              /* Formula                       */
} FUNCTION_B;

#define iswhite(c)  (c == ' ' || c == '\t')
#define isnumer(c)  ((c >= '0' && c <= '9') || c == '.')
#define isalphab(c)  ((c == '@') || (c >= 'a' && c <= 'z') || (c >= '0' && c <= '9') \
                    || c == '_' || c == 127 || c == 126 || (c >= 'A' && c <= 'Z'))
#define isdelim(c)  (c == '+' || c == '-' || c == '*' || c == '/' || c == '%' \
                    || c == '^' || c == '(' || c == ')' || c == ',' \
                    || c == '=')

/* Codes returned from the evaluator */
#define E_OK           0        /* Successful evaluation                */
#define E_SYNTAX       1        /* Syntax error                         */
#define E_UNBALAN      2        /* Unbalanced parenthesis               */
#define E_DIVZERO      3        /* Attempted division by zero           */
#define E_UNKNOWN      4        /* Reference to unknown variable        */
#define E_MAXXVARS     5        /* Maximum variables exceeded           */
#define E_BADFUNC      6        /* Unrecognised function                */
#define E_NUMARGS      7        /* Wrong number of arguments to funtion */
#define E_NOARG        8        /* Missing an argument to a funtion     */
#define E_EMPTY        9        /* Empty expression                     */


/*---- defined in ee.c ------------*/
double          deg(double);
double          rad(double);
/*---------------------------------*/


extern XVARIABLE    Consts[];
extern FUNCTION    *Funcs;
extern FUNCTION     Funcs_interp[];
extern int          _NBFONC,
                    _NBFONC0,
                    _NBFONC_B;

extern XVARIABLE    VArs_global[MAXXVARS];
                        /* Array for user-defined variables
                           defined in all threads           */

void                strlwr2(char*);

extern char        *_EXPREV_errmessg[];
                        /* Error messages */
/*-------- End of "ee.h" ---------------------------------------------
--------------------------------------------------------------------*/






/*--------------------------------------------------------------------
----------------------------------------------------------------------
    Definition of complex numbers
----------------------------------------------------------------------
--------------------------------------------------------------------*/
typedef struct FCOMPLEX {
    float           r,
                    i;
} fcomplex;
typedef struct DCOMPLEX {
    double          r,
                    i;
} dcomplex;
typedef struct FPOLAIRE {
    float           rm,
                    th;
} fpolaire;
typedef struct DPOLAIRE {
    double          rm,
                    th;
} dpolaire;

typedef struct
{
   char            *name;   /* Variable name  */
   dcomplex         value;  /* Variable value */
} XVARIABLE_C;
extern XVARIABLE_C  VArs_global_C[MAXXVARS];
                 /* Array for user-defined variables */

typedef struct
{
   char*            name;                   /* Function name                 */
   int              args;                   /* Number of arguments to expect */
   dcomplex       (*func)(dcomplex *);    /* Pointer to function           */
} FUNCTIONC;

extern FUNCTIONC   *Funcs_C;
extern FUNCTIONC    Funcs_interp_C[];
extern int          _NBFONC_C;
extern int          _NBFONC0_C;
/*--------------------------------------------------------------------
--------------------------------------------------------------------*/






/*--------------------------------------------------------------------
----------------------------------------------------------------------
    GENERIC EXPRESSION EVALUATOR
----------------------------------------------------------------------
--------------------------------------------------------------------*/
typedef void      (*oper2_expr)(void *, void *, void *);
typedef void      (*oper1_expr)(void *);
typedef int       (*oper0_expr)(void *);
typedef void      (*oper3_expr)(void *, char *);
typedef void      (*oper4_expr)(void **);
typedef char*     (*oper5_expr)(void *);
typedef void*     (*oper6_expr)(void *);
typedef void*     (*oper)(char *);

typedef struct
{
    oper1_expr      clear;
    oper1_expr      Zero;
    oper1_expr      neg;
    oper0_expr      iszero;
    oper2_expr      add;
    oper2_expr      sub;
    oper2_expr      mul;
    oper2_expr      pow;
    oper2_expr      div;
    oper3_expr      set;
    oper4_expr      Init;
    oper5_expr      print;
    oper6_expr      copy;
    oper0_expr      sign;
    int             n_oper;
    char          **oper_name;
    oper           *OPER;
    int             n_var;
    char           *desc;
} EXPREVAL_GEN;

extern EXPREVAL_GEN   *Expreval_ops;

typedef struct {
    char           *name;
    void          **value;
} Vars_Gen;


extern Vars_Gen   **VArs_global_Gen;
extern int         *_NBFONC_Gen;
typedef struct
{
   char*            name;                   /* Function name                 */
   int              args;                   /* Number of arguments to expect */
   void           (*func)(void**, void**);  /* Pointer to function           */
} FUNCTIONGen;
extern FUNCTIONGen *Funcs_Gen[];
extern int          NB_expr_eval;           /* Number of supplementary
                                               evaluators */
/*--------------------------------------------------------------------
--------------------------------------------------------------------*/






/*--------------------------------------------------------------------
----------------------------------------------------------------------
    THREADS
----------------------------------------------------------------------
--------------------------------------------------------------------*/
typedef char       *argv_t[20];
typedef char        v_c[300];
typedef struct      _EVAL_EXPR {
    unsigned char  *expression;
    unsigned char   token[TOKLEN + 1];
    int             type;
    int             _EERROR;
    char            ERTOK[TOKLEN + 1];
    int             ERPOS;
    char           *ERANC;
    jmp_buf         jb;
} expr_eval;

#define __EXTRA_PARAM 100

typedef struct _flow_data {
    int             MODE_FONCT_;
    int             I__COM_CUR;
    int             I_LIGNE_CUR;
    int             I_LEC_CUR;
    int             I_SPEED;
    char           *HCOM;
    int             IS_COM;
    int             PRLEVEL;
    int             IND_COM;
    char           *_H_LIGNE;
    char          **_ARGV_C;
    int             _ARGC_C;
    char           *CMDX;
    int             CURVOICE;
    int             HORLOGE;
    long            I_TIME_X;
    long            _I_TIME_X;
    int             PR_COM;
    int             IX_COM;
    FILE          **INP;
    int            *ARGC_X;
    argv_t         *ARGV_X;
    FILE           *MON_FILE;
    int             I_COM;
    char          **COM_PREC;
    XVARIABLE      *VARS;
    double         *VARS2;
    XVARIABLE_C    *VARS_C;
    dcomplex       *VARS2_C;
    Vars_Gen      **VARS_Gen;
    Vars_Gen      **VARS2_Gen;
    void         ***VARSX_Gen;
    char           *name;
    int             used;
    int             thread;
    char          **extra;
    char          **h;
    char           *var;
    int             num;
    int             pause;
    int             mutex_obj;
    int             parse_com;
    int             kill;
    int             n_instr;
    int             expr_ev;
    int             is_ret;
} flow_data;

extern flow_data   *flow_I;
extern int        __n_max_threads,      /* Maximal number of threads */
                    num_thread_run;     /* Number of threads running */

#define INIT_FLOW(x)   {x=(flow_data *) argv[-1];}

/*--------------------------------------------------------------------
--------------------------------------------------------------------*/






/*--------------------------------------------------------------------
----------------------------------------------------------------------
    Memory management structures
----------------------------------------------------------------------
--------------------------------------------------------------------*/
typedef struct _MEMM {
    int             dim;
    char         ***ad;
    int             type;
    int             prof;
} memm;

typedef   int         (*i_fonc)(int, int);
typedef   double      (*d_fonc)(double, double);
typedef   fcomplex    (*fc_fonc)(fcomplex, fcomplex);
typedef   dcomplex    (*dc_fonc)(dcomplex, dcomplex);
typedef   fpolaire    (*fp_fonc)(fpolaire, fpolaire);
typedef   dpolaire    (*dp_fonc)(dpolaire, dpolaire);

typedef   int         (*Ri_fonc)(double, int);
typedef   double      (*Rd_fonc)(double, double);
typedef   fcomplex    (*Rfc_fonc)(double, fcomplex);
typedef   dcomplex    (*Rdc_fonc)(double, dcomplex);
typedef   fpolaire    (*Rfp_fonc)(double, fpolaire);
typedef   dpolaire    (*Rdp_fonc)(double, dpolaire);

typedef   int         (*i_fonc1)(int);
typedef   double      (*d_fonc1)(double);
typedef   fcomplex    (*fc_fonc1)(fcomplex);
typedef   dcomplex    (*dc_fonc1)(dcomplex);
typedef   fpolaire    (*fp_fonc1)(fpolaire);
typedef   dpolaire    (*dp_fonc1)(dpolaire);

typedef   int         (*i_foncX)(double);
typedef   double      (*d_foncX)(double);
typedef   fcomplex    (*fc_foncX)(double);
typedef   dcomplex    (*dc_foncX)(double);
typedef   fpolaire    (*fp_foncX)(double);
typedef   dpolaire    (*dp_foncX)(double);

typedef   fcomplex    (*fc_foncC)(dcomplex, fcomplex);
typedef   fpolaire    (*fp_foncC)(dcomplex, fpolaire);
typedef   dpolaire    (*dp_foncC)(dcomplex, dpolaire);

typedef   fcomplex    (*fc_foncCR)(double, double);
typedef   dcomplex    (*dc_foncCR)(double, double);
typedef   fpolaire    (*fp_foncCR)(double, double);
typedef   dpolaire    (*dp_foncCR)(double, double);

typedef   fcomplex    (*fc_fonc1C)(dcomplex);
typedef   fpolaire    (*fp_fonc1C)(dcomplex);
typedef   dpolaire    (*dp_fonc1C)(dcomplex);

typedef   double      (*dfonc2C)(dcomplex);
typedef   double      (*dfonc2P)(dpolaire);

typedef struct T_FONC {
    i_fonc          F_i;
    d_fonc          F_d;
    fc_fonc         F_fc;
    dc_fonc         F_dc;
    fp_fonc         F_fp;
    dp_fonc         F_dp;
} T_fonc;

typedef struct RT_FONC {
    Ri_fonc         F_i;
    Rd_fonc         F_d;
    Rfc_fonc        F_fc;
    Rdc_fonc        F_dc;
    Rfp_fonc        F_fp;
    Rdp_fonc        F_dp;
} RT_fonc;

typedef struct CC_FONC {
    dfonc2C        CF_d;
    dfonc2P        PF_d;
} CC_fonc;

typedef struct T_FONC1 {
    i_fonc1         F_i;
    d_fonc1         F_d;
    fc_fonc1        F_fc;
    dc_fonc1        F_dc;
    fp_fonc1        F_fp;
    dp_fonc1        F_dp;
} T_fonc1;

typedef struct T_FONCX {
    i_foncX         F_i;
    d_foncX         F_d;
    fc_foncX        F_fc;
    dc_foncX        F_dc;
    fp_foncX        F_fp;
    dp_foncX        F_dp;
} T_foncX;

typedef struct T_FONCC {
    fc_foncC        F_fc;
    dc_fonc         F_dc;
    fp_foncC        F_fp;
    dp_foncC        F_dp;
} T_foncC;

typedef struct T_FONCRC {
    Rfc_fonc        F_fc;
    Rdc_fonc        F_dc;
    Rfp_fonc        F_fp;
    Rdp_fonc        F_dp;
} T_foncRC;

typedef struct T_FONCCR {
    fc_foncCR       F_fc;
    dc_foncCR       F_dc;
    fp_foncCR       F_fp;
    dp_foncCR       F_dp;
} T_foncCR;

typedef struct T_FONCXd {
    fc_fonc1C       F_fc;
    dc_fonc1        F_dc;
    fp_fonc1C       F_fp;
    dp_fonc1C       F_dp;
} T_foncXd;

extern T_fonc       T_add;
extern T_fonc       T_sub;
extern T_fonc       T_mult;
extern T_fonc1      T_inv;
extern RT_fonc      T_Rmul;
extern T_foncX      R_cnst;
extern T_foncC      C_mul;
extern T_foncXd     C_cnst;
extern T_foncRC     Rpol_cnst;
extern T_foncRC     Thpol_cnst;
extern T_fonc1      T_id;
extern T_fonc1      C_conjg;
extern T_foncCR     C_Complex;
extern T_foncCR     C_Polaire;
extern CC_fonc      CC_Reel;
extern CC_fonc      CC_R;
extern CC_fonc      CC_Imag;
extern CC_fonc      CC_Th;

#define TEST_ALLOC(x) null_alloc((char *)(x))
#define XFREE(x) Xfree_b((char **)(x))
#define _INITIALISE    0
#define _NONINITIALISE 1
/*--------------------------------------------------------------------
--------------------------------------------------------------------*/





/*-------------------------------------------------------------------
---------------------------------------------------------------------
    Function types
---------------------------------------------------------------------
-------------------------------------------------------------------*/
typedef int       (*pfi)(int argc, char *argv[]);
typedef int       (*pfib)(FILE *, char *, flow_data *);
typedef void      (*pfi_print)(flow_data *, char *);
typedef int       (*pfi_ev)(char *, double *, int *, flow_data *);
typedef double    (*pfi_conv)(char *, flow_data *);
typedef void      (*pfi_char)(char *, char **, flow_data *);
typedef char *    (*pfi_get)(char *, int, FILE *);

typedef struct MISC_FUNC {
    pfi_print       Print;
    pfi_ev          Ev;
    pfi_conv        Conv;
} Misc_Func;
/*--------------------------------------------------------------------
--------------------------------------------------------------------*/





/*--------------------------------------------------------------------
----------------------------------------------------------------------
    Definition of objects and structures
----------------------------------------------------------------------
--------------------------------------------------------------------*/
typedef void      (*pfi_obj)(int, int, int, int, char *argv[], flow_data *);

typedef struct TYPE_OBJET {
    int             type;
    int             nombre;
    int             nbdim;
    char          **nom_dim;
    char           *nom;
    char           *comment;
    char           *alias;
    pfi_obj         X;
} obj_typ;

typedef struct _OBJET {
    obj_typ        *typ_obj;
    char           *adresse;
    int             occup;
    char           *nom_obj;
    int            *dim;
    int            *dep;
} obj;


typedef struct TYPE_STRUC {
    int             nb_membres;
    int            *type_mb;
    char          **membre_id;
    char           *nom;
    char           *comment;
    int             nombre;
} struc_typ;

typedef struct _STRUC {
    char           *nom_struc;
    int             occup;
    struc_typ      *type;
    char          **nom_mb;
    int            *dep;
} strucb;
/*--------------------------------------------------------------------
--------------------------------------------------------------------*/





/*--------------------------------------------------------------------
----------------------------------------------------------------------
    Substitution patterns
----------------------------------------------------------------------
--------------------------------------------------------------------*/
typedef struct SUBST_DELIM {
    char            begin;
    char            end;
} subst_delim;

typedef struct SUBST_PAT {
    char            orig;
    subst_delim     delim;
    pfi_char        F;
} subst_pat;

extern int          n_subst_pat,
                    __max_subst_pat;
extern subst_delim *Subst_Delim;
extern subst_pat   *Subst_Pat;
/*--------------------------------------------------------------------
--------------------------------------------------------------------*/





/*--------------------------------------------------------------------
----------------------------------------------------------------------
    Function prototypes
----------------------------------------------------------------------
--------------------------------------------------------------------*/

/*====================================================================
                        com_sys.c
====================================================================*/
int             lookup_b(char *, flow_data *);
void            execute(int i, int argc, char *argv[], flow_data *);
int             sign_conv(char *, flow_data *);
char           *fgets_count(char *, int, FILE *);
void            charge_com(char *, char int_ini_c[], int, int, flow_data *,
                    FILE *, int);
int             isnon_0(char *);
void            Find_subst(int, int);
void            interp_com(FILE *, char *, flow_data *);
int             mode_cmd(FILE *, char *, flow_data *);
int             greet_cmd(FILE *, char *, flow_data *);
int             gr_cmd(FILE *, char *, char **, int *, int);
int             func_cmd(FILE *, char *, flow_data *);
int             mess_cmd(FILE *, char *, flow_data *);
int             test_param(int, int);
int             param_cmd(FILE *, char *, flow_data *);
int             var_cmd(FILE *, char *, flow_data *);
int             rep_cmd(FILE *, char *, flow_data *);
int             init_interp_cmd(FILE *, char *, flow_data *);
void            exit_interp(char *);
void            err_mess(int);
void            traite_label(flow_data *);
void            extrait_argv_com(int, int *, int, char *argv_com[],
                    flow_data *thread_interp);
void            supprime_ligne_com(int, int);
void            insere_ligne_com(int, int, char *, flow_data *);
int             exist_com(char *);
void            ch_prem(void);
int             est_avant(char *, char *);
void            echange_nom(int, int);
int             delprog_cmd(int argc, char *argv[]);
FILE           *fmemopen_(void *, int, char *, FILE *);
char           *__Xdecode(char xg[], int, char *);
void            __perms(char *, char *);
char            __chx(int);
int             include_cmd(FILE *, char *, flow_data *);
int             color_cmd(FILE *, char *, flow_data *);
void            interpcom_dummy(int *);
int             help_file_cmd(FILE *, char *, flow_data *);
void            help_add_file(FILE *);
void            help_reorder(void);


/*====================================================================
                        command.c
====================================================================*/
int             bouclex(int argc, char *argv[]);
int             bouclex_f(int argc, char *argv[]);
int             echo_cmd(int argc, char *argv[]);
int             repetex(int argc, char *argv[]);
int             editeur(int argc, char *argv[]);
int             delcom(int argc, char *argv[]);
int             deldon(int argc, char *argv[]);
int             delres(int argc, char *argv[]);
int             si_cmd(int argc, char *argv[]);
int             is_cmd(int argc, char *argv[]);
int             file_cmd(int argc, char *argv[]);
int             close_file_cmd(int argc, char *argv[]);
int             question_cmd(int argc, char *argv[]);
int             var_list_cmd(int argc, char *argv[]);
int             undef_VARS(char *, XVARIABLE *);
int             undef_cmd(int argc, char *argv[]);
int             echo_int_cmd(int argc, char *argv[]);
int             echo_float_cmd(int argc, char *argv[]);
int             fread_cmd(int argc, char *argv[]);
int             silence_cmd(int argc, char *argv[]);
int             send_print_cmd(int argc, char *argv[]);
int             temps(int argc, char *argv[]);
int             exit_cmd(int argc, char *argv[]);
int             proglist_cmd(int argc, char *argv[]);
int             load_cmd(int argc, char *argv[]);
int             loadcmd(char *);
int             shell_cmd(int argc, char *argv[]);
void            init_var(flow_data *);
int             init_var_cmd(int argc, char *argv[]);
int             num_com(int argc, char *argv[]);
int             hist_cmd(int argc, char *argv[]);
int             mon_cmd(int argc, char *argv[]);
int             fin_mon_cmd(int argc, char *argv[]);
void            print(flow_data *, char*, ...);
void            null_print(flow_data *, char *);
void            extern_print(flow_data *, char *);
int             flush_cmd(int argc, char *argv[]);
int             greetb_cmd(int argc, char *argv[]);
void            error_mess(flow_data *, int);
FILE           *Copen(char *, char *, char *);
FILE           *CCopen(char *, char *, char *, char *);
FILE           *CIopen(char *, char *, char *, int);
int             time_cmd(int argc, char *argv[]);
int             setdir_cmd(int argc, char *argv[]);
int             setstring_cmd(int argc, char *argv[]);
int             delstring_cmd(int argc, char *argv[]);
int             liststring_cmd(int argc, char *argv[]);
int             ch_expr_cmd(int argc, char *argv[]);
int             color_echo_cmd(int argc, char *argv[]);
void            de_color_(char *);
void            de_bb_(char *);
int             help_cmd(int argc, char *argv[]);
int             comp_deb(char *, char *, int);
int             trace_cmd(int argc, char *argv[]);
int             Function_cmd(int argc, char *argv[]);
int             ListFunc_cmd(int argc, char *argv[]);
char           *_print_funct(char *, int, char *);


/*====================================================================
                        complex.c
====================================================================*/
fcomplex        Cadd(fcomplex, fcomplex);
fcomplex        Csub(fcomplex, fcomplex);
fcomplex        Cnegat(fcomplex);
fcomplex        Cmul(fcomplex, fcomplex);
fcomplex        Complex(float, float);
fcomplex        Conjg(fcomplex);
fcomplex        Cdiv(fcomplex, fcomplex);
float           Cabs(fcomplex);
fcomplex        Csqrt(fcomplex);
fcomplex        RCmul(float, fcomplex);
fcomplex        RCdiv(fcomplex, float);
fcomplex        Cinv(fcomplex);
dcomplex        dCadd(dcomplex, dcomplex);
dcomplex        dCsub(dcomplex, dcomplex);
dcomplex        dCnegat(dcomplex);
dcomplex        dCmul(dcomplex, dcomplex);
dcomplex        dComplex(double, double);
dcomplex        dConjg(dcomplex);
dcomplex        dCdiv(dcomplex, dcomplex);
double          dCabs(dcomplex);
dcomplex        xdCsqrt(dcomplex);
dcomplex        dRCmul(double, dcomplex);
dcomplex        dRCdiv(dcomplex, double);
dcomplex        xdCinv(dcomplex);
dcomplex        dCpow(dcomplex, dcomplex);
fpolaire        Conv_ctp(fcomplex);
fcomplex        Conv_ptc(fpolaire);
dcomplex        dConv_ptc(dpolaire);
dpolaire        dConv_ctp(dcomplex);
fcomplex        Clog_p(fpolaire);
fcomplex        Clog_c(fcomplex);
fpolaire        Plog_c(fpolaire);
fcomplex        Csin_c(fcomplex);
fcomplex        Ccos_c(fcomplex);
fpolaire        Pexp_c(fcomplex);
fcomplex        Cexp_c(fcomplex);
fpolaire        Pexp_p(fpolaire);
dcomplex        dClog_p(dpolaire);
dcomplex        dClog_c(dcomplex);
dpolaire        dPlog_c(dpolaire);
dcomplex        dCsin_c(dcomplex);
dcomplex        dCcos_c(dcomplex);
dpolaire        dPexp_c(dcomplex);
dcomplex        dCexp_c(dcomplex);
dpolaire        dPexp_p(dpolaire);
fpolaire        Polaire(float, float);
dpolaire        dPolaire(double, double);
fpolaire        Padd(fpolaire, fpolaire);
fpolaire        Psub(fpolaire, fpolaire);
fpolaire        Pmul(fpolaire, fpolaire);
fpolaire        Pdiv(fpolaire, fpolaire);
fpolaire        Pconjg(fpolaire);
fpolaire        Psqrt(fpolaire);
fpolaire        Ppuis(fpolaire, float);
fcomplex        Cpuis(fcomplex, float);
fpolaire        RPmul(float, fpolaire);
fpolaire        RPdiv(fpolaire, float);
fpolaire        Pinv(fpolaire);
dpolaire        dPadd(dpolaire, dpolaire);
dpolaire        dPsub(dpolaire, dpolaire);
dpolaire        dPmul(dpolaire, dpolaire);
dpolaire        dPdiv(dpolaire, dpolaire);
dpolaire        dPconjg(dpolaire);
dpolaire        dPsqrt(dpolaire);
dpolaire        dPpuis(dpolaire, double);
dcomplex        dCpuis(dcomplex, double);
dpolaire        dRPmul(double, dpolaire);
dpolaire        dRPdiv(dpolaire, double);
dpolaire        dPinv(dpolaire);
dcomplex        dCtan_c(dcomplex);
dcomplex        dCsinh_c(dcomplex);
dcomplex        dCcosh_c(dcomplex);
dcomplex        dCtanh_c(dcomplex);
fcomplex        Ctan_c(fcomplex);
fcomplex        Csinh_c(fcomplex);
fcomplex        Ccosh_c(fcomplex);
fcomplex        Ctanh_c(fcomplex);
dcomplex        CdCabs(dcomplex);


/*====================================================================
                        ee.c
====================================================================*/
int             ClearVar(char *, flow_data *);
int             GetValue(char *, double *, flow_data *);
int             SetValue(char *, double *, flow_data *);
int             Isalphab(char *);
int             Evaluate(char *, double *, int *, flow_data *);
int             GetValue_Global(char *, double *);
int             GetValue_Global_hidden(char *, double *);
void            PTHREAD_LOCK_EV(void);
void            PTHREAD_UNLOCK_EV(void);
double          deg(double);
double          rad(double);
void            Parse(expr_eval *);
int             Level1(double *, expr_eval *, flow_data *);
void            Level2(double *, expr_eval *, flow_data *);
void            Level3(double *, expr_eval *, flow_data *);
void            Level4(double *, expr_eval *, flow_data *);
void            Level5(double *, expr_eval *, flow_data *);
void            Level6(double *, expr_eval *, flow_data *);
double          Eval_Funcs_b(char *, double *, int, flow_data *);
int             ClearVar_C(char *, flow_data *);
int             GetValue_C(char *, dcomplex *, flow_data *);
int             SetValue_C(char *, dcomplex *, flow_data *);
int             Evaluate_C(char *, dcomplex *, int *, flow_data *);
int             GetValue_Global_C(char *, dcomplex *);
int             GetValue_Global_hidden_C(char *, dcomplex *);
void            Parse_C(expr_eval *);
int             Level1_C(dcomplex *, expr_eval *, flow_data *);
void            Level2_C(dcomplex *, expr_eval *, flow_data *);
void            Level3_C(dcomplex *, expr_eval *, flow_data *);
void            Level4_C(dcomplex *, expr_eval *, flow_data *);
void            Level5_C(dcomplex *, expr_eval *, flow_data *);
void            Level6_C(dcomplex *, expr_eval *, flow_data *);
dcomplex        Eval_Funcs_b_C(char *, dcomplex *, int, flow_data *);
int             ClearVar_Gen(int, char *, flow_data *);
int             GetValue_Gen(int, char *, void **, flow_data *);
int             SetValue_Gen(int, char *, void **, flow_data *);
int             Evaluate_Gen(int, char *, void **, int *, flow_data *);
int             GetValue_Global_Gen(int, char *, void **);
int             GetValue_Global_hidden_Gen(int, char *, void **);
void            Parse_Gen(expr_eval *);
int             Level1_Gen(int, void **, expr_eval *, flow_data *);
void            Level2_Gen(int, void **, expr_eval *, flow_data *);
void            Level3_Gen(int, void **, expr_eval *, flow_data *);
void            Level4_Gen(int, void **, expr_eval *, flow_data *);
void            Level5_Gen(int, void **, expr_eval *, flow_data *);
void            Level6_Gen(int, void **, expr_eval *, flow_data *);
void            Eval_Funcs_b_Gen(int, void **, char *, void **, int,
                    flow_data *);
void           *GetValue_Gen_0(int, char *, flow_data *);


/*====================================================================
        interface (functions defined in the applications)
====================================================================*/
void            exit_prog(void);
void            init_prog(flow_data *);
void            init_expr_GEN(flow_data *);
void            dest_prop(int, int, flow_data *);
void            init_thread_param(flow_data *);
void            clean_thread_param(flow_data *);
void            init_expr_GEN(flow_data *);


/*====================================================================
                        interp.c
====================================================================*/
void            prog_c(int argc, char *argv[], char *, char int_ini_c[], int,
                    char *, int);
void            lect_com(flow_data *);
void            extrait_arg(flow_data *);
void            execute_c(void);
int             change_lev(flow_data *);
void            Subst_Pat_act0(char *, char **, flow_data *);
void            Subst_Pat_act1(char *, char **, flow_data *);
void            Subst_Pat_act2(char *, char **, flow_data *);
void            Subst_Pat_act3(char *, char **, flow_data *);
void            Subst_Pat_act4(char *, char **, flow_data *);
void            substit_argc_c2(char *, char **, flow_data *, int*);
void            default_Subst_Pat(void);
void            substit(flow_data *);
void            substit2(flow_data *);
void            shell_c(int argc, char *argv[], flow_data *);
int             lookup_c(char *names[], char *);
void            prTime(flow_data *);
double          (*fonc(char *))(double ,double);
int             convert_int(char *, flow_data*);
int             S_convert_int(char *, flow_data*);
double          convert_float(char *, flow_data*);
double          S_convert_float(char *, flow_data*);
int             eval_cmd(int argc, char *argv[]);
int             is_alphab(char);
int             is_alphab_or_num(char);
int             parse_com(char *, char *, flow_data *);
void            read_float(float *, flow_data *);
void            read_int(int *, flow_data *);
void            read_char(char *, flow_data *);
int             parse_def(char *, int *, char *, flow_data *);
void            extrait_arg_s(char *, double *, int *, flow_data *);
dcomplex        calc_val_obj(int, double *, char *, int, flow_data *);
void            print_ev_error(int, flow_data *);
void            init_VARS(XVARIABLE *);
void            init_VARS_C(XVARIABLE_C *);
void            init_flow_param(flow_data *);
void            init_flow_param_b(flow_data *);
void            init_expr_eval(Vars_Gen ***, int);
void            init_expr_GEN_def(int);
void            clean_flow_param(flow_data *);
void            Run_interp(flow_data *);
void            Exec_interp_command(char *);
void            clean_exit_interp(flow_data *);
int             thread_interp_create(int argc, char *argv[]);
void           *thread_exec(void *);
int             list_threads(int argc, char *argv[]);
int             pause_thread(int argc, char *argv[]);
int             wait_thread(int argc, char *argv[]);
int             kill_thread(int argc, char *argv[]);


/*====================================================================
                        mem.c
====================================================================*/
memm           *memm_alloc(int *, int, int);
memm           *_M(char **);
char           *addr_eff(memm *);
char           *addr_eff_b(memm *);
void            Xfree(memm *);
void            Xfree_b(char **);
int             longueur(memm *M);
int             compare_tab(memm *, memm *);
int             compare_dim(memm *, memm *);
int            *int_alloc1(int);
int           **int_alloc2(int, int);
int          ***int_alloc3(int, int, int);
int         ****int_alloc4(int, int, int, int);
float          *float_alloc1(int);
float         **float_alloc2(int, int);
float        ***float_alloc3(int, int, int);
float       ****float_alloc4(int, int, int, int);
double         *double_alloc1(int);
double        **double_alloc2(int, int);
double       ***double_alloc3(int, int, int);
double      ****double_alloc4(int, int, int, int);
fcomplex       *fcomplex_alloc1(int);
fcomplex      **fcomplex_alloc2(int, int);
fcomplex     ***fcomplex_alloc3(int, int, int);
fcomplex    ****fcomplex_alloc4(int, int, int, int);
dcomplex       *dcomplex_alloc1(int);
dcomplex      **dcomplex_alloc2(int, int);
dcomplex     ***dcomplex_alloc3(int, int, int);
dcomplex    ****dcomplex_alloc4(int, int, int, int);
fpolaire       *fpolaire_alloc1(int);
fpolaire      **fpolaire_alloc2(int, int);
fpolaire     ***fpolaire_alloc3(int, int, int);
fpolaire    ****fpolaire_alloc4(int, int, int, int);
dpolaire       *dpolaire_alloc1(int);
dpolaire      **dpolaire_alloc2(int, int);
dpolaire     ***dpolaire_alloc3(int, int, int);
dpolaire    ****dpolaire_alloc4(int, int, int, int);
void            manip1_tab(memm *, memm *, memm *, T_fonc *, RT_fonc *,
                    T_foncC *, T_fonc1 *, T_foncX *,  T_foncRC *,  T_foncCR *,
                        T_foncXd *, double, dcomplex,int, CC_fonc *);
void            ajoute_tab(memm *, memm *, memm *);
void            soustrait_tab(memm *, memm *, memm *);
void            multiplie_tab(memm *, memm *, memm *);
void            Rmul_tab(memm *, double, memm *);
void            Cmul_tab(memm *, dcomplex, memm *);
void            Conjg_tab(memm *, memm *);
void            Module_tab(memm *, memm *);
void            Phase_tab(memm *, memm *);
void            copie_tab(memm *, memm *);
void            reel_tab(memm *, memm *);
void            imag_tab(memm *, memm *);
void            Polaire_tab(memm *, memm *, memm *);
void            Complex_tab(memm *, memm *, memm *);
void            cnst_tab(memm *, double);
void            Ccnst_tab(memm *, dcomplex);
void            RPolcnst_tab(memm *, double);
void            ThPolcnst_tab(memm *, double);
memm           *_M(char **);
int             svg_tab(FILE *, char *, int);
int             iadd(int, int);
double          dadd(double, double);
int             isub(int, int);
double          dsub(double, double);
int             i_Rmul(double, int);
double          d_mult(double, double);
fcomplex        RCmulb(double, fcomplex);
fpolaire        RPmulb(double, fpolaire);
int             i_mult(int, int);
int             i_inv(int);
double          d_inv(double);
int             i_cnst(double);
double          d_cnst(double);
fcomplex        fc_cnst(double);
dcomplex        dc_cnst(double);
fpolaire        fp_cnst(double);
dpolaire        dp_cnst(double);
fcomplex        Ccnst(dcomplex);
dcomplex        dCcnst(dcomplex);
fpolaire        Pcnst(dcomplex);
dpolaire        dPcnst(dcomplex);
int             icnst(int);
double          dcnst(double);
fcomplex        Ccnst_id(fcomplex);
fpolaire        Pcnst_id(fpolaire);
dpolaire        dPcnst_id(dpolaire);
fcomplex        Cmul_x(dcomplex, fcomplex);
fpolaire        Pmul_x(dcomplex, fpolaire);
dpolaire        dPmul_x(dcomplex, dpolaire);
fcomplex        fRPolcnst(double, fcomplex);
dcomplex        dRPolcnst(double, dcomplex);
fpolaire        PRPolcnst(double, fpolaire);
dpolaire        dPRPolcnst(double, dpolaire);
fcomplex        fThPolcnst(double, fcomplex);
dcomplex        dThPolcnst(double, dcomplex);
fpolaire        PThPolcnst(double, fpolaire);
dpolaire        dPThPolcnst(double, dpolaire);
fpolaire        PComplex(double, double);
dpolaire        dPComplex(double, double);
fcomplex        CPolaire(double, double);
dcomplex        dCPolaire(double, double);
double          Reel_dC(dcomplex);
double          Reel_dP(dpolaire);
double          Imag_dC(dcomplex);
double          Imag_dP(dpolaire);
double          R_dC(dcomplex);
double          R_dP(dpolaire);
double          Th_dC(dcomplex);
double          Th_dP(dpolaire);


/*====================================================================
                        objdef.c
====================================================================*/
int             objdef_cmd(FILE *, char *, flow_data *);
void            nettoie(char *);
void            make_com(char *, int, int *);
void            PTHREAD_LOCK_OBJ(flow_data *);
void            PTHREAD_UNLOCK_OBJ(flow_data *);
int             obj_create(int argc, char *argv[]);
int             test_nom_obj(char *, int, flow_data *);
int             detruit_obj(int argc, char *argv[]);
int             liste_obj(int argc, char *argv[]);
void            init_obj(int, flow_data *);
int             sketch_obj(char *, int *, flow_data *);
int             sketch_obj_restr(char *, int *, int, flow_data *);
int             sketch_obj_restr_b(char *, int *, int, flow_data *);
int             sketch_obj2(char *, char *, int *, int *, flow_data *);
int             sketch_obj3(char *, char *, char *,int *, int *, int *,
                    flow_data *);
int             add_objet(int argc, char *argv[]);
int             sub_objet(int argc, char *argv[]);
int             Xmult_objet(int argc, char *argv[]);
int             copie_objet(int argc, char *argv[]);
int             svg_cmd(int argc, char *argv[]);
int             restore_cmd(int argc, char *argv[]);
int             set_obj_val_cmd(int argc, char *argv[]);
int             fill_obj_cmd(int argc, char *argv[]);
int             cnst(int argc, char *argv[]);
int             mult_objet(int argc, char *argv[]);
int             Cmult_objet(int argc, char *argv[]);
char           *ch_copy(char *);
char           *ch_copy_int(int);
char           *ch_copy_float(double);
int             Ccnst_cmd(int argc, char *argv[]);
int             RPolcnst(int argc, char *argv[]);
int             ThPolcnst(int argc, char *argv[]);
int             Real_part(int argc, char *argv[]);
int             Imag_part(int argc, char *argv[]);
int             Radius_part(int argc, char *argv[]);
int             Phase_part(int argc, char *argv[]);
int             conjg_cmd(int argc, char *argv[]);
int             mkcomplex_cmd(int argc, char *argv[]);
int             mkpolaire_cmd(int argc, char *argv[]);
int             struct_cmd(FILE *, char *, flow_data *);
int             test_name_str(char *);
void            make_com_str(char *, int, int *);
int             test_nom_str(char *, int);
void            init_str(int, flow_data *);
int             assign_membre(int argc, char *argv[]);
int             desassign_membre(int argc, char *argv[]);
int             desc_struct(int argc, char *argv[]);
int             struc_create(int argc, char *argv[]);
int             sketch_struc(int, int, char *, int *, flow_data *);
int             dep_obj_struct(int argc, char *argv[]);
int             undep_obj_struct(int argc, char *argv[]);
int             update_obj_struct(int argc, char *argv[]);
/*--------------------------------------------------------------------
----------------------------------------------------------------------
--------------------------------------------------------------------*/




#define comp(s1,s2) (!strcmp((s1),(s2)))


/*--------------------------------------------------------------------
    Some of the parameters set in the section !param
    of the initialization file
--------------------------------------------------------------------*/
extern int          __nbcom,       /* Maximal number of programs           */
                    __nblignes,    /* Maximal number of lines in a program */
                    __com_max,     /* Maximal number of commands           */
                    __greet_max,   /* Maximal number of lines in the
                                      greeting message                     */
                    __mess_max,    /* Maximal number of messages           */
                    __nss,         /* Maximal number of open files         */
                    __nmode_fonc,  /* Maximal number of running modes      */
                    __maxvoice,    /* Maximal number of voices             */
                    __nbcond,      /* Maximal number of conditions         */
                    __nbtypmax,    /* Maximal number of object types       */
                    __nbstrucmax,  /* Maximal number of structure types    */
                    __nbargmax,    /* Maximal number of arguments          */
                    __nblabelmax,  /* Maximal number of labels in programs */
                    __max_quest,   /* Maximal number of questions          */
                    __n_com_prec,  /* Maximal number of stored preceeding
                                      commands                             */
                    __max_Env_st,  /* Maximal number of string variables   */
                    __n_expr_eval, /* Maximal number of expression
                                      evaluators */
                    __nbhelpsubj, /* Maximal number of help subjects       */
                    __nbuserfunc; /* Maximal number of user defined
                                     functions */
/*------------------------------------------------------------------*/

extern int          ind_x_mode,
                    ind_x_func,
                    ind_x_param,
                    ind_x_rep,
                    ind_run,
                    i_func_ind,
                    Init_File_Line;

extern  pfi        *proc[];
extern  pfi        *procw;
extern  pfib        procb[],
                    procb_user[];
extern  char       *namesb[],
                   *namesb_user[];
extern  char       *mess_interp[];
extern  pfi_get     _interp_fgets;

extern char       **mess,
                  **prompt_mode,
                  **greet,
                  **names,
                  **nom_com,
                 ***ligne_com,
                ****Argv_com,
                   *argv_init_interp_b[21],
                  **argv_init_interp;

extern int          i_greet,
                   *nb_label,
                   *tr_label,
                  **num_label,
                  **Argc_com,
                  **Is_subst_ligne,
                 ***Is_subst_arg,
                   *Is_Del_prog,
                   *Is_visible_prog,
                   *len_n,
                   *sil_com,
                  **mode_com_int,
                  **mode_com_str,
                  **mode_com_obj,
                   *nb_par,
                    nb_com,
                   *nb_lignes,
                    nb_mode_fonc,
                    nb_commandes,
                    i_message,
                  **mode_com,
                   *par_com,
                   *deb_nom,
                   *fin_nom,
                    i_init_interp,
                    sil_init_interp,
                    argc_init_interp,
                    ind_greet_x,
                    func_init;



/*--------------------------------------------------------------------
    Variables used by conditions (commands 'si' and 'is')
--------------------------------------------------------------------*/
extern int          n_cond,
                   *s_cond;
extern v_c         *v_cond;
/*------------------------------------------------------------------*/



/*---- Variables used to manage files (monitor files, commands
       'read', 'write', etc...) ------------------------------------*/
extern FILE       **sS;
extern int         *sS_cmpt,
                   *sS_i_o;
extern char       **sS_nom;
/*-------------------------------------------------------------------*/



/*--------------------------------------------------------------------
    Variable used with question files
---------------------------------------------------------------------*/
extern char       **ques;
/*-------------------------------------------------------------------*/



/*--------------------------------------------------------------------
    String variables
--------------------------------------------------------------------*/
extern char       **Env_string,
                  **Env_string2;
/*------------------------------------------------------------------*/



/*--------------------------------------------------------------------
    Predefined directories (section '!rep')
--------------------------------------------------------------------*/
extern char         command_rep[80],
                    data_rep[80],
                    data_rep2[80],
                    result_rep[80];
/*------------------------------------------------------------------*/



/*--------------------------------------------------------------------
    Creation of objects
--------------------------------------------------------------------*/
extern int          nb_typ;

extern obj_typ     *Obj_typ;
extern obj        **Obj;
extern int         *Obj_alias;
/*------------------------------------------------------------------*/



/*--------------------------------------------------------------------
    Creation of structures
--------------------------------------------------------------------*/
extern int          nb_struc;

extern struc_typ   *Struc_typ;
extern strucb     **Struc;
/*------------------------------------------------------------------*/


/*--------------------------------------------------------------------
    Name of user's initialization file
--------------------------------------------------------------------*/
extern char        *User_Init_File;
/*------------------------------------------------------------------*/



/*--------------------------------------------------------------------
    Colors
--------------------------------------------------------------------*/
extern char        *col_prompt,
                   *col_mess,
                   *col_prog,
                   *end_col_prompt,
                   *end_col_mess,
                   *col_echo;
extern int          ind_monfile_color;
/*------------------------------------------------------------------*/



/*--------------------------------------------------------------------
    Help files
--------------------------------------------------------------------*/
extern char       **help_name,
                  **help_content;
extern int          nb_help_subj,
                   *help_name_length,
                   *nb_help_print;
/*------------------------------------------------------------------*/



/*--------------------------------------------------------------------
    User defined functions
--------------------------------------------------------------------*/
extern FUNCTION_B  *Funcs_b;
extern char       **Vars_F_name;
/*------------------------------------------------------------------*/




#define _GOTO       15
#define _IFGT       16
#define _IFLT       17
#define _IFEQ       18
#define _EVAL       19
#define _PRIME_C    8803
#define _P_MULT     2743
#define _L_OBJNAME  80
#define MAX_CARS_COM 300

#endif
