/*************************************************************************
**                                                                       **
** ee.c         Expression Evaluator (in interpcom-3.1  )                **
**                                                                       **
** AUTHOR:      Mark Morley (modified by J.M. Drezet)                    **
** COPYRIGHT:   (c) 1992 by Mark Morley                                  **
** DATE:        December 1991                                            **
** HISTORY:     Jan 1992 - Made it squash all command line arguments     **
**                         into one big long string.                     **
**                       - It now can set/get VMS symbols as if they     **
**                         were variables.                               **
**                       - Changed max variable name length from 5 to 15 **
**              Jun 1992 - Updated comments and docs                     **
**              Sep 1999 - Local and global variables (JMD)              **
**                                                                       **
** You are free to incorporate this code into your own works, even if it **
** is a commercial application.  However, you may not charge anyone else **
** for the use of this code!  If you intend to distribute your code,     **
** I'd appreciate it if you left this message intact.  I'd like to       **
** receive credit wherever it is appropriate.  Thanks!                   **
**                                                                       **
** I don't promise that this code does what you think it does...         **
**                                                                       **
** For the original version:                                             **
** Please mail any bug reports/fixes/enhancments to me at:               **
**      morley@camosun.bc.ca                                             **
** or                                                                    **
**      Mark Morley                                                      **
**      3889 Mildred Street                                              **
**      Victoria, BC  Canada                                             **
**      V8Z 7G1                                                          **
**      (604) 479-7861                                                   **
**                                                                       **
** For the modified version:                                             **
** Please mail any bug reports/fixes/enhancements to me at:              **
**      drezet@math.jussieu.fr                                           **
** or                                                                    **
**      Jean-Marc Drezet                                                 **
**      Institut de Mathematiques                                        **
**      UMR 7586 du CNRS                                                 **
**      173, rue du Chevaleret                                           **
**      75013 Paris                                                      **
**      France                                                           **
**                                                                       **
 *************************************************************************/


#include <stdlib.h>
#include <math.h>
#include <setjmp.h>

#include "interp.h"             /* Modification JMDR */

#define ERR(n) {eval0->_EERROR=n; eval0->ERPOS=(int)eval0->expression-(int)eval0->ERANC-1; strcpy(eval0->ERTOK,(char*)eval0->token); longjmp(eval0->jb,1);}

/*
 * These defines only happen if the values are not already defined!  You
 * may want to add more precision here, if your machine supports it.
 */
#ifndef M_PI
#define M_PI    3.14159265358979323846
#endif
#ifndef M_E
#define M_E     2.71828182845904523536
#endif
#define Nfree(a) {if ((a) != NULL) free((a));}


XVARIABLE       Consts[] =
{
   /* name, value */
    {"pi", M_PI},
    {"e", M_E},

    {"\0", 0.},
};

extern FUNCTION    *Funcs;
extern int          _NBFONC;

double
deg(double x)
{
    return (x * 180.0 / M_PI);
}

double
rad(double x)
{
    return (x * M_PI / 180.0);
}



/*--------------------------------------------------------------------
--------------------------------------------------------------------*/
void
PTHREAD_LOCK_EV(void)
{
#ifdef _DEBUG_THREADS
    printf("mutex_ev = 1\n");
#endif
#ifdef _HAVE_THREADS
    pthread_mutex_lock(&mutex_ev);
#endif
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
--------------------------------------------------------------------*/
void
PTHREAD_UNLOCK_EV(void)
{
#ifdef _DEBUG_THREADS
    printf("mutex_ev = 0\n");
#endif
#ifdef _HAVE_THREADS
    pthread_mutex_unlock(&mutex_ev);
#endif
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Erases the user-defined variable that is called 'name' from memory.
    Note that constants are not affected.
    Returns 1 if the variable was found and erased, or 0 if it didn't
    exist.
--------------------------------------------------------------------*/
int
ClearVar(char *name, flow_data *flow_interp)
{
    int             i;
    char            c;

    if (*name == '@' && strlen(name) > 1) {
        c = name[1];
        if (c >= 'a' && c <= 'z') {
            flow_interp->VARS2[c - 'a'] = 0.;
            return 1;
        }
        if (c >= 'A' && c <= 'Z') {
            flow_interp->VARS2[c - 'A' + 26] = 0.;
            return 1;
        }
        return (0);
    }
    if (*name != '_' && *name != 127) {
        for (i = 0; i < MAXXVARS; i++)
            if (*flow_interp->VARS[i].name && !strcmp(name,
                flow_interp->VARS[i].name)) {
                *flow_interp->VARS[i].name = 0;
                flow_interp->VARS[i].value = 0.;
                return (1);
            }
    } else {
        PTHREAD_LOCK_EV();

        for (i = 0; i < MAXXVARS; i++)
            if (*VArs_global[i].name && !strcmp(name, VArs_global[i].name)) {
                *VArs_global[i].name = 0;
                VArs_global[i].value = 0.;
                PTHREAD_UNLOCK_EV();
                return (1);
            }

        PTHREAD_UNLOCK_EV();
    }
    return (0);
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Looks up the specified variable (or constant) known as NAME and
    returns its contents in VALUE.
    First the user-defined variables are searched, then the constants are
    searched.
    Returns 1 if the value was found, or 0 if it wasn't.                         **
--------------------------------------------------------------------*/
int
GetValue(char *name, double *value, flow_data *flow_interp)
{
    int             i;
    char            c;

    *value = 0.;
    if (name == NULL)
        return (0);
    if (*name == '@') {
        c = name[1];
        if (c >= 'a' && c <= 'z') {
            *value = flow_interp->VARS2[c - 'a'];
            return 1;
        }
        if (c >= 'A' && c <= 'Z') {
            *value = flow_interp->VARS2[c - 'A' + 26];
            return 1;
        }
        return (0);
    }
    if (*name != '_' && *name != 127) {
        for (i = 0; i < MAXXVARS; i++)
            if (*flow_interp->VARS[i].name &&
                !strcmp(name, flow_interp->VARS[i].name)) {
                *value = flow_interp->VARS[i].value;
                return (1);
            }
        for (i = 0; *Consts[i].name; i++)
            if (*Consts[i].name && !strcmp(name, Consts[i].name)) {
                *value = Consts[i].value;
                return (1);
            }
    }
    else {
        PTHREAD_LOCK_EV();

        for (i = 0; i < MAXXVARS; i++)
            if (*VArs_global[i].name && !strcmp(name, VArs_global[i].name)) {
                *value = VArs_global[i].value;
                PTHREAD_UNLOCK_EV();
                return (1);
            }

        PTHREAD_UNLOCK_EV();
    }
    return (0);
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
--------------------------------------------------------------------*/
int
GetValue_Global_hidden(char *name, double *value)
{
    int             i,
                    l;
    char           *h;

    PTHREAD_LOCK_EV();
    *value = 0.;
    if (name == NULL)
        return (0);
    l = strlen(name) + 2;
    h = (char *) malloc((size_t) l * sizeof(char));
    memset(h, 0, l);
    h[0] = 127;

    for (i = 0; i < l - 2; i++)
        h[i + 1] = name[i];

    for (i = 0; i < MAXXVARS; i++)
        if (*VArs_global[i].name && !strcmp(h, VArs_global[i].name)) {
            *value = VArs_global[i].value;
            PTHREAD_UNLOCK_EV();
            free(h);
            return (1);
        }

    PTHREAD_UNLOCK_EV();
    free(h);
    return (0);
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
--------------------------------------------------------------------*/
int
GetValue_Global(char *name, double *value)
{
    int             i,
                    l;
    char           *h;

    PTHREAD_LOCK_EV();
    *value = 0.;
    if (name == NULL)
        return (0);
    l = strlen(name) + 2;
    h = (char *) malloc((size_t) l * sizeof(char));
    memset(h, 0, l);
    h[0] = '_';
    for (i = 0; i < l - 2; i++)
        h[i + 1] = name[i];

    for (i = 0; i < MAXXVARS; i++)
        if (*VArs_global[i].name && !strcmp(name, VArs_global[i].name)) {
            *value = VArs_global[i].value;
            PTHREAD_UNLOCK_EV();
            free(h);
            return (1);
        }

    PTHREAD_UNLOCK_EV();
    free(h);
    return (0);
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    First, it erases any user-defined variable that is called NAME.  Then
    it creates a new variable called NAME and gives it the value VALUE.
    Returns 1 if the value was added, or 0 if there was no more room.
--------------------------------------------------------------------*/
int
SetValue(char *name, double *value, flow_data *flow_interp)
{
    int             i;
    char            c;

    ClearVar(name, flow_interp);
    if (Isalphab(name) == 0)
        return 0;

    if (*name == '@') {
        c = name[1];
        if (c >= 'a' && c <= 'z') {
            flow_interp->VARS2[c - 'a'] = *value;
            return 1;
        }
        if (c >= 'A' && c <= 'Z') {
            flow_interp->VARS2[c - 'A' + 26] = *value;
            return 1;
        }
        return (1);
    }
    if (*name != '_' && *name != 127) {
        for (i = 0; i < MAXXVARS; i++)
            if (!*flow_interp->VARS[i].name) {
                strcpy(flow_interp->VARS[i].name, name);
                flow_interp->VARS[i].name[XVARLEN] = 0;
                flow_interp->VARS[i].value = *value;
                return (1);
            }
    } else {
        PTHREAD_LOCK_EV();
        for (i = 0; i < MAXXVARS; i++)
            if (!*VArs_global[i].name) {
                strcpy(VArs_global[i].name, name);
                VArs_global[i].name[XVARLEN] = 0;
                VArs_global[i].value = *value;
                PTHREAD_UNLOCK_EV();
                return (1);
            }

        PTHREAD_UNLOCK_EV();
    }
    return (0);
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
--------------------------------------------------------------------*/
int
Isalphab(char *name)
{
    int             i;

    for (i = 0; i < (int) strlen(name); i++)
        if (!isalphab(name[i]))
        return 0;
    return 1;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Internal use only
    This function is used to grab the next token from the expression that
    is being evaluated.
--------------------------------------------------------------------*/
void
Parse(expr_eval *eval0)
{
    unsigned char  *t;

    eval0->type = 0;
    t = eval0->token;
    while (iswhite(*(eval0->expression)))
        (eval0->expression)++;
    if (isdelim(*(eval0->expression))) {
        eval0->type = DEL;
        *t++ = *(eval0->expression)++;
    } else if (isnumer(*(eval0->expression))) {
        eval0->type = NUM;
        while (isnumer(*(eval0->expression)))
            *t++ = *(eval0->expression)++;
        if (*(eval0->expression) == 'e' || *(eval0->expression) == 'E') {
            *t++ = *(eval0->expression)++;;
            if (*(eval0->expression) == '+' || *(eval0->expression) == '-')
                *t++ = *(eval0->expression)++;;
            while (isnumer(*(eval0->expression)))
                *t++ = *(eval0->expression)++;
        }
    } else if (isalphab(*(eval0->expression))) {
        eval0->type = XVAR;
        while (isalphab(*(eval0->expression)))
            *t++ = *(eval0->expression)++;
        eval0->token[XVARLEN] = 0;
    } else if (*(eval0->expression)) {
        *t++ = *(eval0->expression)++;
        *t = 0;
        ERR(E_SYNTAX);
    }
    *t = 0;
    while (iswhite(*(eval0->expression)))
        (eval0->expression)++;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Internal use only
    This function handles any variable assignment operations.
    It returns a value of 1 if it is a top-level assignment operation,
    otherwise it returns 0
--------------------------------------------------------------------*/
int
Level1(double *r, expr_eval *eval0, flow_data *flow_interp)
{
    unsigned char   t[XVARLEN + 1];

    if (eval0->type == XVAR)
        if (*(eval0->expression) == '=') {
            strcpy((char *) t, (char *) eval0->token);
            Parse(eval0);
            Parse(eval0);
            if (!*(eval0->token)) {
                ClearVar((char *) t, flow_interp);
                return (1);
            }
            Level2(r, eval0, flow_interp);
            if (!SetValue((char *) t, r, flow_interp))
                ERR(E_MAXXVARS);
            return (1);
        }
    Level2(r, eval0, flow_interp);
    return (0);
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Internal use only
    This function handles any addition and subtraction operations.
--------------------------------------------------------------------*/
void
Level2(double *r, expr_eval *eval0, flow_data *flow_interp)
{
    double          t = 0;
    char            o;

    Level3(r, eval0, flow_interp);
    while ((o = *(eval0->token)) == '+' || o == '-') {
        Parse(eval0);
        Level3(&t, eval0, flow_interp);
        if (o == '+')
            *r = *r + t;
        else if (o == '-')
            *r = *r - t;
    }
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Internal use only
    This function handles any multiplication, division, or modulo.
--------------------------------------------------------------------*/
void
Level3(double *r, expr_eval *eval0, flow_data *flow_interp)
{
    double          t;
    char            o;

    Level4(r, eval0, flow_interp);
    while ((o = *(eval0->token)) == '*' || o == '/' || o == '%') {
        Parse(eval0);
        Level4(&t, eval0, flow_interp);
        if (o == '*')
            *r = *r * t;
        else if (o == '/') {
            if (t == 0)
                ERR(E_DIVZERO);
            *r = *r / t;
        } else if (o == '%') {
            if (t == 0)
                ERR(E_DIVZERO);
            *r = fmod(*r, t);
        }
    }
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Internal use only
    This function handles any "to the power of" operations.
--------------------------------------------------------------------*/
void
Level4(double *r, expr_eval *eval0, flow_data *flow_interp)
{
    double          t;

    Level5(r, eval0, flow_interp);
    if (*(eval0->token) == '^') {
        Parse(eval0);
        Level5(&t, eval0, flow_interp);
        *r = pow(*r, t);
    }
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Internal use only
    This function handles any unary + or - signs.
--------------------------------------------------------------------*/
void
Level5(double *r, expr_eval *eval0, flow_data *flow_interp)
{
    char            o = 0;

    if (*(eval0->token) == '+' || *(eval0->token) == '-') {
        o = *(eval0->token);
        Parse(eval0);
    }
    Level6(r, eval0, flow_interp);
    if (o == '-')
        *r = -*r;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Internal use only
    This function handles any literal numbers, variables, or functions.
--------------------------------------------------------------------*/
void
Level6(double *r, expr_eval *eval0, flow_data *flow_interp)
{
    int             i,
                    j;
    int             n;
    double          a[MAX_F_ARGS];

    if (*(eval0->token) == '(') {
        Parse(eval0);
        if (*(eval0->token) == ')')
            ERR(E_NOARG);
        Level1(r, eval0, flow_interp);
        if (*(eval0->token) != ')')
            ERR(E_UNBALAN);
        Parse(eval0);
    } else {
        if (eval0->type == NUM) {
            *r = (double) atof((char *) eval0->token);
            Parse(eval0);
        } else if (eval0->type == XVAR) {
            if (*(eval0->expression) == '(') {
                for (i = 0; i < _NBFONC; i++)
                    if (!strcmp((char *) eval0->token, Funcs[i].name)) {
                        Parse(eval0);
                        n = 0;
                        do {
                            Parse(eval0);
                            if (*(eval0->token) == ')' ||
                                *(eval0->token) == ',')
                                ERR(E_NOARG);
                            a[n] = 0;
                            Level1(&a[n], eval0, flow_interp);
                            n++;
                        } while (n < MAX_F_ARGS && *(eval0->token) == ',');
                        Parse(eval0);
                        if (n != Funcs[i].args) {
                            strcpy((char *) eval0->token, Funcs[i].name);
                            ERR(E_NUMARGS);
                        }
                        *r = Funcs[i].func(a);
                        return;
                    }
                if (i == _NBFONC) {
                    for (j = 0; j < _NBFONC_B; j++) {
                        if (!strcmp((char *) eval0->token, Funcs_b[j].name)) {
                            Parse(eval0);
                            n = 0;
                            do {
                                Parse(eval0);
                                if (*(eval0->token) == ')' ||
                                    *(eval0->token) == ',')
                                    ERR(E_NOARG);
                                a[n] = 0;
                                Level1(&a[n], eval0, flow_interp);
                                n++;
                            } while (n < MAX_F_ARGS && *(eval0->token) == ',');
                            Parse(eval0);
                            if (n != Funcs_b[j].args) {
                                strcpy((char *) eval0->token, Funcs_b[j].name);
                                ERR(E_NUMARGS);
                            }
                           *r = Eval_Funcs_b(Funcs_b[j].form, a, n,
                                flow_interp);
                            return;
                        }
                   }

                    if (j == _NBFONC_B)
                        ERR(E_BADFUNC);
                }
            } else if (!GetValue((char *) eval0->token, r, flow_interp))
                ERR(E_UNKNOWN);
            Parse(eval0);
        } else
            ERR(E_SYNTAX);
    }
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
--------------------------------------------------------------------*/
double
Eval_Funcs_b(char *form, double *a, int n, flow_data *flow_interp)
{
    int             i,
                    l0,
                    l1,
                    len,
                    iy[MAX_F_ARGS];
    double          x,
                    y[MAX_F_ARGS];
    char           *h,
                   *k;

    l0 = strlen(Vars_F_name[n - 1]) + 10;
    l1 = 0;
    h = (char *) malloc((size_t) sizeof(char));

    for (i = 0; i < n; i++) {
        iy[i] = GetValue(Vars_F_name[i], &y[i], flow_interp);
        k = ch_copy_float(a[i]);
        len = l0 + strlen(k);
        if (len > l1) {
            free(h);
            h = (char *) malloc((size_t) len * sizeof(char));
            l1 = len;
        }
        memset(h, 0, l1);
        sprintf(h, "%s=%s", Vars_F_name[i], k);
        free(k);
        x = convert_float(h, flow_interp);
    }

    Evaluate(form, &x, &i, flow_interp);

    for (i = 0; i < n; i++) {
        if (iy[i] == 1) {
            k = ch_copy_float(y[i]);
            len = l0 + strlen(k);
            if (len > l1) {
                free(h);
                h = (char *) malloc((size_t) len * sizeof(char));
                l1 = len;
            }
            memset(h, 0, l1);
            sprintf(h, "%s=%s", Vars_F_name[i], k);
            free(k);
            convert_float(h, flow_interp);
        }
        else
            ClearVar(Vars_F_name[i], flow_interp);
    }

    free(h);
    return x;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    This function is called to evaluate the expression E and return the
    answer in RESULT.  If the expression was a top-level assignment, a
    value of 1 will be returned in A, otherwise it will contain 0.
    Returns E_OK if the expression is valid, or an eval0->ERROR code.
--------------------------------------------------------------------*/
int
Evaluate(char *e, double *result, int *a, flow_data *flow_interp)
{
    expr_eval       ev;

    if (setjmp(ev.jb))
        return (ev._EERROR);
    ev.expression = (unsigned char *) e;
    ev.ERANC = e;
    *result = 0;
    Parse(&ev);
    if (!*(ev.token)) {
        ev._EERROR = E_EMPTY;
        ev.ERPOS = (int) ev.expression - (int) ev.ERANC - 1;
        strcpy(ev.ERTOK, (char *) ev.token);
        longjmp(ev.jb, 1);
    }
    *a = Level1(result, &ev, flow_interp);
    return (E_OK);
}
/*------------------------------------------------------------------*/








/*--------------------------------------------------------------------
----------------------------------------------------------------------
        COMPLEX EXPRESSION EVALUATOR
----------------------------------------------------------------------
--------------------------------------------------------------------*/

/*--------------------------------------------------------------------
    Erases the user-defined variable that is called 'name' from memory.
    Note that constants are not affected.
    Returns 1 if the variable was found and erased, or 0 if it didn't
    exist.
--------------------------------------------------------------------*/
int
ClearVar_C(char *name, flow_data *flow_interp)
{
    int             i;
    char            c;

    if (*name == '@' && strlen(name) > 1) {
        c = name[1];
        if (c >= 'a' && c <= 'z') {
            flow_interp->VARS2_C[c - 'a'].r = 0.;
            flow_interp->VARS2_C[c - 'a'].i = 0.;
            return 1;
        }
        if (c >= 'A' && c <= 'Z') {
            flow_interp->VARS2_C[c - 'A' + 26].r = 0.;
            flow_interp->VARS2_C[c - 'A' + 26].i = 0.;
            return 1;
        }
        return (0);
    }
    if (*name != '_' && *name != 127) {
        for (i = 0; i < MAXXVARS; i++)
            if (*flow_interp->VARS_C[i].name && !strcmp(name,
                flow_interp->VARS_C[i].name)) {
                *flow_interp->VARS_C[i].name = 0;
                (flow_interp->VARS_C[i].value).r = 0.;
                (flow_interp->VARS_C[i].value).i = 0.;
                return (1);
            }
    } else {
        PTHREAD_LOCK_EV();

        for (i = 0; i < MAXXVARS; i++)
            if (*VArs_global_C[i].name && !strcmp(name, VArs_global_C[i].name)) {
                *VArs_global_C[i].name = 0;
                (VArs_global_C[i].value).r = 0.;
                (VArs_global_C[i].value).i = 0.;
                PTHREAD_UNLOCK_EV();
                return (1);
            }

        PTHREAD_UNLOCK_EV();
    }
    return (0);
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Looks up the specified variable (or constant) known as NAME and
    returns its contents in VALUE.
    First the user-defined variables are searched, then the constants are
    searched.
    Returns 1 if the value was found, or 0 if it wasn't.                         **
--------------------------------------------------------------------*/
int
GetValue_C(char *name, dcomplex *value, flow_data *flow_interp)
{
    int             i;
    char            c;

    if (comp(name, "I") == 1) {
        (*value).r = 0.;
        (*value).i = 1.;
        return 1;
    }

    (*value).r = 0.;
    (*value).i = 1.;
    if (name == NULL)
        return (0);
    if (*name == '@') {
        c = name[1];
        if (c >= 'a' && c <= 'z') {
            *value = flow_interp->VARS2_C[c - 'a'];
            return 1;
        }
        if (c >= 'A' && c <= 'Z') {
            *value = flow_interp->VARS2_C[c - 'A' + 26];
            return 1;
        }
        return (0);
    }
    if (*name != '_' && *name != 127) {
        for (i = 0; i < MAXXVARS; i++)
            if (*flow_interp->VARS_C[i].name &&
                !strcmp(name, flow_interp->VARS_C[i].name)) {
                *value = flow_interp->VARS_C[i].value;
                return (1);
            }
        for (i = 0; *Consts[i].name; i++)
            if (*Consts[i].name && !strcmp(name, Consts[i].name)) {
                (*value).r = Consts[i].value;
                return (1);
            }
    }
    else {
        PTHREAD_LOCK_EV();

        for (i = 0; i < MAXXVARS; i++)
            if (*VArs_global_C[i].name && !strcmp(name, VArs_global_C[i].name)) {
                *value = VArs_global_C[i].value;
                PTHREAD_UNLOCK_EV();
                return (1);
            }

        PTHREAD_UNLOCK_EV();
    }
    return (0);
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
--------------------------------------------------------------------*/
int
GetValue_Global_hidden_C(char *name, dcomplex *value)
{
    int             i,
                    l;
    char           *h;

    PTHREAD_LOCK_EV();
    (*value).r = 0.;
    (*value).i = 1.;
    if (name == NULL)
        return (0);
    l = strlen(name) + 2;
    h = (char *) malloc((size_t) l * sizeof(char));
    memset(h, 0, l);
    h[0] = 127;
    for (i = 0; i < l - 2; i++)
        h[i + 1] = name[i];

    for (i = 0; i < MAXXVARS; i++)
        if (*VArs_global_C[i].name && !strcmp(h, VArs_global_C[i].name)) {
            *value = VArs_global_C[i].value;
            PTHREAD_UNLOCK_EV();
            free(h);
            return (1);
        }
    PTHREAD_UNLOCK_EV();
    free(h);
    return (0);
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
--------------------------------------------------------------------*/
int
GetValue_Global_C(char *name, dcomplex *value)
{
    int             i,
                    l;
    char           *h;

    PTHREAD_LOCK_EV();
    (*value).r = 0.;
    (*value).i = 1.;
    if (name == NULL)
        return (0);
    l = strlen(name) + 2;
    h = (char *) malloc((size_t) l * sizeof(char));
    memset(h, 0, l);
    h[0] = '_';
    for (i = 0; i < l - 2; i++)
        h[i + 1] = name[i];

    for (i = 0; i < MAXXVARS; i++)
        if (*VArs_global_C[i].name && !strcmp(name, VArs_global_C[i].name)) {
            *value = VArs_global_C[i].value;
            PTHREAD_UNLOCK_EV();
            free(h);
            return (1);
        }

    PTHREAD_UNLOCK_EV();
    free(h);
    return (0);
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    First, it erases any user-defined variable that is called NAME.  Then
    it creates a new variable called NAME and gives it the value VALUE.
    Returns 1 if the value was added, or 0 if there was no more room.
--------------------------------------------------------------------*/
int
SetValue_C(char *name, dcomplex *value, flow_data *flow_interp)
{
    int             i;
    char            c;

    ClearVar_C(name, flow_interp);

    if (*name == '@') {
        c = name[1];
        if (c >= 'a' && c <= 'z') {
            flow_interp->VARS2_C[c - 'a'] = *value;
            return 1;
        }
        if (c >= 'A' && c <= 'Z') {
            flow_interp->VARS2_C[c - 'A' + 26] = *value;
            return 1;
        }
        return (1);
    }
    if (*name != '_' && *name != 127) {
        for (i = 0; i < MAXXVARS; i++)
            if (!*flow_interp->VARS_C[i].name) {
                strcpy(flow_interp->VARS_C[i].name, name);
                flow_interp->VARS_C[i].name[XVARLEN] = 0;
                flow_interp->VARS_C[i].value = *value;
                return (1);
            }
    } else {
        PTHREAD_LOCK_EV();
        for (i = 0; i < MAXXVARS; i++)
            if (!*VArs_global_C[i].name) {
                strcpy(VArs_global_C[i].name, name);
                VArs_global_C[i].name[XVARLEN] = 0;
                VArs_global_C[i].value = *value;
                PTHREAD_UNLOCK_EV();
                return (1);
            }

        PTHREAD_UNLOCK_EV();
    }
    return (0);
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Internal use only
    This function is used to grab the next token from the expression that
    is being evaluated.
--------------------------------------------------------------------*/
void
Parse_C(expr_eval *eval0)
{
    unsigned char  *t;

    eval0->type = 0;
    t = eval0->token;
    while (iswhite(*(eval0->expression)))
        (eval0->expression)++;
    if (isdelim(*(eval0->expression))) {
        eval0->type = DEL;
        *t++ = *(eval0->expression)++;
    } else if (isnumer(*(eval0->expression))) {
        eval0->type = NUM;
        while (isnumer(*(eval0->expression)))
            *t++ = *(eval0->expression)++;
        if (*(eval0->expression) == 'e' || *(eval0->expression) == 'E') {
            *t++ = *(eval0->expression)++;;
            if (*(eval0->expression) == '+' || *(eval0->expression) == '-')
                *t++ = *(eval0->expression)++;;
            while (isnumer(*(eval0->expression)))
                *t++ = *(eval0->expression)++;
        }
    } else if (isalphab(*(eval0->expression))) {
        eval0->type = XVAR;
        while (isalphab(*(eval0->expression)))
            *t++ = *(eval0->expression)++;
        eval0->token[XVARLEN] = 0;
    } else if (*(eval0->expression)) {
        *t++ = *(eval0->expression)++;
        *t = 0;
        ERR(E_SYNTAX);
    }
    *t = 0;
    while (iswhite(*(eval0->expression)))
        (eval0->expression)++;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Internal use only
    This function handles any variable assignment operations.
    It returns a value of 1 if it is a top-level assignment operation,
    otherwise it returns 0
--------------------------------------------------------------------*/
int
Level1_C(dcomplex *r, expr_eval *eval0, flow_data *flow_interp)
{
    unsigned char   t[XVARLEN + 1];

    if (eval0->type == XVAR)
        if (*(eval0->expression) == '=') {
            strcpy((char *) t, (char *) eval0->token);
            Parse_C(eval0);
            Parse_C(eval0);
            if (!*(eval0->token)) {
                ClearVar_C((char *) t, flow_interp);
                return (1);
            }
            Level2_C(r, eval0, flow_interp);
            if (!SetValue_C((char *) t, r, flow_interp))
                ERR(E_MAXXVARS);
            return (1);
        }
    Level2_C(r, eval0, flow_interp);
    return (0);
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Internal use only
    This function handles any addition and subtraction operations.
--------------------------------------------------------------------*/
void
Level2_C(dcomplex *r, expr_eval *eval0, flow_data *flow_interp)
{
    dcomplex        t;
    char            o;

    t.r = 0.;
    t.i = 0.;
    Level3_C(r, eval0, flow_interp);
    while ((o = *(eval0->token)) == '+' || o == '-') {
        Parse_C(eval0);
        Level3_C(&t, eval0, flow_interp);
        if (o == '+')
            *r = dCadd(*r, t);
        else if (o == '-')
            *r = dCsub(*r, t);
    }
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Internal use only
    This function handles any multiplication, division, or modulo.
--------------------------------------------------------------------*/
void
Level3_C(dcomplex *r, expr_eval *eval0, flow_data *flow_interp)
{
    dcomplex        t;
    char            o;

    Level4_C(r, eval0, flow_interp);
    while ((o = *(eval0->token)) == '*' || o == '/') {
        Parse_C(eval0);
        Level4_C(&t, eval0, flow_interp);
        if (o == '*')
            *r = dCmul(*r, t);
        else if (o == '/') {
            if (t.r == 0 && t.i == 0)
                ERR(E_DIVZERO);
            *r = dCdiv(*r, t);
        }
    }
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Internal use only
    This function handles any "to the power of" operations.
--------------------------------------------------------------------*/
void
Level4_C(dcomplex *r, expr_eval *eval0, flow_data *flow_interp)
{
    dcomplex        t;

    Level5_C(r, eval0, flow_interp);
    if (*(eval0->token) == '^') {
        Parse_C(eval0);
        Level5_C(&t, eval0, flow_interp);
        *r = dCpow(*r, t);
    }
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Internal use only
    This function handles any unary + or - signs.
--------------------------------------------------------------------*/
void
Level5_C(dcomplex *r, expr_eval *eval0, flow_data *flow_interp)
{
    char            o = 0;

    if (*(eval0->token) == '+' || *(eval0->token) == '-') {
        o = *(eval0->token);
        Parse_C(eval0);
    }
    Level6_C(r, eval0, flow_interp);
    if (o == '-')
        *r = dCnegat(*r);
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Internal use only
    This function handles any literal numbers, variables, or functions.
--------------------------------------------------------------------*/
void
Level6_C(dcomplex *r, expr_eval *eval0, flow_data *flow_interp)
{
    int             i,
                    j;
    int             n;
    dcomplex        a[MAX_F_ARGS];

    if (*(eval0->token) == '(') {
        Parse_C(eval0);
        if (*(eval0->token) == ')')
            ERR(E_NOARG);
        Level1_C(r, eval0, flow_interp);
        if (*(eval0->token) != ')')
            ERR(E_UNBALAN);
        Parse_C(eval0);
    } else {
        if (eval0->type == NUM) {
            (*r).r = (double) atof((char *) eval0->token);
            (*r).i = 0.;
            Parse_C(eval0);
        } else if (eval0->type == XVAR) {
            if (*(eval0->expression) == '(') {
                for (i = 0; i < _NBFONC_C; i++)
                    if (!strcmp((char *) eval0->token, Funcs_C[i].name)) {
                        Parse_C(eval0);
                        n = 0;
                        do {
                            Parse_C(eval0);
                            if (*(eval0->token) == ')' ||
                                *(eval0->token) == ',')
                                ERR(E_NOARG);
                            a[n].r = 0;
                            a[n].i = 0;
                            Level1_C(&a[n], eval0, flow_interp);
                            n++;
                        } while (n < MAX_F_ARGS && *(eval0->token) == ',');
                        Parse_C(eval0);
                        if (n != Funcs_C[i].args) {
                            strcpy((char *) eval0->token, Funcs_C[i].name);
                            ERR(E_NUMARGS);
                        }
                        (*r) = Funcs_C[i].func(a);
                        return;
                    }
                if (i == _NBFONC_C) {
                    for (j = 0; j < _NBFONC_B; j++) {
                        if (!strcmp((char *) eval0->token, Funcs_b[j].name)) {
                            Parse_C(eval0);
                            n = 0;
                            do {
                                Parse_C(eval0);
                                if (*(eval0->token) == ')' ||
                                    *(eval0->token) == ',')
                                    ERR(E_NOARG);
                                a[n].r = 0;
                                a[n].i = 0;
                                Level1_C(&a[n], eval0, flow_interp);
                                n++;
                            } while (n < MAX_F_ARGS && *(eval0->token) == ',');
                            Parse_C(eval0);
                            if (n != Funcs_b[j].args) {
                                strcpy((char *) eval0->token, Funcs_b[j].name);
                                ERR(E_NUMARGS);
                            }
                           *r = Eval_Funcs_b_C(Funcs_b[j].form, a, n,
                                flow_interp);
                            return;
                        }
                    }

                    if (j == _NBFONC_B)
                        ERR(E_BADFUNC);
                }
            } else if (!GetValue_C((char *) eval0->token, r, flow_interp))
                ERR(E_UNKNOWN);
            Parse_C(eval0);
        } else
            ERR(E_SYNTAX);
    }
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
--------------------------------------------------------------------*/
dcomplex
Eval_Funcs_b_C(char *form, dcomplex *a, int n, flow_data *flow_interp)
{
    int             i,
                    j,
                    l0,
                    l1,
                    len,
                    len_b,
                    iy[MAX_F_ARGS];
    dcomplex        x,
                    z,
                    y[MAX_F_ARGS];
    char           *h,
                   *kr,
                   *ki;

    l0 = strlen(Vars_F_name[n - 1]) + 10;
    l1 = 0;
    h = (char *) malloc((size_t) sizeof(char));

    for (i = 0; i < n; i++) {
        iy[i] = GetValue_C(Vars_F_name[i], &y[i], flow_interp);
        kr = ch_copy_float(a[i].r);
        len = l0 + strlen(kr);
        ki = ch_copy_float(a[i].i);
        len_b = l0 + strlen(ki);
        if (len_b > len)
            len = len_b;
        if (len > l1) {
            free(h);
            h = (char *) malloc((size_t) len * sizeof(char));
            l1 = len;
        }
        memset(h, 0, l1);
        sprintf(h, "%s=%s+I*%s", Vars_F_name[i], kr, ki);
        free(kr);
        free(ki);
        Evaluate_C(h, &x, &j, flow_interp);
    }

    Evaluate_C(form, &z, &j, flow_interp);

    for (i = 0; i < n; i++) {
        if (iy[i] == 1) {
            kr = ch_copy_float(y[i].r);
            len = l0 + strlen(kr);
            ki = ch_copy_float(y[i].i);
            len_b = l0 + strlen(ki);
            if (len_b > len)
                len = len_b;
            if (len > l1) {
                free(h);
                h = (char *) malloc((size_t) len * sizeof(char));
                l1 = len;
            }
            memset(h, 0, l1);
            sprintf(h, "%s=%s+I*%s", Vars_F_name[i], kr, ki);
            free(kr);
            free(ki);
            Evaluate_C(h, &x, &j, flow_interp);
        }
        else
            ClearVar_C(Vars_F_name[i], flow_interp);
    }

    free(h);
    return z;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    This function is called to evaluate the expression E and return the
    answer in RESULT.  If the expression was a top-level assignment, a
    value of 1 will be returned in A, otherwise it will contain 0.
    Returns E_OK if the expression is valid, or an eval0->ERROR code.
--------------------------------------------------------------------*/
int
Evaluate_C(char *e, dcomplex *result, int *a, flow_data *flow_interp)
{
    expr_eval       ev;

    if (setjmp(ev.jb))
        return (ev._EERROR);
    ev.expression = (unsigned char *) e;
    ev.ERANC = e;
    (*result).r = 0;
    (*result).i = 0;
    Parse_C(&ev);
    if (!*(ev.token)) {
        ev._EERROR = E_EMPTY;
        ev.ERPOS = (int) ev.expression - (int) ev.ERANC - 1;
        strcpy(ev.ERTOK, (char *) ev.token);
        longjmp(ev.jb, 1);
    }
    *a = Level1_C(result, &ev, flow_interp);
    return (E_OK);
}
/*------------------------------------------------------------------*/









/*--------------------------------------------------------------------
----------------------------------------------------------------------
        GENERIC EXPRESSION EVALUATOR
----------------------------------------------------------------------
--------------------------------------------------------------------*/

/*--------------------------------------------------------------------
    Erases the user-defined variable that is called 'name' from memory.
    Note that constants are not affected.
    Returns 1 if the variable was found and erased, or 0 if it didn't
    exist.
--------------------------------------------------------------------*/
int
ClearVar_Gen(int n, char *name, flow_data *flow_interp)
{
    int             i;
    char            c;

    if (*name == '@' && strlen(name) > 1) {
        c = name[1];
        if (c >= 'a' && c <= 'z') {
            Expreval_ops[n].Zero(flow_interp->VARS2_Gen[n][c - 'a'].value[0]);
            return 1;
        }
        if (c >= 'A' && c <= 'Z') {
            Expreval_ops[n].Zero(
            flow_interp->VARS2_Gen[n][c - 'A' + 26].value[0]);
            return 1;
        }
        return (0);
    }
    if (*name != '_' && *name != 127) {
        for (i = 0; i < MAXXVARS; i++)
            if (*flow_interp->VARS_Gen[n][i].name && !strcmp(name,
                flow_interp->VARS_Gen[n][i].name)) {
                *flow_interp->VARS_Gen[n][i].name = 0;
                Expreval_ops[n].Zero(flow_interp->VARS_Gen[n][i].value[0]);
                return (1);
            }
    } else {
        PTHREAD_LOCK_EV();

        for (i = 0; i < MAXXVARS; i++)
            if (*VArs_global_Gen[n][i].name &&
                !strcmp(name, VArs_global_Gen[n][i].name)) {
                *VArs_global_Gen[n][i].name = 0;
                Expreval_ops[n].Zero(VArs_global_Gen[n][i].value[0]);
                PTHREAD_UNLOCK_EV();
                return (1);
            }

        PTHREAD_UNLOCK_EV();
    }
    return (0);
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Looks up the specified variable (or constant) known as NAME and
    returns its contents in VALUE.
    First the user-defined variables are searched, then the constants are
    searched.
    Returns 1 if the value was found, or 0 if it wasn't.                         **
--------------------------------------------------------------------*/
int
GetValue_Gen(int n, char *name, void **value, flow_data *flow_interp)
{
    int             i;
    char            c;

    if (*name == '@') {
        c = name[1];
        if (c >= 'a' && c <= 'z') {
            Expreval_ops[n].clear(value[0]);
            *value = Expreval_ops[n].copy(
                flow_interp->VARS2_Gen[n][c - 'a'].value[0]);
            return 1;
        }
        if (c >= 'A' && c <= 'Z') {
            Expreval_ops[n].clear(value[0]);
            *value = Expreval_ops[n].copy(
                flow_interp->VARS2_Gen[n][c - 'A' + 26].value[0]);
            return 1;
        }
        return (0);
    }
    if (*name != '_' && *name != 127) {
        for (i = 0; i < MAXXVARS; i++)
            if (*flow_interp->VARS_Gen[n][i].name &&
                !strcmp(name, flow_interp->VARS_Gen[n][i].name)) {
                Expreval_ops[n].clear(value[0]);
                *value = Expreval_ops[n].copy(
                    flow_interp->VARS_Gen[n][i].value[0]);
                return (1);
            }
    }
    else {
        PTHREAD_LOCK_EV();

        for (i = 0; i < MAXXVARS; i++)
            if (*VArs_global_Gen[n][i].name && !strcmp(name,
                VArs_global_Gen[n][i].name)) {
                Expreval_ops[n].clear(value[0]);
                *value = Expreval_ops[n].copy(VArs_global_Gen[n][i].value[0]);
                PTHREAD_UNLOCK_EV();
                return (1);
            }

        PTHREAD_UNLOCK_EV();
    }
    return (0);
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
--------------------------------------------------------------------*/
int
GetValue_Global_hidden_Gen(int n, char *name, void **value)
{
    int             i,
                    l;
    char           *h;

    PTHREAD_LOCK_EV();
    if (name == NULL)
        return (0);
    l = strlen(name) + 2;
    h = (char *) malloc((size_t) l * sizeof(char));
    memset(h, 0, l);
    h[0] = 127;
    for (i = 0; i < l - 2; i++)
        h[i + 1] = name[i];

    for (i = 0; i < MAXXVARS; i++)
        if (*VArs_global_Gen[n][i].name &&
            !strcmp(h, VArs_global_Gen[n][i].name)) {
        Expreval_ops[n].clear(value[0]);
            *value = Expreval_ops[n].copy(VArs_global_Gen[n][i].value);
            PTHREAD_UNLOCK_EV();
            free(h);
            return (1);
        }
    PTHREAD_UNLOCK_EV();
    free(h);
    return (0);
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
--------------------------------------------------------------------*/
int
GetValue_Global_Gen(int n, char *name, void **value)
{
    int             i,
                    l;
    char           *h;

    PTHREAD_LOCK_EV();

    if (name == NULL)
        return (0);
    l = strlen(name) + 2;
    h = (char *) malloc((size_t) l * sizeof(char));
    memset(h, 0, l);
    h[0] = '_';
    for (i = 0; i < l - 2; i++)
        h[i + 1] = name[i];

    for (i = 0; i < MAXXVARS; i++)
        if (*VArs_global_Gen[n][i].name &&
            !strcmp(name, VArs_global_Gen[n][i].name)) {
        Expreval_ops[n].clear(value[0]);
            *value = Expreval_ops[n].copy(VArs_global_Gen[n][i].value);
            PTHREAD_UNLOCK_EV();
            free(h);
            return (1);
        }

    PTHREAD_UNLOCK_EV();
    free(h);
    return (0);
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    First, it erases any user-defined variable that is called NAME.  Then
    it creates a new variable called NAME and gives it the value VALUE.
    Returns 1 if the value was added, or 0 if there was no more room.
--------------------------------------------------------------------*/
int
SetValue_Gen(int n, char *name, void **value, flow_data *flow_interp)
{
    int             i;
    char            c;

    ClearVar_Gen(n, name, flow_interp);

    if (*name == '@') {
        c = name[1];
        if (c >= 'a' && c <= 'z') {
            Expreval_ops[n].clear(flow_interp->VARS2_Gen[n][c - 'a'].value[0]);
            flow_interp->VARS2_Gen[n][c - 'a'].value[0] =
                Expreval_ops[n].copy(*value);
            return 1;
        }
        if (c >= 'A' && c <= 'Z') {
            Expreval_ops[n].clear(
            flow_interp->VARS2_Gen[n][c - 'A' + 26].value[0]);
            flow_interp->VARS2_Gen[n][c - 'A' + 26].value[0] =
                Expreval_ops[n].copy(*value);
            return 1;
        }
        return (1);
    }
    if (*name != '_' && *name != 127) {
        for (i = 0; i < MAXXVARS; i++)
            if (!*flow_interp->VARS_Gen[n][i].name) {
                strcpy(flow_interp->VARS_Gen[n][i].name, name);
                flow_interp->VARS_Gen[n][i].name[XVARLEN] = 0;
                Expreval_ops[n].clear(flow_interp->VARS_Gen[n][i].value[0]);
                flow_interp->VARS_Gen[n][i].value[0] =
                    Expreval_ops[n].copy(*value);
                return (1);
            }
    } else {
        PTHREAD_LOCK_EV();
        for (i = 0; i < MAXXVARS; i++)
            if (!*VArs_global_Gen[n][i].name) {
                strcpy(VArs_global_Gen[n][i].name, name);
                VArs_global_Gen[n][i].name[XVARLEN] = 0;
                Expreval_ops[n].clear(VArs_global_Gen[n][i].value[0]);
                VArs_global_Gen[n][i].value[0] = Expreval_ops[n].copy(*value);
                PTHREAD_UNLOCK_EV();
                return (1);
            }

        PTHREAD_UNLOCK_EV();
    }
    return (0);
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Internal use only
    This function is used to grab the next token from the expression that
    is being evaluated.
--------------------------------------------------------------------*/
void
Parse_Gen(expr_eval *eval0)
{
    unsigned char  *t;

    eval0->type = 0;
    t = eval0->token;
    while (iswhite(*(eval0->expression)))
        (eval0->expression)++;
    if (isdelim(*(eval0->expression))) {
        eval0->type = DEL;
        *t++ = *(eval0->expression)++;
    } else if (isnumer(*(eval0->expression))) {
        eval0->type = NUM;
        while (isnumer(*(eval0->expression)))
            *t++ = *(eval0->expression)++;
        if (*(eval0->expression) == 'e' || *(eval0->expression) == 'E') {
            *t++ = *(eval0->expression)++;;
            if (*(eval0->expression) == '+' || *(eval0->expression) == '-')
                *t++ = *(eval0->expression)++;;
            while (isnumer(*(eval0->expression)))
                *t++ = *(eval0->expression)++;
        }
    } else if (isalphab(*(eval0->expression))) {
        eval0->type = XVAR;
        while (isalphab(*(eval0->expression)))
            *t++ = *(eval0->expression)++;
        eval0->token[XVARLEN] = 0;
    } else if (*(eval0->expression)) {
        *t++ = *(eval0->expression)++;
        *t = 0;
        ERR(E_SYNTAX);
    }
    *t = 0;
    while (iswhite(*(eval0->expression)))
        (eval0->expression)++;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Internal use only
    This function handles any variable assignment operations.
    It returns a value of 1 if it is a top-level assignment operation,
    otherwise it returns 0
--------------------------------------------------------------------*/
int
Level1_Gen(int n, void **r, expr_eval *eval0, flow_data *flow_interp)
{
    unsigned char   t[XVARLEN + 1];

    if (eval0->type == XVAR)
        if (*(eval0->expression) == '=') {
            strcpy((char *) t, (char *) eval0->token);
            Parse_Gen(eval0);
            Parse_Gen(eval0);
            if (!*(eval0->token)) {
                ClearVar_Gen(n, (char *) t, flow_interp);
                return (1);
            }
            Level2_Gen(n, r, eval0, flow_interp);
            if (!SetValue_Gen(n, (char *) t, r, flow_interp))
                ERR(E_MAXXVARS);
            return (1);
        }
    Level2_Gen(n, r, eval0, flow_interp);
    return (0);
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Internal use only
    This function handles any addition and subtraction operations.
--------------------------------------------------------------------*/
void
Level2_Gen(int n, void **r, expr_eval *eval0, flow_data *flow_interp)
{
    void          **t;
    char            o;

    t = (void **) malloc((size_t) sizeof(void *));
    Expreval_ops[n].Init(t);
    Expreval_ops[n].Zero(*t);
    Level3_Gen(n, r, eval0, flow_interp);
    while ((o = *(eval0->token)) == '+' || o == '-') {
        Parse_Gen(eval0);
        Level3_Gen(n, t, eval0, flow_interp);
        if (o == '+')
            (Expreval_ops[n].add)(*r, *r, *t);
        else if (o == '-')
            (Expreval_ops[n].sub)(*r, *r, *t);
    }
    Expreval_ops[n].clear(*t);
    free(t);
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Internal use only
    This function handles any multiplication, division, or modulo.
--------------------------------------------------------------------*/
void
Level3_Gen(int n, void **r, expr_eval *eval0, flow_data *flow_interp)
{
    void          **t;
    char            o;

    t = (void **) malloc((size_t) sizeof(void *));
    Expreval_ops[n].Init(t);
    Expreval_ops[n].Zero(*t);
    Level4_Gen(n, r, eval0, flow_interp);
    while ((o = *(eval0->token)) == '*' || o == '/') {
        Parse_Gen(eval0);
        Level4_Gen(n, t, eval0, flow_interp);
        if (o == '*')
            (Expreval_ops[n].mul)(*r, *r, *t);
        else if (o == '/') {
            if (Expreval_ops[n].iszero(*t) == 1)
                ERR(E_DIVZERO);
            (Expreval_ops[n].div)(*r, *r, *t);
        }
    }
    Expreval_ops[n].clear(*t);
    free(t);
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Internal use only
    This function handles any "to the power of" operations.
--------------------------------------------------------------------*/
void
Level4_Gen(int n, void **r, expr_eval *eval0, flow_data *flow_interp)
{
    void          **t;

    t = (void **) malloc((size_t) sizeof(void *));
    Expreval_ops[n].Init(t);
    Expreval_ops[n].Zero(*t);
    Level5_Gen(n, r, eval0, flow_interp);
    if (*(eval0->token) == '^') {
        Parse_Gen(eval0);
        Level5_Gen(n, t, eval0, flow_interp);
        (Expreval_ops[n].pow)(*r, *r, *t);
    }
    Expreval_ops[n].clear(*t);
    free(t);
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Internal use only
    This function handles any unary + or - signs.
--------------------------------------------------------------------*/
void
Level5_Gen(int n, void **r, expr_eval *eval0, flow_data *flow_interp)
{
    char            o = 0;

    if (*(eval0->token) == '+' || *(eval0->token) == '-') {
        o = *(eval0->token);
        Parse_Gen(eval0);
    }
    Level6_Gen(n, r, eval0, flow_interp);
    if (o == '-')
        (Expreval_ops[n].neg)(*r);
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Internal use only
    This function handles any literal numbers, variables, or functions.
--------------------------------------------------------------------*/
void
Level6_Gen(int n, void **r, expr_eval *eval0, flow_data *flow_interp)
{
    int             i,
                    j,
                    k,
                    m;
    void           *a[MAX_F_ARGS];

    if (*(eval0->token) == '(') {
        Parse_Gen(eval0);
        if (*(eval0->token) == ')')
            ERR(E_NOARG);
        Level1_Gen(n, r, eval0, flow_interp);
        if (*(eval0->token) != ')')
            ERR(E_UNBALAN);
        Parse_Gen(eval0);
    } else {
        if (eval0->type == NUM) {
            (Expreval_ops[n].set)(*r, (char *) eval0->token);
            Parse_Gen(eval0);
        } else if (eval0->type == XVAR) {
            if (*(eval0->expression) == '(') {
                for (i = 0; i < _NBFONC_Gen[n]; i++)
                    if (!strcmp((char *) eval0->token,
                        Funcs_Gen[n][i].name)) {
                        Parse_Gen(eval0);
                        m = 0;
                        do {
                            Parse_Gen(eval0);
                            if (*(eval0->token) == ')' ||
                                *(eval0->token) == ',')
                                ERR(E_NOARG);
                            Expreval_ops[n].Init(&a[m]);
                            Expreval_ops[n].Zero(a[m]);
                            Level1_Gen(n, &a[m], eval0, flow_interp);
                            m++;
                        } while (m < MAX_F_ARGS && *(eval0->token) == ',');
                        Parse_Gen(eval0);
                        if (m != Funcs_Gen[n][i].args) {
                            strcpy((char *) eval0->token,
                                Funcs_Gen[n][i].name);
                            ERR(E_NUMARGS);
                        }
                        Funcs_Gen[n][i].func(r, a);
                        for (j = 0; j < m; j++)
                            Expreval_ops[n].clear(a[j]);
                        return;
                    }

                if (i == _NBFONC_Gen[n]) {
                    for (j = 0; j < _NBFONC_B; j++) {
                        if (!strcmp((char *) eval0->token, Funcs_b[j].name)) {
                            Parse_Gen(eval0);
                            m = 0;
                            do {
                                Parse_Gen(eval0);
                                if (*(eval0->token) == ')' ||
                                    *(eval0->token) == ',')
                                    ERR(E_NOARG);
                                Expreval_ops[n].Init(&a[m]);
                                Expreval_ops[n].Zero(a[m]);
                                Level1_Gen(n, &a[m], eval0, flow_interp);
                                m++;
                            } while (m < MAX_F_ARGS && *(eval0->token) == ',');
                            Parse_Gen(eval0);
                            if (m != Funcs_b[j].args) {
                                strcpy((char *) eval0->token, Funcs_b[j].name);
                                ERR(E_NUMARGS);
                            }
                            Eval_Funcs_b_Gen(n, r, Funcs_b[j].form, a, m,
                                flow_interp);
                            for (k = 0; k < m; k++)
                                Expreval_ops[n].clear(a[k]);
                            return;
                        }
                    }

                    if (j == _NBFONC_B)
                        ERR(E_BADFUNC);
                }
            } else if (!GetValue_Gen(n, (char *) eval0->token, r,
                flow_interp))
                ERR(E_UNKNOWN);
            Parse_Gen(eval0);
        } else
            ERR(E_SYNTAX);
    }
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
--------------------------------------------------------------------*/
void
Eval_Funcs_b_Gen(int n, void **r, char *form, void **a, int m, flow_data
    *flow_interp)
{
    int             i,
                    j,
                    l0,
                    l1,
                    len;
    void           *x,
                   *y,
                   *yy[MAX_F_ARGS];
    char           *h,
                   *k;

    l0 = strlen(Vars_F_name[m - 1]) + 10;
    l1 = 0;
    h = (char *) malloc((size_t) sizeof(char));

    for (i = 0; i < m; i++) {
        y = GetValue_Gen_0(n, Vars_F_name[i], flow_interp);
        yy[i] = NULL;
        if (y != NULL) {
            yy[i] = Expreval_ops[n].copy(y);
            k = Expreval_ops[n].print(y);
            free(k);
        }
        k = Expreval_ops[n].print(a[i]);
        len = l0 + strlen(k);
        if (len > l1) {
            free(h);
            h = (char *) malloc((size_t) len * sizeof(char));
            l1 = len;
        }
        memset(h, 0, l1);
        sprintf(h, "%s=%s", Vars_F_name[i], k);
        free(k);
        Evaluate_Gen(n, h, &x, &j, flow_interp);
    }

    Evaluate_Gen(n, form, r, &j, flow_interp);

    for (i = 0; i < m; i++) {
        if (yy[i] != NULL) {
            k = Expreval_ops[n].print(yy[i]);
            len = l0 + strlen(k);
            if (len > l1) {
                free(h);
                h = (char *) malloc((size_t) len * sizeof(char));
                l1 = len;
            }
            memset(h, 0, l1);
            sprintf(h, "%s=%s", Vars_F_name[i], k);
            free(k);
            Expreval_ops[n].clear(yy[i]);
            Evaluate_Gen(n, h, &x, &j, flow_interp);
        }
        else
            ClearVar_Gen(n, Vars_F_name[i], flow_interp);
    }

    free(h);
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    This function is called to evaluate the expression E and return the
    answer in RESULT.  If the expression was a top-level assignment, a
    value of 1 will be returned in A, otherwise it will contain 0.
    Returns E_OK if the expression is valid, or an eval0->ERROR code.
--------------------------------------------------------------------*/
int
Evaluate_Gen(int n, char *e, void **result, int *a, flow_data *flow_interp)
{
    expr_eval       ev;

    if (setjmp(ev.jb))
        return (ev._EERROR);
    ev.expression = (unsigned char *) e;
    ev.ERANC = e;
    Expreval_ops[n].Init(result);
    Expreval_ops[n].Zero(*result);
    Parse_Gen(&ev);
    if (!*(ev.token)) {
        ev._EERROR = E_EMPTY;
        ev.ERPOS = (int) ev.expression - (int) ev.ERANC - 1;
        strcpy(ev.ERTOK, (char *) ev.token);
        longjmp(ev.jb, 1);
    }
    *a = Level1_Gen(n, result, &ev, flow_interp);
    return (E_OK);
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
--------------------------------------------------------------------*/
void *
GetValue_Gen_0(int n, char *name, flow_data *flow_interp)
{
    int             i;
    char            c;
    void           *d;

    d = NULL;
    if (*name == '@') {
        c = name[1];
        if (c >= 'a' && c <= 'z') {
            return flow_interp->VARS2_Gen[n][c - 'a'].value[0];
        }
        if (c >= 'A' && c <= 'Z') {
            return flow_interp->VARS2_Gen[n][c - 'A' + 26].value[0];
        }
    }
    if (*name != '_' && *name != 127) {
        for (i = 0; i < MAXXVARS; i++)
            if (*flow_interp->VARS_Gen[n][i].name &&
                !strcmp(name, flow_interp->VARS_Gen[n][i].name)) {
                return flow_interp->VARS_Gen[n][i].value[0];
            }
    }
    else {
        PTHREAD_LOCK_EV();

        for (i = 0; i < MAXXVARS; i++)
            if (*VArs_global_Gen[n][i].name && !strcmp(name,
                VArs_global_Gen[n][i].name)) {
                d = VArs_global_Gen[n][i].value[0];
                PTHREAD_UNLOCK_EV();
                return d;
            }

        PTHREAD_UNLOCK_EV();
    }
    return d;
}
/*------------------------------------------------------------------*/
