/*
 * Copyright (c) 1998 by David I. Bell
 * Permission is granted to use, distribute, or modify this source,
 * provided that this copyright notice remains intact.
 *
 * Utility routines.
 *
 * Modified by Radovan Garabik
 */

#include "fosh.h"

#include <sys/types.h>
#include <sys/stat.h>
#include <dirent.h>
#include <utime.h>


typedef	struct	chunk	CHUNK;
#define	CHUNKINITSIZE	4

struct	chunk	{
	CHUNK *	next;
	char	data[CHUNKINITSIZE];	/* actually of varying length */
};


static	CHUNK *	chunklist;







/*
 * Return TRUE if a filename is a directory.
 * Nonexistant files return FALSE.
 */
BOOL
isadir(name)
	const char *	name;
{
	struct	stat	statbuf;

	if (stat(name, &statbuf) < 0)
		return FALSE;

	return S_ISDIR(statbuf.st_mode);
}




/*
 * Expand the wildcards in a filename, if any.
 * Returns an argument list with matching filenames in sorted order.
 * The expanded names are stored in memory chunks which can later all
 * be freed at once.  Returns zero if the name is not a wildcard, or
 * returns the count of matched files if the name is a wildcard and
 * there was at least one match, or returns -1 if either no filenames
 * matched or too many filenames matched (with an error output).
 */
int
expandwildcards(name, maxargc, retargv)
	const char *	name;
	int		maxargc;
	const char *	retargv[];
{
	const char *	last;
	const char *	cp1;
	const char *	cp2;
	const char *	cp3;
	char *		str;
	DIR *		dirp;
	struct dirent *	dp;
	int		dirlen;
	int		matches;
	char		dirname[PATHLEN];

	last = strrchr(name, '/');

	if (last)
		last++;
	else
		last = name;

	cp1 = strchr(name, '*');
	cp2 = strchr(name, '?');
	cp3 = strchr(name, '[');

	if ((cp1 == NULL) && (cp2 == NULL) && (cp3 == NULL))
		return 0;

	if ((cp1 && (cp1 < last)) || (cp2 && (cp2 < last)) ||
		(cp3 && (cp3 < last)))
	{
		fprintf(stderr, "Wildcards only implemented for last filename component\n");

		return -1;
	}

	dirname[0] = '.';
	dirname[1] = '\0';

	if (last != name) {
		memcpy(dirname, name, last - name);
		dirname[last - name - 1] = '\0';

		if (dirname[0] == '\0') {
			dirname[0] = '/';
			dirname[1] = '\0';
		}
	}

	dirp = opendir(dirname);

	if (dirp == NULL) {
		perror(dirname);

		return -1;
	}

	dirlen = strlen(dirname);

	if (last == name) {
		dirlen = 0;
		dirname[0] = '\0';
	} else if (dirname[dirlen - 1] != '/') {
		dirname[dirlen++] = '/';
		dirname[dirlen] = '\0';
	}

	matches = 0;

	while ((dp = readdir(dirp)) != NULL) {
		if ((strcmp(dp->d_name, ".") == 0) ||
			(strcmp(dp->d_name, "..") == 0))
		{
			continue;
		}

		if (!match(dp->d_name, last))
			continue;

		if (matches >= maxargc) {
			fprintf(stderr, "Too many filename matches\n");
			closedir(dirp);

			return -1;
		}

		str = getchunk(dirlen + strlen(dp->d_name) + 1);

		if (str == NULL) {
			fprintf(stderr, "No memory for filename\n");
			closedir(dirp);

			return -1;
		}

		if (dirlen)
			memcpy(str, dirname, dirlen);

		strcpy(str + dirlen, dp->d_name);

		retargv[matches++] = str;
	}

	closedir(dirp);

	if (matches == 0) {
		fprintf(stderr, "No matches\n");

		return -1;
	}

	qsort((void *) retargv, matches, sizeof(char *), namesort);

	return matches;
}


/*
 * Sort routine for list of filenames.
 */
int
namesort(p1, p2)
	const void *	p1;
	const void *	p2;
{
	const char **	s1;
	const char **	s2;

	s1 = (const char **) p1;
	s2 = (const char **) p2;

	return strcmp(*s1, *s2);
}


/*
 * Routine to see if a text string is matched by a wildcard pattern.
 * Returns TRUE if the text is matched, or FALSE if it is not matched
 * or if the pattern is invalid.
 *  *		matches zero or more characters
 *  ?		matches a single character
 *  [abc]	matches 'a', 'b' or 'c'
 *  \c		quotes character c
 *  Adapted from code written by Ingo Wilken.
 */
BOOL
match(text, pattern)
	const char *	text;
	const char *	pattern;
{
	const char *	retrypat;
	const char *	retrytxt;
	int		ch;
	BOOL		found;

	retrypat = NULL;
	retrytxt = NULL;

	while (*text || *pattern) {
		ch = *pattern++;

		switch (ch) {
			case '*':  
				retrypat = pattern;
				retrytxt = text;
				break;

			case '[':  
				found = FALSE;

				while ((ch = *pattern++) != ']') {
					if (ch == '\\')
						ch = *pattern++;

					if (ch == '\0')
						return FALSE;

					if (*text == ch)
						found = TRUE;
				}

				if (!found) {
					pattern = retrypat;
					text = ++retrytxt;
				}

				/* fall into next case */

			case '?':  
				if (*text++ == '\0')
					return FALSE;

				break;

			case '\\':  
				ch = *pattern++;

				if (ch == '\0')
					return FALSE;

				/* fall into next case */

			default:        
				if (*text == ch) {
					if (*text)
						text++;
					break;
				}

				if (*text) {
					pattern = retrypat;
					text = ++retrytxt;
					break;
				}

				return FALSE;
		}

		if (pattern == NULL)
			return FALSE;
	}

	return TRUE;
}


/*
 * Take a command string, and break it up into an argc, argv list.
 * The returned argument list and strings are in static memory, and so
 * are overwritten on each call.  The argument array is ended with an
 * extra NULL pointer for convenience.  Returns TRUE if successful,
 * or FALSE on an error with a message already output.
 */
BOOL
makeargs(cmd, argcptr, argvptr)
	const char *	cmd;
	int *		argcptr;
	const char ***	argvptr;
{
	char *			cp;
	int			argc;
	static char		strings[CMDLEN+1];
	static const char *	argtable[MAXARGS+1];

	/*
	 * Copy the command string and then break it apart
	 * into separate arguments.
	 */
	strcpy(strings, cmd);
	argc = 0;
	cp = strings;

	while (*cp) {
		if (argc >= MAXARGS) {
			fprintf(stderr, "Too many arguments\n");

			return FALSE;
		}

		argtable[argc++] = (const char *) cp;

		while (*cp && !isblank(*cp))
			cp++;

		while (isblank(*cp))
 			*cp++ = '\0';
	}

	argtable[argc] = NULL;

	*argcptr = argc;
	*argvptr = argtable;

 	return TRUE;
}


/*
 * Make a NULL-terminated string out of an argc, argv pair.
 * Returns TRUE if successful, or FALSE if the string is too long,
 * with an error message given.  This does not handle spaces within
 * arguments correctly.
 */
BOOL
makestring(argc, argv, buf, buflen)
	int		argc;
	const char **	argv;
	char *		buf;
	int		buflen;
{
	int	len;

	while (argc-- > 0) {
		len = strlen(*argv);

		if (len >= buflen) {
			fprintf(stderr, "Argument string too long\n");

			return FALSE;
		}

		strcpy(buf, *argv++);

		buf += len;
		buflen -= len;

		if (argc)
			*buf++ = ' ';

		buflen--; 
	}

	*buf = '\0';

	return TRUE;
}


/*
 * Allocate a chunk of memory (like malloc).
 * The difference, though, is that the memory allocated is put on a
 * list of chunks which can be freed all at one time.  You CAN NOT free
 * an individual chunk.
 */
char *
getchunk(size)
	int	size;
{
	CHUNK *	chunk;

	if (size < CHUNKINITSIZE)
		size = CHUNKINITSIZE;

	chunk = (CHUNK *) malloc(size + sizeof(CHUNK) - CHUNKINITSIZE);

	if (chunk == NULL)
		return NULL;

	chunk->next = chunklist;
	chunklist = chunk;

	return chunk->data;
}


/*
 * Duplicate a string value using the chunk allocator.
 * The returned string cannot be individually freed, but can only be freed
 * with other strings when freechunks is called.  Returns NULL on failure.
 */
char *
chunkstrdup(str)
	const char *	str;
{
	int	len;
	char *	newstr;

	len = strlen(str) + 1;
	newstr = getchunk(len);

	if (newstr)
		memcpy(newstr, str, len);

	return newstr;
}


/*
 * Free all chunks of memory that had been allocated since the last
 * call to this routine.
 */
void
freechunks()
{
	CHUNK *	chunk;

	while (chunklist) {
		chunk = chunklist;
		chunklist = chunk->next;
		free((char *) chunk);
	}
}



/* END CODE */
