/*
 * Copyright (c) 1998 by David I. Bell
 * Permission is granted to use, distribute, or modify this source,
 * provided that this copyright notice remains intact.
 *
 * modified by Radovan Garabik
 */

#include <wait.h>
#include <signal.h>
#include <errno.h>
#include "getline/getline.h"
#include "fosh.h"

static const char * const	version = "1.0";

typedef struct {
	char *	name;
	char *	usage;
	void	(*func) PROTO((int argc, const char ** argv));
	int	minargs;
	int	maxargs;
} CMDTAB;


static const CMDTAB	cmdtab[] =
{
	{"alias",	"[name [command]]", 	do_alias,
	1,		MAXARGS},

	{"cd",		"[dirname]",		do_cd,
	1,		2},

	{"echo",	"[args] ...",		do_echo,
	1,		MAXARGS},

	{"exec",		"filename [args]",	do_exec,
	2,		MAXARGS},

	{"exit",		"",			do_exit,
	1,		1},


	{"kill",	"[-sig] pid ...",	do_kill,
	2,		MAXARGS},


	{"prompt",	"string",		do_prompt,
	2,		MAXARGS},

	{"pwd",		"",			do_pwd,
	1,		1},

	{"quit",		"",			do_exit,
	1,		1},


	{"setenv",	"name value",		do_setenv,
	3,		3},

	{".",	"filename",		do_source,
	2,		2},

	{"unalias",	"name",			do_unalias,
	2,		2},

	{"where",	"program",		do_where,
	2,		2},

	{NULL,		0,			0,
	0,		0}
};


typedef struct {
	char *	name;
	char *	value;
} ALIAS;


/*
 * Local data.
 */
static	ALIAS *	aliastable;
static	int	aliascount;

static	FILE *	sourcefiles[MAXSOURCE];
static	int	sourcecount;

static	BOOL	intcrlf = TRUE;
static	char *	prompt;


/*
 * Local procedures.
 */
static	void	catchint PROTO((int));
static	void	catchquit PROTO((int));
static	void	readfile PROTO((const char * name));
static	void	command PROTO((const char * cmd));
static	void	runcmd PROTO((const char * cmd, int argc, const char ** argv));
static	void	usage PROTO((void));
static	BOOL	trybuiltin PROTO((int argc, const char ** argv));
static	ALIAS *	findalias PROTO((const char * name));


/*
 * Global interrupt flag.
 */
BOOL	intflag;


int
main(argc, argv)
	int		argc;
	const char **	argv;
{
	const char *	cp;
	const char *	singleCommand;
	char		buf[PATHLEN];

	singleCommand = NULL;

	/*
	 * Look for options.
	 */
	argv++;
	argc--;
	while ((argc > 0) && (**argv == '-'))
	{
		cp = *argv++ + 1;
		argc--;

		while (*cp) switch (*cp++)
		{
			case 'c':
				/*
				 * Execute specified command.
				 */
				if ((argc != 1) || singleCommand)
					usage();

				singleCommand = *argv++;
				argc--;

				break;

				 
			case 'h':
			case '?':
				usage();
				break;

			default:
				fprintf(stderr, "Unknown option -%c\n", cp[-1]);

				return 1;
		}
	}

	/*
	 * No more arguments are allowed.
	 */
	if (argc > 0)
		usage();


	prompt=malloc(5); strcpy(prompt,"PWD ");
	/*
	 * Default our path if it is not set.
	 */
	if (getenv("PATH") == NULL)
		putenv("PATH=/bin:/usr/bin:/sbin:/usr/sbin:");

	/*
	 * If we are to execute a single command, then do so and exit.
	 */
	if (singleCommand)
	{
		command(singleCommand);

		return 0;
	}

	/*
	 * Print a hello message unless we are told to be silent.
	 */
	if (isatty(STDIN))
		printf("fosh %s\n", version);

	signal(SIGINT, catchint);
	signal(SIGQUIT, catchquit);

	if (getenv("PATH") == NULL)
		putenv("PATH=/bin:/usr/bin:/sbin:/usr/sbin");

	cp = getenv("HOME");

	if (cp) {
		strcpy(buf, cp);
		strcat(buf, "/");
		strcat(buf, ".foshrc");

		if ((access(buf, 0) == 0) || (errno != ENOENT))
			readfile(buf);
	}

	readfile(NULL);

	return 0;
}


/*
 * Read commands from the specified file.
 * A null name pointer indicates to read from stdin.
 */
static void
readfile(name)
	const char *	name;
{
	FILE *	fp;
	int	cc;
	BOOL	ttyflag;
	char	buf[CMDLEN];
	char 	*line=buf;

	if (sourcecount >= MAXSOURCE) {
		fprintf(stderr, "Too many source files\n");

		return;
	}

	fp = stdin;

	if (name) {
		fp = fopen(name, "r");

		if (fp == NULL) {
			perror(name);

			return;
		}
	}

	sourcefiles[sourcecount++] = fp;

	ttyflag = isatty(fileno(fp));

	while (TRUE) {
		if (ttyflag) {
		   char *p=NULL;
		   char *p1;
		   if (!strcmp(prompt,"PWD ")) {
			if ( (p=malloc(PATHLEN+2)) ) {
			    p[0]='\0';
			    getcwd(p, PATHLEN);
			    strcat(p,": ");
			} 
		   }
		   if (p==NULL) {
		       p1=prompt;
		    } else {
			p1=p;
		    }
		    
		    line = getline(p1);
		    gl_histadd(line);
		    if (p) free(p);
					

		} else {
		if (intflag && !ttyflag && (fp != stdin)) {
			fclose(fp);
			sourcecount--;

			return;
		}
	
		if (fgets(buf, CMDLEN - 1, fp) == NULL) {
			if (ferror(fp) && (errno == EINTR)) {
				clearerr(fp);

				continue;
			}

			break;
		}
		}

		if ( (cc = strlen(line)) == 0 ) exit(0);

		if (line[cc - 1] == '\n')
			cc--;

		while ((cc > 0) && isblank(line[cc - 1]))
			cc--;

		line[cc] = '\0';

		command(line);
	}

	if (ferror(fp)) {
		perror("Reading command line");

		if (fp == stdin)
			exit(1);
	}

	clearerr(fp);

	if (fp != stdin)
		fclose(fp);

	sourcecount--;
}


/*
 * Parse and execute one null-terminated command line string.
 * This breaks the command line up into words, checks to see if the
 * command is an alias, and expands wildcards.
 */
static void
command(cmd)
	const char *	cmd;
{
	char *		cp;
	const ALIAS *	alias;
	const char **	argv;
	int		argc;
	char		buf[CMDLEN];

	intflag = FALSE;

	freechunks();

	while (isblank(*cmd))
		cmd++;

	if ((*cmd == '\0') || (*cmd == '#') || !makeargs(cmd, &argc, &argv))
		return;

	/*
	 * Search for the command in the alias table.
	 * If it is found, then replace the command name with
	 * the alias, and append any other arguments to it.
	 */
	alias = findalias(argv[0]);

	if (alias) {
		cp = buf;
		strcpy(cp, alias->value);
		cp += strlen(cp);

		while (--argc > 0) {
			strcat(cp, " ");
			strcat(cp, *++argv);
			cp += strlen(cp);
		}
		cmd = buf;

		if (!makeargs(cmd, &argc, &argv))
			return;
	}

	/*
	 * Now look for the command in the builtin table, and execute
	 * the command if found.
	 */
	if (trybuiltin(argc, argv))
		return;

	/*
	 * Not found, run the program along the PATH list.
	 */
	runcmd(cmd, argc, argv);
}


/*
 * Try to execute a built-in command.
 * Returns TRUE if the command is a built in, whether or not the
 * command succeeds.  Returns FALSE if this is not a built-in command.
 */
static BOOL
trybuiltin(argc, argv)
	int		argc;
	const char **	argv;
{
	const CMDTAB *	cmdptr;
	int		oac;
	int		newargc;
	int		matches;
	int		i;
	const char *	newargv[MAXARGS];
	const char *	nametable[MAXARGS];

	cmdptr = cmdtab - 1;

	do {
		cmdptr++;

		if (cmdptr->name == NULL)
			return FALSE;

	} while (strcmp(argv[0], cmdptr->name));

	/*
	 * Give a usage string if the number of arguments is too large
	 * or too small.
	 */
	if ((argc < cmdptr->minargs) || (argc > cmdptr->maxargs)) {
		fprintf(stderr, "usage: %s %s\n",
			cmdptr->name, cmdptr->usage);

		return TRUE;
	}

	/*
	 * Check here for several special commands which do not
	 * have wildcarding done for them.
	 */
	if ((cmdptr->func == do_alias) || (cmdptr->func == do_prompt)) {
		(*cmdptr->func)(argc, argv);

		return TRUE;
	}

	/*
	 * Now for each command argument, see if it is a wildcard, and if
	 * so, replace the argument with the list of matching filenames.
	 */
	newargv[0] = argv[0];
	newargc = 1;
	oac = 0;

	while (++oac < argc) {
		matches = expandwildcards(argv[oac], MAXARGS, nametable);

		if (matches < 0)
			return TRUE;

		if ((newargc + matches) >= MAXARGS) {
			fprintf(stderr, "Too many arguments\n");

			return TRUE;
		}

		if (matches == 0)
			newargv[newargc++] = argv[oac];

		for (i = 0; i < matches; i++)
			newargv[newargc++] = nametable[i];
	}

	(*cmdptr->func)(newargc, newargv);

	return TRUE;
}


/*
 * Execute the specified command.
 */
static void
runcmd(cmd, argc, argv)
	const char *	cmd;
	int		argc;
	const char **	argv;
{
	const char *	cp;
	BOOL		magic;
	int		pid;
	int		status;

	magic = FALSE;

	for (cp = cmd; *cp; cp++) {
		if ((*cp >= 'a') && (*cp <= 'z'))
			continue;

		if ((*cp >= 'A') && (*cp <= 'Z'))
			continue;

		if (isdecimal(*cp))
			continue;

		if (isblank(*cp))
			continue;

		if ((*cp == '.') || (*cp == '/') || (*cp == '-') ||
			(*cp == '+') || (*cp == '=') || (*cp == '_') ||
			(*cp == ':') || (*cp == ','))
		{
			continue;
		}

		magic = TRUE;
	}

	if (magic) {
		system(cmd);

		return;
	}

	/*
	 * No magic characters in the command, so do the fork and
	 * exec ourself.  If this fails with ENOEXEC, then run the
	 * shell anyway since it might be a shell script.
	 */
	pid = fork();

	if (pid < 0) {
		perror("fork failed");

		return;
	}

	if (pid) {
		status = 0;
		intcrlf = FALSE;

		while (((pid = wait(&status)) < 0) && (errno == EINTR))
			;

		intcrlf = TRUE;

		if ((status & 0xff) == 0)
			return;

		fprintf(stderr, "pid %d: %s (signal %d)\n", pid,
			(status & 0x80) ? "core dumped" : "killed",
			status & 0x7f);

		return;
	}

	/*
	 * We are the child, so run the program.
	 * First close any extra file descriptors we have opened.
	 */	
	while (--sourcecount >= 0) {
		if (sourcefiles[sourcecount] != stdin)
			fclose(sourcefiles[sourcecount]);
	}

	execvp(argv[0], (char **) argv);

	if (errno == ENOEXEC) {
		system(cmd);
		exit(0);
	}

	perror(argv[0]);
	exit(1);
}


void
do_alias(argc, argv)
	int		argc;
	const char **	argv;
{
	const char *	name;
	char *	value;
	ALIAS *	alias;
	int	count;
	char	buf[CMDLEN];

	if (argc < 2) {
		count = aliascount;

		for (alias = aliastable; count-- > 0; alias++)
			printf("%s\t%s\n", alias->name, alias->value);

		return;
	}

	name = argv[1];

	if (argc == 2) {
		alias = findalias(name);

		if (alias)
			printf("%s\n", alias->value);
		else
			fprintf(stderr, "Alias \"%s\" is not defined\n", name);

		return;	
	}

	if (strcmp(name, "alias") == 0) {
		fprintf(stderr, "Cannot alias \"alias\"\n");

		return;
	}

	if (!makestring(argc - 2, argv + 2, buf, CMDLEN))
		return;

	value = malloc(strlen(buf) + 1);

	if (value == NULL) {
		fprintf(stderr, "No memory for alias value\n");

		return;
	}

	strcpy(value, buf);

	alias = findalias(name);

	if (alias) {
		free(alias->value);
		alias->value = value;

		return;
	}

	if ((aliascount % ALIASALLOC) == 0) {
		count = aliascount + ALIASALLOC;

		if (aliastable)
			alias = (ALIAS *) realloc(aliastable,
				sizeof(ALIAS *) * count);
		else
			alias = (ALIAS *) malloc(sizeof(ALIAS *) * count);

		if (alias == NULL) {
			free(value);
			fprintf(stderr, "No memory for alias table\n");

			return;
		}

		aliastable = alias;
	}

	alias = &aliastable[aliascount];

	alias->name = malloc(strlen(name) + 1);

	if (alias->name == NULL) {
		free(value);
		fprintf(stderr, "No memory for alias name\n");

		return;
	}

	strcpy(alias->name, name);
	alias->value = value;
	aliascount++;
}


/*
 * Look up an alias name, and return a pointer to it.
 * Returns NULL if the name does not exist.
 */
static ALIAS *
findalias(name)
	const char *	name;
{
	ALIAS *	alias;
	int	count;

	count = aliascount;

	for (alias = aliastable; count-- > 0; alias++) {
		if (strcmp(name, alias->name) == 0)
			return alias;
	}

	return NULL;
}


void
do_source(argc, argv)
	int		argc;
	const char **	argv;
{
	readfile(argv[1]);
}


void
do_exec(argc, argv)
	int		argc;
	const char **	argv;
{
	const char *	name;

	name = argv[1];

	if (access(name, 4)) {
		perror(name);

		return;
	}

	while (--sourcecount >= 0) {
		if (sourcefiles[sourcecount] != stdin)
			fclose(sourcefiles[sourcecount]);
	}

	argv[argc] = NULL;

	execv(name, (char **) argv + 1);
	exit(1);
}


void
do_prompt(argc, argv)
	int		argc;
	const char **	argv;
{
	char *	cp;
	char	buf[CMDLEN];

	if (!makestring(argc - 1, argv + 1, buf, CMDLEN))
		return;

	cp = malloc(strlen(buf) + 2);

	if (cp == NULL) {
		fprintf(stderr, "4 Out of memory, 0:1\n");

		return;
	}

	strcpy(cp, buf);
	strcat(cp, " "); 

	if (prompt)
		free(prompt);

	prompt = cp;
}


void
do_unalias(argc, argv)
	int		argc;
	const char **	argv;
{
	ALIAS *	alias;

	while (--argc > 0) {
		alias = findalias(*++argv);

		if (alias == NULL)
			continue;

		free(alias->name);
		free(alias->value);
		aliascount--;
		alias->name = aliastable[aliascount].name;
		alias->value = aliastable[aliascount].value;	
	}
}



static void
catchint(val)
	int	val;
{
	signal(SIGINT, catchint);

	intflag = TRUE;

	if (intcrlf)
		write(STDOUT, "\n", 1);
}


static void
catchquit(val)
	int	val;
{
	signal(SIGQUIT, catchquit);

	intflag = TRUE;

	if (intcrlf)
		write(STDOUT, "\n", 1);
}


/*
 * Print the usage information and quit.
 */
static void
usage()
{
	fprintf(stderr, "Stand-alone shell (version %s)\n", version);
	fprintf(stderr, "\n");
	fprintf(stderr, "Usage: sash [-q] [-c command] [-p prompt] [-i]\n");

	exit(1);
}

/* END CODE */
