/*
 * Copyright (c) 1998 by David I. Bell
 * Permission is granted to use, distribute, or modify this source,
 * provided that this copyright notice remains intact.
 *
 * Most simple built-in commands are here.
 *
 * Modified by Radovan Garabik : removed unneccessary commands
 */

#include "fosh.h"

#include <sys/types.h>
#include <sys/stat.h>
#include <signal.h>
#include <utime.h>
#include <errno.h>


void
do_echo(argc, argv)
	int		argc;
	const char **	argv;
{
	BOOL	first;

	first = TRUE;

	while (argc-- > 1) {
		if (!first)
			fputc(' ', stdout);

		first = FALSE;
		fputs(*++argv, stdout);
	}

	fputc('\n', stdout);
}


void
do_pwd(argc, argv)
	int		argc;
	const char **	argv;
{
	char	buf[PATHLEN];

	if (getcwd(buf, PATHLEN) == NULL) {
		fprintf(stderr, "Cannot get current directory\n");

		return;
	}

	printf("%s\n", buf);
}


void
do_cd(argc, argv)
	int		argc;
	const char **	argv;
{
	const char *	path;

	if (argc > 1)
		path = argv[1];
	else {
		path = getenv("HOME");

		if (path == NULL) {
			fprintf(stderr, "No HOME environment variable\n");

			return;
		}
	}

	if (chdir(path) < 0)
		perror(path);
}



void
do_exit(argc, argv)
	int		argc;
	const char **	argv;
{
	exit(0);
}


void
do_setenv(argc, argv)
	int		argc;
	const char **	argv;
{
	const char *	name;
	const char *	value;
	char *		str;

	name = argv[1];
	value = argv[2];

	/*
	 * The value given to putenv must remain around, so we must malloc it.
	 * Note: memory is not reclaimed if the same variable is redefined.
	 */
	str = malloc(strlen(name) + strlen(value) + 2);

	if (str == NULL)
	{
		fprintf(stderr, "Cannot allocate memory\n");

		return;
	}

	strcpy(str, name);
	strcat(str, "=");
	strcat(str, value);

	putenv(str);
}



void
do_umask(argc, argv)
	int		argc;
	const char **	argv;
{
	const char *	cp;
	int		mask;

	if (argc <= 1) {
		mask = umask(0);
		umask(mask);
		printf("%03o\n", mask);

		return;
	}

	mask = 0;
	cp = argv[1];

	while (isoctal(*cp))
		mask = mask * 8 + *cp++ - '0';

	if (*cp || (mask & ~0777)) {
		fprintf(stderr, "Bad umask value\n");

		return;
	}

	umask(mask);
}


void
do_kill(argc, argv)
	int		argc;
	const char **	argv;
{
	const char *	cp;
	int		sig;
	int		pid;

	sig = SIGTERM;

	if (argv[1][0] == '-') {
		cp = &argv[1][1];

		if (strcmp(cp, "HUP") == 0)
			sig = SIGHUP;
		else if (strcmp(cp, "INT") == 0)
			sig = SIGINT;
		else if (strcmp(cp, "QUIT") == 0)
			sig = SIGQUIT;
		else if (strcmp(cp, "KILL") == 0)
			sig = SIGKILL;
		else {
			sig = 0;

			while (isdecimal(*cp))
				sig = sig * 10 + *cp++ - '0';

			if (*cp) {
				fprintf(stderr, "Unknown signal\n");

				return;
			}
		}

		argc--;
		argv++;
	}

	while (argc-- > 1) {
		cp = *++argv;
		pid = 0;

		while (isdecimal(*cp))
			pid = pid * 10 + *cp++ - '0';

		if (*cp) {
			fprintf(stderr, "Non-numeric pid\n");

			return;
		}

		if (kill(pid, sig) < 0)
			perror(*argv);
	}
}


void
do_where(argc, argv)
	int		argc;
	const char **	argv;
{
	const char *	program;
	const char *	dirName;
	char *		path;
	char *		endPath;
	char *		fullPath;
	BOOL		found;

	found = FALSE;
	program = argv[1];

	if (strchr(program, '/') != NULL)
	{
		fprintf(stderr, "Program name cannot include a path\n");

		return;
	}

	path = getenv("PATH");

	fullPath = getchunk(strlen(path) + strlen(program) + 2);
	path = chunkstrdup(path);

	if ((path == NULL) || (fullPath == NULL))
	{
		fprintf(stderr, "Memory allocation failed\n");

		return;
	}

	/*
	 * Check out each path to see if the program exists and is
	 * executable in that path.
	 */
	for (; path; path = endPath)
	{
		/*
		 * Find the end of the next path and NULL terminate
		 * it if necessary.
		 */
		endPath = strchr(path, ':');

		if (endPath)
			*endPath++ = '\0';

		/*
		 * Get the directory name, defaulting it to DOT if
		 * it is null.
		 */
		dirName = path;

		if (dirName == '\0')
			dirName = ".";

		/*
		 * Construct the full path of the program.
		 */
		strcpy(fullPath, dirName);
		strcat(fullPath, "/");
		strcat(fullPath, program);

		/*
		 * See if the program exists and is executable.
		 */
		if (access(fullPath, X_OK) < 0)
		{
			if (errno != ENOENT)
				printf("%s: %s\n", fullPath, strerror(errno));

			continue;
		}

		printf("%s\n", fullPath);
		found = TRUE;
	}

	if (!found)
		printf("Program \"%s\" not found in PATH\n", program);
}

/* END CODE */
