#include <signal.h>
#include "log.h"
#include "types.h"
#include "signals.h"

typedef void restorer_t(void);

/*----------------------------------------------------------------------
Install function `handler' as signal handler for signal `signum'. `mask'
is the set of signals to block when `signum' is handled. `Flags' are the
sig_action flags. Save the old setting in `sa_save'. If successful, add
`signum' to the set of handled signals `ss_handled'.
----------------------------------------------------------------------*/
void sa_set(int signum, void (*handler)(int), sigset_t const *mask, int flags, struct sigaction *sa_save, sigset_t *ss_handled)
{   struct sigaction sa;
    sa.sa_handler = handler;
    sa.sa_mask = *mask;
    sa.sa_flags = flags;
    sa.sa_restorer = NIL(restorer_t);
    if (sigaction(signum, &sa, sa_save)<0)
        log(LL_ERR, "Failed to set signal handler for %d, %m", signum);
    else
    {   if (ss_handled)
            sigaddset(ss_handled, signum);
    }
}

/* Reset signal handling to as it was before, if it was actually handled */
void sa_reset(int signum, struct sigaction *sa_save, sigset_t const *ss_handled)
{   if (sigismember(ss_handled, signum))
        if (sigaction(signum, sa_save, NIL(struct sigaction))<0)
            log(LL_ERR, "Failed to reset signal handler for %d, %m", signum);
}
