#include <signal.h>
#include <stdlib.h>
#include <string.h>
#include "rs.h"
#include "log.h"
#include "timeval.h"
#include "config.h"
#include "signals.h"
#include "rsm.h"


/* SIGRING is the signal that simulates a RING */

#define SIGRING SIGUSR1


/*------------------------------------------------------------------------
The ring state machine uses tiny "processes", organised in lists.

A process has an id to distinguish it, a pointer to the ringsequence it
is running, and the current interval in that ringsequence.

A process is created when a ring is detected, provided its initial guard
time has elapsed since the previous ring.  The process is now put in a
list of "active" processes.  When a new ring occurs, all the processes
in the active list are checked.  If the elapsed time matches the current
interval of the process, it goes back into the list for the next interval.
If not, the process dies and goes to a "passive" list.

When a process has reached its last ring successfully, it goes to a
special "final" list.  This list is kept ordered on increasing final
guard time.  Processes whose final guard time has elapsed are taken off
this list, their command executed, and they go to the "passive" list.
Then the real time interval timer is set to warn when the final guard
time of the first remaining process on the list has elapsed.  When the
timer goes off, more processes may be restarted.

When a ring is detected, the "final" list is discarded - the processes
have not reached their final guard time.  A possible scheduled alarm
is left as is; the alarm handling routine can handle spurious calls and
stopping the alarm is about as much trouble as processing it.

Processes are malloced when necessary, but passive processes are
reused.  Processes are never freed.  Incidentally, the maximum number
of needed processes can never be more than the combined length of all
ring sequences.
------------------------------------------------------------------------*/


typedef struct _process
{   struct _process *next;      /* Next process in list             */
    unsigned long id;           /* Process id                       */
    ringseq *master;            /* Ringsequence being run           */
    interval *current;          /* Current interval of ringsequence */
} process;

static process *passive = NIL(process); /* The passive processes */
static process *final   = NIL(process); /* The final   processes */
static process *active  = NIL(process); /* The active  processes */

static sigset_t ss_handled;  /* Signal set we have handlers set up for */
static sigset_t ss_saved;    /* Saved signal set to restore on exit    */

static struct timeval start; /* Start time of current interring period */

/*----------------------------------------------------------------------*/


/*-------------------------------------------------
Create a new process to run the master ringsequence
Take one from the passive list if possible
-------------------------------------------------*/

static process *newprocess(ringseq *master)
{   process *p;
    static unsigned long nextid=0;
    if (passive)
    {   p = passive;
        passive = passive->next;
    }
    else
    {   p = NEW(process);
        if (!p)
            log(LL_ERR, "Failed to allocate new process, %m");
    }
    if (p)
    {   p->id = nextid++;
        p->master = master;
        p->current = master->intervals;
        log(LL_INFO, " *spawn%6lu - ringsequence %lu", p->id, p->master->id);
    }
    return p;
}


/*----------------------------------
Show a list of all running processes
----------------------------------*/

static void ps(void)
{   process *p;
    struct timeval now;
    struct timeval elapsed;
    gettimeofday(&now, NIL(struct timezone));
    tv_subtract(&elapsed, &now, &start);
    log(LL_DEBUG, "Elapsed%10ld.%06ld", elapsed.tv_sec, elapsed.tv_usec);
    for (p = final; p; p = p->next)
        log(LL_DEBUG, " =final%6lu%6ld.%06ld", p->id, p->master->final.tv_sec, p->master->final.tv_usec);
    for (p = active; p; p = p->next)
        log(LL_DEBUG, " =active%5lu%6ld.%06ld-%ld.%06ld", p->id, p->current->ranges->min.tv_sec, p->current->ranges->min.tv_usec, p->current->ranges->max.tv_sec, p->current->ranges->max.tv_usec);
}


/*--------------------------------------------------------
Insert a process into the appropriate active or final list
according to its current interval
--------------------------------------------------------*/

static void insert(process *p)
{   if (p)
    {   if (p->current)
        {   p->next = active;
            active = p;
        }
        else
        {   process **pp;
            for (pp = &final;
                 *pp && tv_lt(&(*pp)->master->final, &p->master->final);
                 pp = &(*pp)->next);
            p->next = *pp;
            *pp = p;
        }
    }
}


/*----------------------------------------------------
Step an active process to its next interval
if the current interval is matched by the elapsed time
Kill the process if not
----------------------------------------------------*/

static void ringstep(struct timeval const *elapsed,process *p)
{   if (tv_findrange(p->current,elapsed))
    {   p->current = p->current->next;
        insert(p);
        log(LL_INFO, " *step%7lu", p->id);
    }
    else
    {   p->next = passive;
        passive = p;
        log(LL_INFO, " *kill%7lu", p->id);
    }
}


/*----------------------------------
Process a ring - check all processes
----------------------------------*/

static void ring(struct timeval const *elapsed)
{   process **pp;
    process *p;
    ringseq *r;

    /* Kill any remaining final processes */
    for (pp = &passive; *pp; pp = &(*pp)->next);
    *pp = final;
    final = NIL(process);

    /* Step all active processes as appropriate */
    p = active;
    active = NIL(process);
    while (p)
    {   process *q;
        q=p->next;
        ringstep(elapsed,p);
        p=q;
    }

    /* Create new processes if appropriate */
    for (r=program;r;r=r->next)
        if (tv_leq(&r->initial,elapsed))
            insert(newprocess(r));
}


/*----------------------------------------------
Determine if we must schedule the interval timer
for a "final" process or not. If so, schedule it
----------------------------------------------*/

static bool schedule(process *p)
{   struct timeval future;
    struct timeval now;
    struct itimerval it;
    bool doschedule;
    it.it_interval.tv_sec = 0;
    it.it_interval.tv_usec = 0;
    tv_add(&future,&start,&p->master->final);
    gettimeofday(&now,NIL(struct timezone));
    tv_subtract(&it.it_value,&future,&now);
    doschedule = it.it_value.tv_sec>0 || (it.it_value.tv_sec==0&&it.it_value.tv_usec>0);
    if (doschedule)
    {   if (setitimer(ITIMER_REAL,&it,NIL(struct itimerval))<0)
            log(LL_ERR, "Failed to set interval timer to %ld.%06ld, %m", it.it_value.tv_sec, it.it_value.tv_usec);
        else
            log(LL_INFO, "Set interval timer to %ld.%06ld", it.it_value.tv_sec, it.it_value.tv_usec);
    }
    return doschedule;
}


/*-------------------------------------------------------------
Run commands of "final" processes whose final guard has expired
Schedule a timeout for the first remaining one
-------------------------------------------------------------*/

static void check_finals(void)
{   while (final && !schedule(final))
    {   process *p;
        p = final;
        final = final->next;
        log(LL_INFO, " *alarm%6lu", p->id);
        rs_exec(p->master);
        p->next = passive;
        passive = p;
    }
}


/*-----------------------------------------------------
Clear all running processes to start with a clean slate
-----------------------------------------------------*/

static void clear_all_processes(void)
{   process **pp;
    log(LL_INFO, "Clearing all processes");
    for (pp=&passive;*pp;pp=&(*pp)->next);
    for (*pp=final;*pp;pp=&(*pp)->next);
    *pp = active;
    final = NIL(process);
    active = NIL(process);
}


/*---------------------
Process a detected RING
---------------------*/

static void process_ring(struct timeval *now, struct timeval *elapsed)
{   if tv_lt(elapsed, &jointime)
        log(LL_NOTICE, "Ignoring ring within join time %ld.%06ld", jointime.tv_sec, jointime.tv_usec);
    else
    {   log(LL_INFO, "Processing ring");
        ring(elapsed);
        tv_copy(&start, now);
        check_finals();
        ps();
    }
}

/*-------------------------
The true modem RING handler
-------------------------*/

void rsm_process_ring(void)
{   sigset_t ss_save;
    struct timeval now;
    struct timeval elapsed;
    sigprocmask(SIG_BLOCK, &ss_handled, &ss_save);
    gettimeofday(&now, NIL(struct timezone));
    tv_subtract(&elapsed, &now, &start);
    log(LL_NOTICE, "Got modem ring after %ld.%06ld seconds", elapsed.tv_sec, elapsed.tv_usec);
    process_ring(&now, &elapsed);
    sigprocmask(SIG_SETMASK, &ss_save, NIL(sigset_t));
}


/* -------------------------------------------- */
/* Signal handler to invoke the above functions */
/* -------------------------------------------- */

static void handler(int signum)
{   struct timeval now;
    struct timeval elapsed;
    gettimeofday(&now, NIL(struct timezone));
    tv_subtract(&elapsed, &now, &start);
    log(LL_NOTICE, "Got signal %d, %s, after %ld.%06ld seconds", signum, strsignal(signum), elapsed.tv_sec, elapsed.tv_usec);
    switch(signum)
    {   case SIGALRM:
            check_finals();
        break;
        case SIGINT:
            clear_all_processes();
            tv_copy(&start, &now);
        break;
        case SIGRING:
            process_ring(&now, &elapsed);
        break;
        default:
            log(LL_ERR, "Ignored unexpected signal");
        break;
    }
}

/* -------------- */
/* Initialisation */
/* -------------- */

static struct sigaction sa_alarm;
static struct sigaction sa_int;
static struct sigaction sa_ring;


/*--------------------------
Start the ring state machine
--------------------------*/

extern void rsm_start(void)
{   sigset_t ss_all;

    gettimeofday(&start, NIL(struct timezone));

    log(LL_NOTICE, "Starting ring state machine");

    sigemptyset(&ss_all);
    sigaddset(&ss_all, SIGALRM);
    sigaddset(&ss_all, SIGINT);
    sigaddset(&ss_all, SIGRING);

    sigemptyset(&ss_handled);
    sa_set(SIGALRM, &handler, &ss_all, SA_RESTART, &sa_alarm, &ss_handled);
    sa_set(SIGINT,  &handler, &ss_all, SA_RESTART, &sa_int,   &ss_handled);
    sa_set(SIGRING, &handler, &ss_all, SA_RESTART, &sa_ring,  &ss_handled);

    sigprocmask(SIG_UNBLOCK, &ss_handled, &ss_saved);
}


/*-------------------------
Stop the ring state machine
-------------------------*/

extern void rsm_stop(void)
{   log(LL_NOTICE, "Stopping ring state machine");

    sigprocmask(SIG_SETMASK, &ss_saved, NIL(sigset_t));

    sa_reset(SIGRING, &sa_ring,  &ss_handled);
    sa_reset(SIGINT,  &sa_int,   &ss_handled);
    sa_reset(SIGALRM, &sa_alarm, &ss_handled);

    clear_all_processes();

}
