#include <ctype.h>
#include <fcntl.h>
#include <grp.h>
#include <pwd.h>
#include <stdlib.h>
#include <string.h>
#include <syslog.h>
#include <unistd.h>
#include "types.h"
#include "log.h"
#include "config.h"
#include "rs.h"

int modemfd; /* Modem device file descriptor */


/*---------------------------------------------
Make a quoted string representation of a string
---------------------------------------------*/

static int quote(char *buf, int room, const char *str)
{   const char *c;
    char const* delim = "\"";
    int n = 0;
    n += snprintf(&buf[n], (unsigned)room-n, delim);
    for (c = str; *c; c++)
        switch (*c)
        {   case '\n': n += snprintf(&buf[n], (unsigned)room-n, "\\n"); break;
            case '"': n += snprintf(&buf[n], (unsigned)room-n, "\\\""); break;
            case '\\': n += snprintf(&buf[n], (unsigned)room-n, "\\\\"); break;
            default: n += snprintf(&buf[n], (unsigned)room-n, "%c", *c); break;
        }
    n += snprintf(&buf[n], (unsigned)room-n, delim);
    return n;
}


/*-------------------
List one ringsequence
-------------------*/

/*
static void lsone(int level, const ringseq *rs)
{   int n=0;
    interval *i;
    char buf[1024];
    n += snprintf(&buf[n], (unsigned)1024-n, "%ld.%06ld*", rs->initial.tv_sec, rs->initial.tv_usec);
    for (i = rs->intervals; i; i=i->next)
    {   range *r;
        for (r = i->ranges; r; r = r->next)
            n += snprintf(&buf[n], (unsigned)1024-n, "%ld.%06ld-%ld.%06ld%s", r->min.tv_sec, r->min.tv_usec, r->max.tv_sec, r->max.tv_usec, r->next?",":"");
        n += snprintf(&buf[n], (unsigned)1024-n, "*");
    }
    n += snprintf(&buf[n], (unsigned)1024-n, "%ld.%06ld:", rs->final.tv_sec, rs->final.tv_usec);
    n += quote(&buf[n], 1024-n, rs->command);
    if (n==1024)
        (void)sprintf(&buf[1020], "...");
    log(level, "%s", buf);
}
*/


/*--------------------
List the ringsequences
--------------------*/

/*
static void ls(int level, const ringseq *program)
{   const ringseq *rs;
    for (rs=program; rs; rs=rs->next)
        lsone(level, rs);
}
*/


/*-------------------------------
Run the command of a ringsequence
-------------------------------*/

extern void rs_exec(const ringseq *r)
{   pid_t pid;
    char buf[1024];
    int n;

    /* Prepare the command */
    n = snprintf(buf, 1024, "Running ");
    n += quote(&buf[n], 1024-n, r->command);
    if (n==1024)
        (void)sprintf(&buf[1020], "...");
    log(LL_NOTICE, "%s", buf);

    /* Fork the child process to exec the command */
    pid = fork();
    if (pid<0)
        log(LL_ERR, "Failed to fork, %m");
    if (pid==0)
    {   int nullfd;

        /* Close files */
        /* For 0-2 we prefer the null device */
        close(modemfd);
        nullfd = open(nulldevice, O_RDWR);
        if (nullfd<0)
        {   log(LL_WARNING, "Could not open null device %s for read/write, %m", nulldevice);
            close(0);
            close(1);
            close(2);
        }
        else
        {   if (nullfd!=0) dup2(nullfd, 0);
            if (nullfd!=1) dup2(nullfd, 1);
            if (nullfd!=2) dup2(nullfd, 2);
            if (nullfd>2)  close(nullfd);
        }
        closelog();

        /* And exec the command */
        execlp(shell, shell, "-c", r->command, NIL(char));
        log(LL_ERR, "Failed to exec %s, %m", r->command);
        exit(127);
    }
    if (pid>0)
        log(LL_INFO, "Forked pid %d", pid);
}

/* Free a list of ranges */
static void freeranges(range *r)
{   while (r)
    {   range *n;
        n = r->next;
        free(r);
        r = n;
    }
}

/* Free a list of intervals */
static void freeintervals(interval *i)
{   while (i)
    {   interval *n;
        freeranges(i->ranges);
        n = i->next;
        free(i);
        i = n;
    }
}

/* Free a list of ringsequences */
void freeringseqs(ringseq *r)
{   while (r)
    {   ringseq *n;
        freeintervals(r->intervals);
        n = r->next;
        free(r);
        r = n;
    }
}

/* Create a new range */
range *newrange
(   range *next,
    long misec, long miusec,
    long masec, long mausec
)
{   range *r;
    r = NEW(range);
    if (r)
    {   r->next = next;
        r->min.tv_sec = misec;
        r->min.tv_usec = miusec;
        r->max.tv_sec = masec;
        r->max.tv_usec = mausec;
    }
    return r;
}

/* Create a new interval */
interval *newinterval
(   interval *next,
    range *ranges
)
{   interval *i;
    i = NEW(interval);
    if (i)
    {   i->next = next;
        i->ranges = ranges;
    }
    return i;
}

/* Create a new ringsequence */
ringseq *newringseq
(   long isec, long iusec,
    interval *intervals,
    long fsec, long fusec,
    char *command
)
{   ringseq *r;
    static unsigned long nextid = 0;
    r = NEW(ringseq);
    if (r)
    {   r->next = NIL(ringseq);
        r->id = nextid++;
        r->initial.tv_sec = isec;
        r->initial.tv_usec = iusec;
        r->intervals = intervals;
        r->final.tv_sec = fsec;
        r->final.tv_usec = fusec;
        r->command = command;
    }
    return r;
}

/* Clear out the current rule set */
void remove_program(void)
{   log(LL_DEBUG, "Removing program %p", program);
    freeringseqs(program);
}

/* Install a new rule set */
void install_program(ringseq *newprogram)
{   log(LL_DEBUG, "Installing program %p", newprogram);
    program = newprogram;
}
