#include <errno.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/ioctl.h>
#include <sys/wait.h>
#include <unistd.h>
#include "config.h"
#include "log.h"
#include "rsm.h"
#include "rs.h"
#include "signals.h"

/* --------------- */
/* Signal handlers */
/* --------------- */

/* Indication of what level of signal we got */
/* 0=none, 2=HUP, 3=TERM, 4=other            */
static volatile short siglevel;

static void handler(int signum)
{   log(LL_NOTICE, "Got signal %d, %s", signum, strsignal(signum));
    switch (signum)
    {   case SIGHUP:  if (siglevel<2) siglevel=2; break;
        case SIGTERM: if (siglevel<3) siglevel=3; break;
        default:      if (siglevel<4) siglevel=4; break;
    }
}

static void childhandler(int signum)
{   pid_t pid;
    int status;
    log(LL_DEBUG, "Got signal %d, %s", signum, strsignal(signum));
    pid = waitpid(-1, &status, WNOHANG|WUNTRACED);
    if (pid<0)
        log(LL_ERR, "Waitpid error, %m");
    else if (pid==0)
        log(LL_WARNING, "Spurious SIGCHLD");
    else
    {   if (WIFEXITED(status))
            log(LL_INFO, "Child %d exited with status %d", pid, WEXITSTATUS(status));
        if (WIFSIGNALED(status))
            log(LL_INFO, "Child %d killed by signal %d, %s", pid, WTERMSIG(status), strsignal(WTERMSIG(status)));
        if (WIFSTOPPED(status))
            log(LL_INFO, "Child %d stopped by signal %d, %s", pid, WSTOPSIG(status), strsignal(WSTOPSIG(status)));
    }
}

/* --------------- */
/* Signal handling */
/* --------------- */

static sigset_t ss_all;
static sigset_t ss_handled;
static sigset_t ss_saved;
static struct sigaction sa_quit;
static struct sigaction sa_term;
static struct sigaction sa_hup;

static void initsignals(void)
{   sigset_t ss_empty;
    sigemptyset(&ss_empty);
    sa_set(SIGCHLD, &childhandler, &ss_empty, SA_RESTART, NIL(struct sigaction), NIL(sigset_t));
}

static void setsignals(void)
{   sigemptyset(&ss_all);
    sigaddset(&ss_all, SIGQUIT);
    sigaddset(&ss_all, SIGTERM);
    sigaddset(&ss_all, SIGHUP );

    sigemptyset(&ss_handled);
    sa_set(SIGQUIT, &handler, &ss_all, (signed)SA_ONESHOT, &sa_quit, &ss_handled);
    sa_set(SIGTERM, &handler, &ss_all, (signed)SA_ONESHOT, &sa_term, &ss_handled);
    sa_set(SIGHUP,  &handler, &ss_all, (signed)SA_ONESHOT, &sa_hup,  &ss_handled);

    siglevel = 0;

    sigprocmask(SIG_UNBLOCK, &ss_handled, &ss_saved);
}

static void resetsignals(void)
{   sigprocmask(SIG_SETMASK, &ss_saved, NIL(sigset_t));

    sa_reset(SIGHUP,  &sa_hup,  &ss_handled);
    sa_reset(SIGTERM, &sa_term, &ss_handled);
    sa_reset(SIGQUIT, &sa_quit, &ss_handled);
}

/* ----------------------------- */
/* Opening and closing utilities */
/* ----------------------------- */

static int openmodem(void)
{   int fd;
    fd = open(modemdevice, O_RDONLY|O_NOCTTY|O_NDELAY);
    if (fd<0)
        log(LL_ERR, "Failed to open modem device %s, %m", modemdevice);
    else
        log(LL_INFO, "Opened modem device %s", modemdevice);
    return fd;
}

static void removepid(void)
{   unlink(pidfile);
}

static bool writepid(pid_t pid)
{   FILE *f;
    f = fopen(pidfile, "w");
    if (!f)
    {   log(LL_ERR, "Failed to open pidfile %s for writing, %m", pidfile);
        return false;
    }
    fprintf(f, "%d\n", pid);
    if (fclose(f))
    {   log(LL_ERR, "Failed to close pidfile %s, %m", pidfile);
        unlink(pidfile);
        return false;
    }
    if (atexit(&removepid)<0)
        log(LL_ERR, "Won't be able to remove pidfile %s at exit, %m");
    return true;
}

/* ------------------------------- */
/* Run rsm until ioctl gives error */
/* ------------------------------- */

static void run(void)
{   int ioerror = 0;
    rsm_start();

    /* Ring state machine loop                             */
    /* Keeps waiting for a ring until some error condition */
    /* The condition can be interruption by a signal       */
    while (ioerror==0)
    {   log(LL_DEBUG, "Waiting for ring");

        /* Here is the central ioctl of this program! */
        ioerror = ioctl(modemfd, TIOCMIWAIT, TIOCM_RNG);

        if (ioerror==0)
            rsm_process_ring();
        else if (errno==EIO)
        {   log(LL_INFO, "%m, probably another application closed the modem device");
            close(modemfd);
            modemfd = openmodem();
            ioerror = modemfd<0?-1:0;
        }
        else if (errno!=EINTR)
            log(LL_ERR, "Error waiting for ring, %m");
    }

    rsm_stop();
}

int main(int argc, char *const* argv)
{   /* Scan command line arguments */
    scan_options(argc, argv);

    /* See what we have to do */
    if (runmode==4)
        return 1;
    if (runmode==3)
        return 0;
    if (!read_configfile())
        return 1;
    if (runmode==2)
        return 0;
    if (runmode==0)
    {   pid_t pid = become_daemon();
        if (pid<0)
            return 1;
        if (pid>0)
            return 0;
    }

    /* Initialise loop */
    log(LL_NOTICE, "Start");
    if (pidfile && !writepid(getpid()))
        pidfile = NIL(char); /* writepid uses atexit() to remove pidfile */
    initsignals();

    /* Main loop, runs ring state machine with HUP signal interruptions */
    while (siglevel<=2)
    {   modemfd = openmodem();
        setsignals();
        run();
        resetsignals();
        close(modemfd);

        if (siglevel==2)
        {   log(LL_NOTICE, "Re-reading configuration file %s", configfile);
            if (!read_configfile())
                log(LL_WARNING, "Faulty configuration file %s not installed - this may eat some memory! - consider killing and restarting %s", configfile, self);
        }
    }

    log(LL_NOTICE, "Exit");
    return 0;
}
