#include <errno.h>
#include <stdio.h>
#include <stdarg.h>
#include <syslog.h>
#include <unistd.h>
#include <string.h>
#include "types.h"
#include "config.h"
#include "log.h"

#define LOGWIDTH 1024  /* Maximum width of logged lines */

static bool log_syslog = false;
static bool log_stderr = true;

static const int sysloglevels[] = {LOG_DEBUG, LOG_INFO, LOG_NOTICE, LOG_WARNING, LOG_ERR, LOG_CRIT, LOG_ALERT, LOG_EMERG};

/* Printf-like log function with log level constraint */
/* Outputs to stderr or syslog as necessary */
void log(const int level, const char *fmt, ...)
{   va_list ap;
    char buf[LOGWIDTH];
    char fmt2[LOGWIDTH];
    int i = 0;
    int j = 0;

    if (level>=loglevel)

    {   /* First expand %m in format */
        /* We silently truncate to LOGWIDTH-1 characters */
        while (fmt[i] && j<LOGWIDTH-1)
            if (fmt[i]=='%' && fmt[i+1]=='m')
            {   j += snprintf(&fmt2[j], (unsigned)LOGWIDTH-j, "%s", strerror(errno));
                i += 2;
            }
            else
                fmt2[j++] = fmt[i++];
        fmt2[j<LOGWIDTH?j:LOGWIDTH-1] = '\0';

        /* Then do printf-like expansion */
        /* Again we silently truncate to LOGWIDTH-1 characters */
        va_start(ap, fmt);
        vsnprintf(buf, LOGWIDTH, fmt2, ap);
        va_end(ap);
        buf[LOGWIDTH-1] = '\0';

        /* Now output to the required destination(s) */
        if (log_syslog)
            syslog(sysloglevels[level], "%s", buf);
        if (log_stderr)
            fprintf(stderr, "%s\n", buf);
    }

}

/* Put pringd in background and switch logging to syslog */
pid_t become_daemon(void)
{   pid_t pid;
    pid = fork();
    if (pid<0)
        log(LL_ERR, "Can't fork, %m");
    if (pid==0)
    {   closelog();
        openlog(self, LOG_PID, LOG_DAEMON);
        log_syslog = true;
        close(0);
        close(1);
        close(2);
        log_stderr = false;
        if (setsid()<0)
            log(LL_ERR, "Can't create session, %m");
    }
    if (pid>0)
        log(LL_NOTICE, "Backgrounded %s process %d", self, pid);
    return pid;
}
