#include <ctype.h>
#include <limits.h>
#include <stdlib.h>
#include <sys/time.h>
#include "types.h"
#include "log.h"
#include "syntax.tab.h"
#include "decode.h"

/*-----------------------------------------
Run once along a word encoding
Call store once with each decoded character
-----------------------------------------*/

static void decode_once(const char *word, void (*store)(char))
{   const char *c = word;
    while (*c!='\0')
    {   switch(*c)
        {   case '\\':
                c++;
                switch(*c)
                {   case 'b': store('\b'); break;
                    case 'f': store('\f'); break;
                    case 'n': store('\n'); break;
                    case 'r': store('\r'); break;
                    case 'v': store('\v'); break;
                    case '\n': break;
                    default: (*store)(*c); break;
                }
                break;
            case '"': break;
            default: store(*c); break;
        }
        c++;
    }
    store('\0');
}

/* Decoded length of word being decoded */
static unsigned long wordlength;

/* Malloc'd decoded word */
static char *wordcontents;

/* Count decoded character */
static void count(char c)
{   wordlength++;
}

/* Store decoded character */
static void store(char c)
{   wordcontents[wordlength++] = c;
}

/*--------------------------------------------
Decode a string representation
I don't like fixed limits on strings
So I run along it once to get length
Then again to copy to suitably malloc'd memory
--------------------------------------------*/

char *decode_string(const char *word)
{   wordlength = 0;
    decode_once(word, &count);
    wordcontents = malloc(wordlength);
    if (wordcontents)
    {   wordlength = 0;
        decode_once(word, &store);
    }
    return wordcontents;
}

/*--------------------------
Decode an int representation
Check for overflow
--------------------------*/

bool decode_int(const char *str, const int max, int *value)
{   int i;
    const char *c;

    /* Decode integer */
    i = 0;
    for (c = str; isdigit(*c); c++)
    {   int d = *c-'0';
        if ((max-d)/10<i)
        {   /* Overflow */
            *value = max;
            log(LL_ERR, "Overflow in number representation: %s", str);
            return false;
        }
        i = i*10+d;
    }

    /* Return */
    *value = i;
    return true;
}

static struct
{   char const* word;
    int code;
} table[] =
{   {"jointime", JOINTIME},
    {"loglevel", LOGLEVEL},
    {"modemdevice", MODEMDEVICE},
    {"shell", SHELL},
    {NIL(char), -1}
};

bool decode_word(char const* word, int * code)
{   int i;
    for (i=0; table[i].word && strcmp(table[i].word, word); i++);
    if (table[i].word)
        *code = table[i].code;
    else
        log(LL_ERR, "Unrecognised word: \"%s\"", word);
    return table[i].word!=NIL(char);
}
