//
// Program for loging input on a serial port
//
// Copyright (c) 1998 Sinkovics Zoltan 
//
// Released under the  conditions  of  the  GNU General 
// Public License.
// 

#include "logserial.h"
#include <stdio.h>
#include <unistd.h>
#include <fcntl.h>
#include <ctype.h>
#include <signal.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/stat.h>

#define 	MAXPATH 	255

char NAME[MAXPATH];	// program name
int outfd;		// output file descriptor
int portfd;		// serial port file descriptor

void closetty(int sig) ;
void version(void) ;
void usage(void) ;

void version(void)
{
    fprintf(stdout,"\n%s-%s by Sinkovics Zoltan \n\n",NAME,VERSION);
}

void usage(void)
{
    fprintf(stderr,"\n %s-%s\n\n",NAME,VERSION);
    fprintf(stderr," usage: %s [parameters]\n",NAME);
    fprintf(stderr,"\tOptional parameters are:\n");
    fprintf(stderr,"\t\t -h prints this help screen\n");
    fprintf(stderr,"\t\t -v prints version of %s\n",NAME);
    fprintf(stderr,"\t\t -t tty        \t default /dev/ttyS1\n");
    fprintf(stderr,"\t\t -s speed      \t default 9600\n");
    fprintf(stderr,"\t\t -d data bits  \t default 8\n");
    fprintf(stderr,"\t\t -p parity     \t default N\n");
    fprintf(stderr,"\t\t -c    if exists software handshake used (by default hw used)\n");
    fprintf(stderr,"\t\t -f output file\t default /dev/stdout\n");
    fprintf(stderr,"\t\t -b    if exists logserial run as a daemon\n\n");
}

int main(int argc, char **argv)
{
    int c,childproc;
    char tty[MAXPATH];		// serial port tty
    char baudrate[10];		// baud rate as a string
    char parity[10];		// parity as a string
    char bits[10];		// data bits as a string
    int hwhandshake=1;		// true if RTS/CTS handshake used
    int swhandshake=0;		// true if XON/XOFF handshake used
    char fileout[MAXPATH];	// output file
    struct stat filestatus;	// status of the output file
    int daemonize=0;		// true if daemonize is enabled
    
    fd_set fds;			// file descriptor set
    struct timeval tv;		// struct for time interval for select
    int buflen ;		// length of buffer
    char buf[128] ;		// input character buffer
    
    strcpy(fileout	, "/dev/stdout"	);
    strcpy(bits		, "8"		);
    strcpy(parity	, "N"		);
    strcpy(baudrate	, "9600"	);
    strcpy(tty		, "/dev/ttyS1"	);
    strcpy(NAME		, "logserial"	);
    
    /* parse command line */
    while ((c = getopt (argc, argv, "t:s:d:p:f:cvhb")) != EOF)
	{
	switch (c)
	    {
	    case 'h':				// help
		usage();
		exit(0);
		break;
	    case 'v':				// version
		version();
		exit(0);
		break;
	    case 't':				// tty
		if (strlen(optarg) > (MAXPATH-1))
		    {
		    fprintf(stderr,"Too long tty path arg!!\n");
		    exit(0);
		    }
		strcpy(tty, optarg);
		break;
	    case 's':				// speed
		if (strlen(optarg)>9)
		    {
		    fprintf(stderr,"Too long baudrate arg!!\n");
		    exit(0);
		    }
		strcpy (baudrate, optarg);
		break;
	    case 'd':				// data
		if (strlen(optarg)>9)
		    {
		    fprintf(stderr,"Too long bit number arg!!\n");
		    exit(0);
		    }
		strcpy (bits, optarg);
		break;
	    case 'p':				// parity
		if (strlen(optarg)>9)
		    {
		    fprintf(stderr,"Too long parity arg!!\n");
		    exit(0);
		    }
		strcpy (parity, optarg);
		break;
	    case 'c':				// flow control (handshake)
		swhandshake=1;
		hwhandshake=0;
		break;
            case 'f':                           // output file
                if (strlen(optarg)>(MAXPATH-1))
                    {
		    fprintf(stderr,"Too long output file arg!!\n");
		    exit(0);
		    }
		strcpy(fileout, optarg);
		break;
            case 'b':
                daemonize=1;
                break;
            }
        }

    // daemonize now if requested...

        if(daemonize==1) {
                childproc=fork();
                if(childproc==-1) {
                        perror("Unable to fork");
                        exit(-1);
                }
                if(childproc!=0) {
                        fprintf(stderr,"Started logserial daemon...\n");
                        exit(0);
                }
        }
	
    // makes new file if logfile not exists, if exists append log to it
    
    outfd = open(fileout,O_WRONLY|O_CREAT|O_APPEND,S_IRUSR|S_IWUSR|S_IRGRP|S_IWGRP|S_IROTH|S_IWOTH);

    if (outfd == -1) 
	{
	fprintf(stderr,"Can't open %s for writing.\n",fileout);
	exit(1);
	}

    portfd = open(tty,O_RDWR);
    
    if (portfd == -1) 
	{
	fprintf(stderr,"Can't open %s .\n",tty);
	close(outfd);
	exit(1);
	}
    
    m_savestate(portfd);
    
    m_setparms(portfd,baudrate,parity,bits,hwhandshake,swhandshake);
    
    m_nohang(portfd);
    
    m_hupcl(portfd,1);
    
    m_flush(portfd);

    signal (SIGHUP, &closetty);
    signal (SIGINT, &closetty);
    signal (SIGQUIT, &closetty);
    signal (SIGTERM, &closetty);
    signal (SIGABRT, &closetty);
    
    fprintf(stderr,"Successfully opened %s. Start loging.\n",tty);

    while(1)
	{
	tv.tv_sec=1  ;			// 1 second
	tv.tv_usec=0 ;
	
	FD_ZERO(&fds );
	FD_SET(portfd, &fds);
	
	if (select(portfd+1, &fds, NULL, NULL, &tv) > 0)
	    {
	    buflen = read(portfd, buf, 127);
	    if (buflen == -1) closetty(-1);
	    if (buflen > 0 )				// got input bytes
		{
		if (stat(fileout , &filestatus)==-1)	// This makes a new file if the logfile was deleted.
		    {
		    outfd = open(fileout,O_WRONLY|O_CREAT,S_IRUSR|S_IWUSR|S_IRGRP|S_IROTH);
		    if (outfd == -1) 
			{
			fprintf(stderr,"Output file was deleted, can't create new %s for writing.\n",fileout);
			close(portfd);
			exit(1);
			}
		    }
		if (write(outfd, buf, buflen) != buflen) closetty(-2) ; // and put it onto output
		fsync(outfd);
		}
	    }
	}

    return(0);
}

void closetty(int sig)
{

    switch (sig) {
	case -1:
	    fprintf(stderr,"Read error on tty. Closing communication port\n");
	    break;
	case -2:
	    fprintf(stderr,"Write error on output. Closing communication port\n");
	    break;
	default:
	    fprintf(stderr,"Got signal %d. Closing communication port\n",sig);
	    break;
	}

    m_restorestate(portfd);
    
    close(portfd);
    
    close(outfd);
    
    exit(0);
}