/* header file for dcf77 decoder */

#ifndef DCF77HDR
#define DCF77HDR

#define VERSION 1
#define SUB_VERSION 1
#define REVISION 2

/*********************************************************************
 * The DCF-77 revision log                                           * 
 *********************************************************************
 *
 *     1.0 initial version adapted from many authors.
 *         Michael Bosch and W. Stein stein@min.uni-koeln.de 
 *     1.1 added pidfile-locking and signal handler.
 *       r1 fixed minor missbehavior.
 */

/*********************************************************************
 * The DCF-77 data format                                            * 
 *********************************************************************
 *
 *       0 - bit 0.1 sec ( 100 msec ) rising flank at second start 
 *       1 - bit 0.2 sec ( 200 msec ) rising flank at second start
 *       
 *       second 0 - 58 one bit (0.1 or 0.2 sec transmission),
 *       second 59 empty (no signal) or leap, then sec. 60 will be empty !
 *
 *       0 - 14         unused (always zero)
 *      15              alternate transmitter
 *      16              expect zone change (1 hour before)
 *      17              MESZ (MED, MET DST) summer time
 *      18              MEZ  (MET)          winter time
 *      19              leap insertion/deletion (1 hour before)
 *      20              start of time code (always 1)
 *      21 - 24          1 Minutes
 *      25 - 27         10 Minutes
 *      28              Minute Parity
 *      29 - 32          1 Hours
 *      33 - 34         10 Hours
 *      35              Hour Parity
 *      36 - 39          1 Days
 *      40 - 41         10 Days
 *      42 - 44         Day of week  Sunday .... Saturday
 *      45 - 49          1 Month
 *      50              10 Months
 *      51 - 53 50-53    1 Years
 *      54 - 57         10 Years
 *      58              Date Parity
 *      59              Minute ready, except for leap insertion
 *
 *      all data are binary coded decimal
 */

/*********************************************************************
 * includes                                                          *
 *********************************************************************/

#include <stdio.h>
#include <signal.h>     /* init_sighandler(), term() */
#include <unistd.h>     /* usleep(), optarg(), unlink() ... */
#include <sys/io.h>     /* ioperm(), inb() */
#include <sys/timex.h>  /* struct timeval */

/* #include <fcntl.h> */
/* #include <sys/time.h> */
/* #include <sys/types.h> */
/* #include <sys/stat.h> */
/* #include <termios.h> */

/*********************************************************************
 * defines                                                           *
 *********************************************************************/

#ifdef LINUX_1_2
   #define TXMODE mode
   #define TXFREQ frequency
#else
   #define TXMODE modes
   #define TXFREQ freq
#endif

#ifndef TRUE
#define TRUE 0
#define FALSE !(TRUE)
#endif

#define FOREVER 1
#define UNTIL_BREAK 1

#ifndef LOGFILE
#define LOGFILE "/var/log/dcf77.log"
#endif

#ifndef PIDFILE
#define PIDFILE "/var/run/dcf77.pid"
#endif

#define POLL_DISTANCE 20000 /* usec */

#define DCF_SIGNAL 1
#define DCF_PAUSE  0

/* error values */
#define TIMEOUT -1
#define NODCFBIT -2

/* dcf-bits */
#define DCF0BIT 100000 /* \mu-sec */
#define DCF1BIT 200000 /* \mu-sec */
#define DCFWIND  30000 /* \mu-sec */

/* loop-status */
#define OFFSYNC -1

/* flags for get_bitlen() */
#define USET1 -1 /* wait only for falling flak */
#define GETALL 0 /* for both flaks rising and falling */

/* the gameport */

#ifndef PORT
#define PORT 0x201
#endif

#ifndef PORTBIT
#define PORTBIT 4
#endif

#ifndef INVERTBIT
#define INVERTBIT 1
#endif

/* macros */
#define COUNT(X) (sizeof(X)/sizeof(*(X))) /* makro to count array size */

/*********************************************************************
 * typedefs                                                          *
 *********************************************************************/

typedef struct
{
   int bit[60]; /* leap second will not be added to this !*/
} dcftype;

typedef struct
{
  int start, end, bit;
  char *desc;
} parity_type;

typedef struct
{
  int bit;
  char *desc;
} status_type;

/*********************************************************************
 * global vars                                                       *
 *********************************************************************/

struct tm timestruct;
int debug = 0;

int cont_loop = TRUE;

char *command;
int settime = TRUE;

int logging = TRUE;
char *logfile = LOGFILE;

int pid = -1;
char *pidfile = PIDFILE;

FILE *fp_data; /* pointer used to write to pid- and log-file */

long one_sec= 1000000 / POLL_DISTANCE; /* how many polls for one second */

parity_type parity[] =
{ /* parity-start, -end, pos of paritybit */
  { 21, 27, 28, "minute" },
  { 29, 34, 35, "hour" },
  { 36, 57, 58, "date" },
};

int psize= COUNT(parity);

static int zl[] = {21, 4, 3, 1, 4, 2, 1, 4, 2, 3, 4, 1, 4, 4, 1};
/* static int zb[] = { 0, 1, 1, 0, 1, 1, 0, 1, 1, 1, 1, 1, 1, 1, 0}; */
int zsize= COUNT(zl);

status_type status[] =
{
  { 15, "Alternate transmitter" },
  { 16, "Time zone change expected" },
  { 17, "Summer Time (METDST)" },
  { 18, "Winter Time (MET)" },
  { 19, "Leapsecond expected" }
};
int statusbits= COUNT(status);

/* Gameport at: (using defaults for my clock) */

unsigned port= PORT, portbit= PORTBIT, invertbit= INVERTBIT;

/*********************************************************************
 * function header                                                   *
 *********************************************************************/

void dcf_unify_time(struct timeval *e);
void dcf_sub_time(struct timeval *e, struct timeval *a, struct timeval *b);
void dcf_add_time(struct timeval *e, struct timeval *a, struct timeval *b);
void dcf_mov_time(struct timeval *e, struct timeval *a);
int dcf_sub_abs_time(struct timeval *e, struct timeval *a, struct timeval *b);

void print_time(struct timeval *t);
void print_delta(struct timeval *a,struct timeval *b);

void fprint_time(FILE *handle, struct timeval *t);
void fprint_delta(FILE *handle, struct timeval *a,struct timeval *b);

int dcf_test_parity(dcftype *info); /* do parity checking for dcf info */
int dcf_get_timeval(dcftype *info, struct timeval *t);

/* int dcf_test_parity(int *von, int *bis); */
/* void dcf_auswert(struct timeval *unixt, int *bit); */
/* void dcf_delta(struct timeval *t1, int bit); */

static inline int dcf_readbit(void);

void init_sighandler();
void term(int sig);

void dcf_poll(void);

int main(int argc, char **argv);

#endif
