/* Portable System Interfaces Copyright (c) 1997-1998 Tycho Softworks.
 * $Id: _rsln.c 1.1 Tue, 23 Jun 1998 11:38:33 -0400 dyfet $
 *
 * Permission is hear-by granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute with modifications, sub-license, and/or sell copies of the
 * software, and to permit persons to whom the Software is furnished to
 * do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * Neither the name of Tycho Softworks nor the names of it's contributors
 * may be used to endorse or promote products derived from this Software
 * without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY TYCHO SOFTWORKS "AS IS" AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL TYCHO SOFTWORKS OR IT'S
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * OR CONSEQUENTIAL DAMAGES (INCLUDING BUT NOT LIMITED TO, PROCUREMENT
 * OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
 * BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS "SOFTWARE",
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

/*
 * Read a line of input from a socket using look-ahead peaking.  This is
 * operates directly on the socket descriptor without requiring the
 * buffering routines of stdio.  Some advantage is taken with the way
 * the recv call processes input queues for a TCP/IP connection and how
 * TCP data is typically passed either in chunks when fragmented or on
 * an interractive session.  This call can be used on 'non-blocking'
 * socket descriptors and will return 'partially' received line data.
 *
 * To compensate for interactive users, a c/r and c/r l/f sequence may
 * both be used to detect 'newline' or 'end of line' condition.  To
 * support strange telnet behavior, a l/f followed by a NULL byte in
 * a single datagram can also be a newline marker.
 */

#include <std/stream.h>
#include <std/socket.h>

ssize_t	sendln(SOCKET so, char *str, size_t request)
{
	int	len = strlen(str);

	if(!request)
		request = len;

	if(len > request)
		len = request;

	return send(so, str, len, 0);
}

ssize_t	recvln(SOCKET so, char *str, size_t request)
{
	size_t	nleft = request;
	ssize_t	nstat;
	int	c;
	bool	nl = FALSE;
	char	*orig;
	bool	crlf = FALSE;

	while(nleft && !nl)
	{
		nstat = recv(so, str, nleft, MSG_PEEK);
		if(nstat <= 0)
			break;

		c = 0;
		while(c < nstat && !nl)
		{
			if(str[c] == '\r' && c < nstat - 1)
			{
				if(str[c + 1] == '\n' || !str[c + 1])
				{
					c += 2;
					nl = TRUE;
					crlf = TRUE;
					break;
				}
			}
			if(str[c] == '\n' || str[c] == '\r')
			{
				++c;
				nl = TRUE;
				break;
			}
			++c;
		}	
		nstat = readn(so, str, c);
		if(nstat <= 0)
			break;

		if(crlf)
			--nstat;

		if(nl)
			str[nstat - 1] = '\n';

		str += nstat;
		nleft -= nstat;		
	}
	*str = 0;
	return request - (ssize_t)nleft;
}
			
