#include <assert.h>
#include <ctype.h>
#include <fcntl.h>
#include <limits.h>
#include <pwd.h>
#include <stdarg.h>
#include <stdlib.h>
#include <string.h>
#include <sys/dir.h>
#include <sys/file.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <unistd.h>
#include "common.h"

void errQuit(char *format, ...) {
  char buf[256];
  va_list ap;
  va_start(ap, format);
  vsprintf(buf, format, ap);
  printf("%s\n", buf);
  if (userFile) userFileClose();
  if (banFile) banFileClose();
  va_end(ap);
  exit(1);
}

void perrQuit(char *format, ...) {
  char buf[256];
  va_list ap;
  va_start(ap, format);
  vsprintf(buf, format, ap);
  perror(buf);
  if (userFile) userFileClose();
  if (banFile) banFileClose();
  va_end(ap);
  exit(1);
}

int devNumFromName(char *devName) {
  struct stat _stat;
  char devPath[16];
  sprintf(devPath, "/dev/%s", devName);
  if (stat(devPath, &_stat) < 0)
    perrQuit("couldn't stat %s", devPath);
  return _stat.st_rdev;
}

int lineNo(dev_t dev) {
  int i;
  for (i = 0; i < nLines; i++)
    if (lineDev[i] == dev) return i;
  return -1;
}

int runCommand(char *path, ...) {
  char *arg[16];
  va_list ap;
  int status, i;
  pid_t pid;
  va_start(ap, path);
  for (i = strlen(path) - 1; i >= 0 && path[i] != '/'; i--);
  arg[0] = path + i + 1;
  for (i = 1; i < 16; i++) {
    arg[i] = va_arg(ap, char*);
    if (arg[i] == NULL) break;
  }
  if ((pid = fork()) < 0)
    perrQuit("fork");
  else if (!pid) {
    execv(path, arg);
    perrQuit("execv");
  } else waitpid(pid, &status, 0);
  va_end(ap);
  return WEXITSTATUS(status);
}

time_t addTime(char *s) {
  time_t res, t = time(NULL), endOfDay, endOfMonth, endOfYear;
  struct tm *tmP = localtime(&t);
  int len = strlen(s);
  char lastC = tolower(s[len - 1]);
  int i, j, n;
  if (!isdigit(lastC)) s[len - 1] = '\0';
  n = atoi(s);
  if (n <= 0) return 0;
  endOfDay = t +
    (60 - tmP->tm_sec) +
    60 * (59 - tmP->tm_min) +
    60 * 60 * (23 - tmP->tm_hour);
  if (isdigit(lastC) || lastC == 'd')
    return endOfDay + (n - 1) * 24 * 60 * 60;
  endOfMonth = endOfDay +
    24 * 60 * 60 * (days[tmP->tm_mon] - 1 - (tmP->tm_mday - 1));
  if (lastC == 'm') {
    for (res = endOfMonth, j = (tmP->tm_mon + 1) % 12, i = 1; i < n; i++) {
      res += days[j] * 24 * 60 * 60;
      j = (j + 1) % 12;
    }
    return res;
  }
  for (endOfYear = endOfMonth, i = tmP->tm_mon + 1; i < 12; i++)
    endOfYear += 24 * 60 * 60 * days[i];
  if (lastC == 'y')
    return endOfYear + 365 * 24 * 60 * 60 * (n - 1);
  return 0;
}

void procList(int *nProcs, ProcRec *procRec) {
  int i, li;
  FILE *f;
  char tpath[32], line[256], token[256];
  DIR *dir = opendir("/proc");
  struct dirent *_dirent;
  if (!dir) perrQuit("opendir");
  *nProcs = 0;
  while ((_dirent = readdir(dir)) != NULL) {
    if (!isdigit(_dirent->d_name[0])) continue;
    procRec[*nProcs].pid = atoi(_dirent->d_name);
#if defined(LINUX)
    sprintf(tpath, "/proc/%s/stat", _dirent->d_name);
    f = fopen(tpath, "r");
    if (!f) continue;
    if (!fgets(line, 256, f)) goto l0;
    for (li = 0, i = 0; i < 2; i++)
      if (getToken(token, line, &li)) goto l0;
    token[strlen(token) - 1] = '\0';
    strncpy(procRec[*nProcs].name, token + 1, 15);
    procRec[*nProcs].name[15] = '\0';
    for (i = 2; i < 7; i++)
      if (getToken(token, line, &li)) goto l0;
    procRec[*nProcs].tty = atoi(token);
    fclose(f);
    sprintf(tpath, "/proc/%s/status", _dirent->d_name);
    f = fopen(tpath, "r");
    if (!f) continue;
    for (i = 0; i < 5; i++)
      if (!fgets(line, 256, f)) goto l0;
    li = 0;
    if (getToken(token, line, &li)) goto l0;
    if (getToken(token, line, &li)) goto l0;
    procRec[*nProcs].uid = atoi(token);
    if (getToken(token, line, &li)) goto l0;
    procRec[*nProcs].euid = atoi(token);
#elif defined(BSD)
    sprintf(tpath, "/proc/%s/status", _dirent->d_name);
    f = fopen(tpath, "r");
    if (!f) continue;
    if (!fgets(line, 256, f)) goto l0;
    li = 0;
    if (getToken(token, line, &li)) goto l0;
    strncpy(procRec[*nProcs].name, token, 15);
    procRec[*nProcs].name[15] = '\0';
    for (i = 1; i < 6; i++)
      if (getToken(token, line, &li)) goto l0;
    procRec[*nProcs].tty = atoi(token) * 256 + atoi(strchr(token, ',') + 1);
    for (; i < 12; i++)
      if (getToken(token, line, &li)) goto l0;
    procRec[*nProcs].euid = atoi(token);
    if (getToken(token, line, &li)) goto l0;
    procRec[*nProcs].uid = atoi(token);
#endif
    (*nProcs)++;
l0:
    fclose(f);
  }
  closedir(dir);
}

int maxKick(int nProcs, ProcRec *procRec) {
  int i, n;
  byte lineBusy[MAX_USERS];
  bzero(lineBusy, MAX_USERS);
  for (i = 0; i < nProcs; i++) {
    n = lineNo(procRec[i].tty);
    if (n >= 0) lineBusy[n] = 1;
  }
  for (n = 0, i = 0; i < nLines; i++)
    if (!lineBusy[i]) n++;
  if (optMaxKick) {
    n = optMaxKick - n;
    return max(0, n);
  } else return !n;
}

void userList(int *nLogins, LoginRec *loginRec) {
  struct utmp _utmp;
  uid_t uid;
  char login[MAX_LOGINCHARS + 1];
  int i;
  FILE *f = fopen(_PATH_UTMP, "rb");
  if (!f) perrQuit("fopen");
  *nLogins = 0;
  while (fread(&_utmp, sizeof(struct utmp), 1, f)) {
#if defined(LINUX)
    if (_utmp.ut_type != USER_PROCESS) continue;
    strncpy(login, _utmp.ut_user, MAX_LOGINCHARS);
#elif defined(BSD)
    strncpy(login, _utmp.ut_name, MAX_LOGINCHARS);
#endif
    for (i = 0; i < MAX_LOGINCHARS && isgraph(login[i]); i++);
    login[i] = '\0';
    if ((uid = UIDfromLogin(login)) == (uid_t)-1) continue;
    loginRec[*nLogins].uid = uid;
    strncpy(loginRec[*nLogins].tty, _utmp.ut_line, 11);
    loginRec[*nLogins].tty[11] = '\0';
    loginRec[(*nLogins)++].time = _utmp.ut_time;
  }
  fclose(f);
}

int userLoggedIn(int nLogins, LoginRec *loginRec, uid_t uid) {
  int i;
  for (i = 0; i < nLogins; i++) if (loginRec[i].uid == uid) return 1;
  return 0;
}

int getToken(char *d, char *s, int *idx) {
  int si, di, sLen = strlen(s);
  for (si = *idx; si < sLen && isspace(s[si]); si++);
  for (di = 0; isgraph(s[si]); si++) d[di++] = s[si];
  d[di] = '\0';
  *idx = si;
  return di == 0;
}

uid_t UIDfromLogin(char *login) {
  uid_t res = (uid_t)-1;
  struct passwd *passwdp;
  if ((passwdp = getpwnam(login)) != NULL)
    res = passwdp->pw_uid;
  return res;
}

char *loginFromUID(char *login, uid_t uid) {
  char *res = NULL;
  struct passwd *passwdp;
  login[0] = '\0';
  if ((passwdp = getpwuid(uid)) != NULL) {
    strncpy(login, passwdp->pw_name, MAX_LOGINCHARS);
    login[MAX_LOGINCHARS] = '\0';
    res = login;
  }
  return res;
}

void userFileOpen() {
  if (!geteuid()) {
    fLock((char*)userFilePath, LOCK_EXCLUSIVE | LOCK_BLOCK);
    userFileFD = open(userFilePath, O_CREAT | O_RDWR, 0644);
    assert(userFileFD >= 0);
    flock(userFileFD, LOCK_EX);
    fcntl(userFileFD, F_SETFD, 1);
    userFile = fdopen(userFileFD, "r+b");
  } else {
    userFileFD = open(userFilePath, O_RDONLY);
    assert(userFileFD >= 0);
    flock(userFileFD, LOCK_SH);
    userFile = fdopen(userFileFD, "rb");
  }
}

void userFileClose() {
  flock(userFileFD, LOCK_UN);
  if (!geteuid()) fLock((char*)userFilePath, LOCK_UNLOCK);
  fclose(userFile);
}

int userFileRead(UserRec *ur) {
  return !fread(ur, sizeof(UserRec), 1, userFile);
}

int userFileWrite(UserRec *ur) {
  return !fwrite(ur, sizeof(UserRec), 1, userFile);
}

void userFileEdit(UserRec *ur) {
  fseek(userFile, -sizeof(UserRec), SEEK_CUR);
  userFileWrite(ur);
}

int userFileSearch(UserRec *ur, uid_t uid) {
  rewind(userFile);
  while (!userFileRead(ur))
    if (ur->uid == uid)
      return 0;
  return 1;
}

void banFileOpen() {
  if (geteuid())
    errQuit("must be root");
  if (!geteuid()) fLock((char*)banFilePath, LOCK_EXCLUSIVE | LOCK_BLOCK);
  banFileFD = open(banFilePath, O_CREAT | O_RDWR, 0644);
  assert(banFileFD >= 0);
  if (!geteuid()) flock(banFileFD, LOCK_EX);
  else flock(banFileFD, LOCK_SH);
  fcntl(banFileFD, F_SETFD, 1);
  banFile = fdopen(banFileFD, "r+b");
}

void banFileClose() {
  flock(banFileFD, LOCK_UN);
  if (!geteuid()) fLock((char*)banFilePath, LOCK_UNLOCK);
  fclose(banFile);
}

int banFileRead(BanRec *br) {
  return !fread(br, sizeof(BanRec), 1, banFile);
}

int banFileWrite(BanRec *br) {
  return !fwrite(br, sizeof(BanRec), 1, banFile);
}

int banFileSearch(BanRec *br, word phNoArea, word phNoLocal) {
  char phNoStr[11], phNoAreaStr[11], phNoLocalStr[11];
  sprintf(phNoAreaStr, "%u", phNoArea);
  sprintf(phNoLocalStr, "%u", phNoLocal);
  rewind(banFile);
  while (!banFileRead(br)) {
    if (br->flags) {
      sprintf(phNoStr, "%u", br->phNoArea);
      if ((phNoArea && !strncmp(phNoStr, phNoAreaStr, strlen(phNoStr))) ||
	  (phNoLocal && !strncmp(phNoStr, phNoLocalStr, strlen(phNoStr))) ||
	  (!phNoArea && !phNoLocal && !br->phNoArea && !br->phNoLocal))
	return 0;
    } else if (br->phNoArea == phNoArea && br->phNoLocal == phNoLocal)
      return 0;
  }
  return 1;
}

int phNo2Words(word *phNoArea, word *phNoLocal, char *phNo, int phNoDigits) {
  int i, len;
  char tPhNo[32], phNoAreaStr[32], phNoLocalStr[32];
  for (len = 0, i = 0; phNo[i] != '\0'; i++)
    if (isdigit(phNo[i]))
      tPhNo[len++] = phNo[i];
  tPhNo[len] = '\0';
  if (!len || len > 18) return 1;
  if (len < phNoDigits) {
    strcpy(phNoAreaStr, tPhNo);
    strcpy(phNoLocalStr, "0");
  } else {
    strcpy(phNoLocalStr, tPhNo + len - phNoDigits);
    strcpy(phNoAreaStr, tPhNo); phNoAreaStr[len - phNoDigits] = '\0';
  }
  *phNoArea = atol(phNoAreaStr);
  *phNoLocal = atol(phNoLocalStr);
  return 0;
}

char *words2PhNo(char *phNo, word phNoArea, word phNoLocal) {
  int i = 0, j = 0, k, l, skipChars;
  char phNoAreaStr[10], phNoLocalStr[10], *s;
  sprintf(phNoAreaStr, "%u", phNoArea);
  sprintf(phNoLocalStr, "%u", phNoLocal);
  if (!phNoLocal) {
    strcpy(phNo, phNoAreaStr);
    return phNo;
  }
  if (phNoArea) {
    s = phNoAreaFormat;
    skipChars = stringSum(phNoAreaFormat) - strlen(phNoAreaStr);
    while (*s != '\0') {
      if (isdigit(*s)) {
	l = strtol(s, &s, 10);
	if (skipChars) {
	  skipChars -= l;
	  if (skipChars == 0)
	    while (*s == ')' || *s == ' ') s++;
	  else if (skipChars < 0) goto err;
	  continue;
	}
	for (k = 0; k < l; k++)
	  phNo[i++] = phNoAreaStr[j++];
      } else while (!isdigit(*s) && *s != '\0')
	if (skipChars) s++;
	else phNo[i++] = *s++;
    }
    if (phNoAreaStr[j] != '\0') goto err;
    phNo[i++] = ' ';
  }
  if (phNoLocal) {
    j = 0;
    s = phNoLocalFormat;
    while (*s != '\0') {
      if (isdigit(*s)) {
	l = strtol(s, &s, 10);
	for (k = 0; k < l; k++)
	  phNo[i++] = phNoLocalStr[j++];
      } else while (!isdigit(*s) && *s != '\0')
	phNo[i++] = *s++;
    }
    if (phNoLocalStr[j] != '\0') goto err;
  }
  phNo[i] = '\0';
  return phNo;
err:
  phNo[0] = '\0';
  return NULL;
}

int stringSum(char *s) {
  int sum = 0;
  while (*s != '\0') {
    while (!isdigit(*s) && *s != '\0') s++;
    if (*s != '\0') sum += strtol(s, &s, 10);
  }
  return sum;
}

void bytes2ASCII(char *str, dword bytes) {
  if (bytes >= 1024 * 1024 * 1024) sprintf(str, "%.2f GB", (float)bytes / (1024 * 1024 * 1024));
  else if (bytes >= 1024 * 1024) sprintf(str, "%.2f MB", (float)bytes / (1024 * 1024));
  else sprintf(str, "%.2f KB", (float)bytes / 1024);
}

int inTimeClass(int tc) {
  time_t t;
  struct tm *tmP;
  time(&t);
  tmP = localtime(&t);
  if (timeClass[tc].startDay <= timeClass[tc].endDay) {
    if ((tmP->tm_wday < timeClass[tc].startDay) ||
	(tmP->tm_wday > timeClass[tc].endDay)) return 0;
  } else {
    if ((tmP->tm_wday > timeClass[tc].endDay) &&
	(tmP->tm_wday < timeClass[tc].startDay)) return 0;
  }
  if (timeClass[tc].startHour <= timeClass[tc].endHour) {
    if ((tmP->tm_hour < timeClass[tc].startHour) ||
	(tmP->tm_hour > timeClass[tc].endHour)) return 0;
  } else {
    if ((tmP->tm_hour > timeClass[tc].endHour) &&
	(tmP->tm_hour < timeClass[tc].startHour)) return 0;
  }
  return 1;
}

int curTimeClass(UserRec *ur) {
  int i, tc = -1, tcLeft = 0;
  for (i = 0; i < nTimeClasses; i++) {
    if (inTimeClass(i)) {
      if (!ur) return i;
      if (tc < 0 || (ur->cLimit[i] >= 0 &&
          (ur->cLimit[tc] < 0 || ur->cLeft[i] < tcLeft)))
      {
	tc = i;
	tcLeft = ur->cLeft[i];
      }
    }
  }
  if (tc >= 0) return tc;
  else return -1;
}

/*
  This only does X-locks.  S-lock implementation is left as an exercise
  for the reader :)
  */
int fLock(char *path, int operation) {
  pid_t myPID = getpid(), theirPID;
  int nProcs;
  ProcRec procRec[MAX_PROCS];
  char *s, hostname[64], theirHostname[64];
  char myName[16] = "", theirName[16];
  char buf[PATH_MAX];
  char linkPath[PATH_MAX];
  char pidPath[PATH_MAX], theirPIDpath[PATH_MAX];
  int i, len = strlen(path), dirChars;
  char *name;
  FILE *f;
  int block = operation & LOCK_BLOCK;
  if (block) operation &= ~LOCK_BLOCK;
  procList(&nProcs, procRec);
  for (i = 0; i < nProcs; i++)
    if (procRec[i].pid == myPID) {
      strcpy(myName, procRec[i].name);
      break;
    }
  for (i = len - 1; i; i--)
    if (path[i] == '/') break;
  if (i < 0) name = path;
  else name = path + i + 1;
  dirChars = name - path;
  strncpy(linkPath, path, dirChars); linkPath[dirChars] = '\0';
  sprintf(linkPath + dirChars, "LOCK_%s", name);
  strncpy(pidPath, path, dirChars); pidPath[dirChars] = '\0';
  gethostname(hostname, 64);
  for (s = hostname; *s && *s != '.'; s++);
  if (*s == '.') *s = '\0';
  sprintf(pidPath + dirChars, "LOCK_%s.%s.%u", name, hostname, myPID);
  if (operation == LOCK_UNLOCK) {
    unlink(pidPath);
    unlink(linkPath);
    return 0;
  }
  f = fopen(pidPath, "w");
  fprintf(f, "%s %s %u\n", hostname, myName, myPID);
  fclose(f);
l0:
  if (symlink(pidPath + dirChars, linkPath) < 0) {
    i = readlink(linkPath, buf, PATH_MAX);
    if (i < 0) goto l0;
    else {
      buf[i] = '\0';
      strncpy(theirPIDpath, path, dirChars); theirPIDpath[dirChars] = '\0';
      strcpy(theirPIDpath + dirChars, buf);
    }
    f = fopen(theirPIDpath, "r");
    if (!f) {
      unlink(linkPath);
      goto l0;
    }
    i = fscanf(f, "%s %s %u", theirHostname, theirName, &theirPID);
    fclose(f);
    if (i < 3) {
      unlink(theirPIDpath);
      unlink(linkPath);
      goto l0;
    }
    if (strcmp(hostname, theirHostname)) goto l1;
    procList(&nProcs, procRec);
    for (i = 0; i < nProcs; i++)
      if (procRec[i].pid == theirPID && !strcmp(procRec[i].name, theirName))
	break;
    if (i == nProcs) {
      unlink(theirPIDpath);
      unlink(linkPath);
      goto l0;
    } else goto l1;
  } else return 0;
l1:
  if (block) {
    sleep(1);
    goto l0;
  }
  unlink(pidPath);
  return 1;
}

void processConfig() {
  int i, li;
  char line[256], token[256], devPath[16];
  FILE *f;
  struct stat _stat;
  if ((f = fopen(configFilePath, "rt")) == NULL)
    errQuit("couldn't open config file");
  nLines = 0;
  optReturnDelay = 0;
  optSendMail = 0;
  optGuestTime = 15;
  optGuestPriority = 4;
  nBootWarnTimes = 0;
  optIdleBoot = 0;
  optSmartBoot = 0; optMaxKick = 0;
  optSessionSmartBoot = 0;
  nTimeClasses = 0;
  optTimeClassSmartBoot = 0;
  while (fgets(line, 256, f) != NULL) {
    li = 0;
    if (getToken(token, line, &li)) continue;
    if (!strcasecmp(token, "Devices")) {
      for (;;) {
	if (getToken(token, line, &li)) break;
	if (isdigit(token[0])) {
	  int n = atoi(token) - 1;
	  if (nLines)
	    for (i = 0; i < n; i++) {
	      lineDev[nLines] = lineDev[nLines - 1] + 1;
              nLines++;
            }
	} else {
	  sprintf(devPath, "/dev/%s", token);
	  if (stat(devPath, &_stat) < 0)
	    perrQuit("couldn't stat %s", devPath);
	  lineDev[nLines++] = _stat.st_rdev;
	}
      }
    } else if (!strcasecmp(token, "ReturnDelay")) {
      if (getToken(token, line, &li)) continue;
      optReturnDelay = atoi(token);
    } else if (!strcasecmp(token, "SendMail")) {
      optSendMail = 1;
    } else if (!strcasecmp(token, "GuestTime")) {
      if (getToken(token, line, &li)) continue;
      optGuestTime = atoi(token);
    } else if (!strcasecmp(token, "GuestPriority")) {
      if (getToken(token, line, &li)) continue;
      optGuestPriority = atoi(token);
    } else if (!strcasecmp(token, "WarnBoot")) {
      while (!getToken(token, line, &li) && nBootWarnTimes < MAX_BOOT_WARN_TIMES)
	bootWarnTime[nBootWarnTimes++] = atoi(token);
    } else if (!strcasecmp(token, "SmartBoot")) {
      optSmartBoot = 1;
    } else if (!strcasecmp(token, "MaxKick")) {
      if (getToken(token, line, &li)) continue;
      optMaxKick = atoi(token);
    } else if (!strcasecmp(token, "IdleBoot")) {
      if (getToken(token, line, &li)) continue;
      optIdleBoot = atoi(token);
    } else if (!strcasecmp(token, "IdleSmartBoot")) {
      optIdleSmartBoot = 1;
    } else if (!strcasecmp(token, "SessionSmartBoot")) {
      optSessionSmartBoot = 1;
    } else if (!strcasecmp(token, "TimeClass")) {
      for (i = li; line[i]; i++)
	if (!isdigit(line[i])) line[i] = ' ';
      for (;;) {
	if (getToken(token, line, &li)) break;
	timeClass[nTimeClasses].startDay = atoi(token);
	if (getToken(token, line, &li)) break;
	timeClass[nTimeClasses].endDay = atoi(token);
	if (getToken(token, line, &li)) break;
	timeClass[nTimeClasses].startHour = atoi(token);
	if (getToken(token, line, &li)) break;
	timeClass[nTimeClasses].endHour = atoi(token) - 1;
        if (timeClass[nTimeClasses].endHour == -1)
          timeClass[nTimeClasses].endHour = 23;
        nTimeClasses++;
      }
    } else if (!strcasecmp(token, "TimeClassSmartBoot")) {
      optTimeClassSmartBoot = 1;
    } else if (!strcasecmp(token, "ForeachExclude")) {
      while (!getToken(token, line, &li)) {
        excluded[nExcluded] = UIDfromLogin(token);
        if (excluded[nExcluded] == (uid_t)-1) continue;
        else nExcluded++;
      }
    } else if (!strcasecmp(token, "ModemDial")) {
      if (getToken(token, line, &li)) continue;
      strcpy(modemDial, token);
    } else if (!strcasecmp(token, "PhNoAreaFormat")) {
      if (getToken(token, line, &li)) continue;
      strcpy(phNoAreaFormat, token);
      while (!getToken(token, line, &li)) {
	strcat(phNoAreaFormat, " ");
	strcat(phNoAreaFormat, token);
      }
    } else if (!strcasecmp(token, "PhNoLocalFormat")) {
      if (getToken(token, line, &li)) continue;
      strcpy(phNoLocalFormat, token);
      while (!getToken(token, line, &li)) {
	strcat(phNoLocalFormat, " ");
	strcat(phNoLocalFormat, token);
      }
      phNoDigits = stringSum(phNoLocalFormat);
    }
  }
  fclose(f);
}
