/*
 * This file is a part of the xnetsentry project.
 * Copyright (C) 1998 Martin Gall
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
/*
 *
 */

#include <ctype.h>
#include <netdb.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include "miscinet.h"

int			port_from_str(str,proto,resolve)
char			*str;
char			*proto;
t_boolean		resolve;
{
  if (resolve)
    {
      struct servent	*serventry;

      setservent(0);
      if (serventry = getservbyname(str,proto))
	{
	  int		port;
	  
	  port = serventry->s_port;

	  endservent();
	  return (port);
	}
      endservent();
    }
  if (uint_match(str))
    return (atoi(str));
  return (-ERR_MALFORMED);
}

t_status		port_to_str(port,proto,str,max_len,resolve)
int			port;
char			*proto;
char			*str;
int			max_len;
t_boolean		resolve;
{
  if (resolve)
    {
      struct servent	*serventry;

      setservent(0);
      if (serventry = getservbyport(htons(port),proto))
	{
	  char		*name;
	  
	  name = serventry->s_name;
	  endservent();
	  return (str_cat_str(str,max_len,name));
	}
      endservent();
    }
  return (int_to_str(port,str,max_len));
}

int			proto_from_str(str,resolve)
char			*str;
t_boolean		resolve;
{
  if (resolve)
    {
      struct protoent	*protoentry;

      setprotoent(0);
      if (protoentry = getprotobyname(str))
	{
	  int		proto;
	  
	  proto = protoentry->p_proto;
	  endprotoent();
	  return (proto);
	}
      endprotoent();
    }
  if (uint_match(str))
    return (atoi(str));
  return (-ERR_MALFORMED);
}

t_status		proto_to_str(proto,str,max_len,resolve)
int			proto;
char			*str;
int			max_len;
t_boolean		resolve;
{
  if (resolve)
    {
      struct protoent	*protoentry;

      setprotoent(0);
      if (protoentry = getprotobynumber(proto))
	{
	  char		*name;
	  
	  name = protoentry->p_name;
	  endprotoent();
	  return (str_cat_str(str,max_len,name));
	}
      endprotoent();
    }
  return (int_to_str(proto,str,max_len));
}

t_status		inaddr_from_str(str,mba_inaddr,resolve)
char			*str;
struct in_addr		*mba_inaddr;
t_boolean		resolve;
{
  struct hostent	*hostentry;
  
  mba_inaddr->s_addr = inet_addr(str);
  if (mba_inaddr->s_addr == -1)
    {
      if (resolve)
	{
	  if ((hostentry = gethostbyname(str)) == NULL)
	    return (-ERR_UNRESOLVABLE);
	  if (hostentry->h_addrtype != AF_INET)
	    return (-ERR_NOMETHOD);
	  bcopy(hostentry->h_addr_list[0],(char *)(mba_inaddr),
		hostentry->h_length);
	}
      else
	return (-ERR_MALFORMED);
    }
  return (0);
}

t_status		inaddr_to_str(mba_inaddr,str,max_len,resolve)
struct in_addr		*mba_inaddr;
char			*str;
int			max_len;
t_boolean		resolve;
{
  struct hostent	*hostentry;
  
  if (resolve)
    if ((hostentry = gethostbyaddr((char *)mba_inaddr,
				   sizeof (struct in_addr),
				   AF_INET)) != NULL)
      return (str_cat_str(str,max_len,hostentry->h_name));
  return (str_cat_str(str,max_len,inet_ntoa(*mba_inaddr)));
}

int			in_class_default_nshift(addr)
struct in_addr		*addr;
{
  if (IN_CLASSA(ntohl(addr->s_addr)))
    return (IN_CLASSA_NSHIFT);
  if (IN_CLASSB(ntohl(addr->s_addr)))
    return (IN_CLASSB_NSHIFT);
  if (IN_CLASSC(ntohl(addr->s_addr)))
    return (IN_CLASSC_NSHIFT);
  return (0);
}

t_status		subnet_from_str(str,subnet)
char			*str;
t_subnet		*subnet;
{
  char			*bits_str;
  int			nshift;
  char			*mask_str;
  
  if (indexcount(str,'.') != 3)
    return (-ERR_MALFORMED);
  mask_str = NULL;
  if (bits_str = index(str,'/'))
    {
      *bits_str++ = 0;
    }
  else
    if (mask_str = index(str,':'))
      {
	*mask_str++ = 0;
      }
  subnet->addr.s_addr = inet_addr(str);
  if (mask_str)
    {
      subnet->mask.s_addr = inet_addr(mask_str);
    }
  else
    {
      if (bits_str)
	nshift = 32 - atoi(bits_str);
      else
	nshift = in_class_default_nshift(&(subnet->addr));
      subnet->mask.s_addr = ~0 << nshift;
      subnet->mask.s_addr = htonl(subnet->mask.s_addr);
    }
  subnet->cable.s_addr = subnet->addr.s_addr & subnet->mask.s_addr;
  subnet->broadcast.s_addr = subnet->addr.s_addr | (~subnet->mask.s_addr);
  return (0);
}

t_boolean		is_subnet_member(inaddr,subnet)
struct in_addr		*inaddr;
t_subnet		*subnet;
{
  int			hip;
  int			hmask;
  int			hcable;
    
  hip = ntohl(inaddr->s_addr);
  hmask = ntohl(subnet->mask.s_addr);
  hcable = ntohl(subnet->cable.s_addr); 
  return ((hip & hmask) == hcable);
}

#ifdef DEBUG
VOID_FUNC		subnet_show(subnet)
t_subnet		*subnet;
{
  fprintf(stderr,"addr=%s\n",inet_ntoa(subnet->addr));
  fprintf(stderr,"mask=%s\n",inet_ntoa(subnet->mask));
  fprintf(stderr,"cable=%s\n",inet_ntoa(subnet->cable));
  fprintf(stderr,"broadcast=%s\n",inet_ntoa(subnet->broadcast));
}
#endif

/* WARNING: DON'T WORK WITH HOLES IN NETMASK */ 
t_status		subnet_walk(subnet,proc,data)
t_subnet		*subnet;
t_subnet_walk_proc	proc;
VOID_PTR		data;
{
  int			s;
  int			haddr;
  int			hmask;
  int			hcable;
  int			hbroadcast;
  t_status		status;

  haddr = ntohl(subnet->addr.s_addr);
  hmask = ntohl(subnet->mask.s_addr);
  hcable = ntohl(subnet->cable.s_addr);
  hbroadcast = ntohl(subnet->broadcast.s_addr);
  s = haddr;
  while ((s & hmask) == hcable) 
    {
      struct in_addr	addr;
      int		flag;

      addr.s_addr = htonl(s);
      if (s == hcable)
	flag = SUBNET_CABLE;
      else
	if (s == hbroadcast)
	  flag = SUBNET_BROADCAST;
	else
	  flag = SUBNET_ADDR;
      if ((status = proc(&addr,flag,data)) < 0)
	return (status);
      s++;
    }
  return (0);
} 

int			inaddr_cmp(addr1,addr2)
struct in_addr		*addr1;
struct in_addr		*addr2;
{
  unsigned int		saddr1;
  unsigned int		saddr2;
  
  saddr1 = (unsigned int)(htonl(addr1->s_addr));
  saddr2 = (unsigned int)(htonl(addr2->s_addr));
  return ((int)(saddr1 - saddr2));
}
