/*  sas - sane authentication scheme.
    Copyright (C) 2001 Quentin Sharpe <qsharpe@btinternet.com>.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111, USA.
*/
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <ctype.h>
#include <pwd.h>
#include <grp.h>
#include <sys/types.h>
#include "confread.h"
#include "global.h"

#define MAX_PASSWD_LINE_LEN 500	// number of characters on a passwd file line
#define MAX_PASSWD_LINES 100	// number of lines in a passwd file
#define MAX_PASSWD_FIELD 100	// number of characters in a passwd field
#define MAX_GROUP_FIELD 150	// number of characters in a group field
#define MAX_GROUP_MEMBERS 20    // number of group members

void bk_passwd_warn(char *error_string,int line_num,char *line_string,int error_level);
void bk_group_warn(char *error_string,int line_num,char *line_string,int error_level);
char *chomp_str(char *crlf_str);


int trigger_safe;

void sys_group_check(char *group_file)
{
	FILE *group;
	struct group *group_fields;
	int line=1,w,z;
	
	if((group=fopen(group_file,"r"))==NULL) {
		fprintf(stderr,"%s: %s: %s\n",__progname,"could not open file",group_file);
		exit(1);
	}
	
	printf("\n%s: \n\n",group_file);
	
	while((group_fields=fgetgrent(group))) {
		// group_name
		if(strlen(group_fields->gr_name)==0) {
			bk_group_warn("nameless group",line,group_fields->gr_name,INFO);
		}
		// passwd
		if(strlen(group_fields->gr_passwd)==0) {
			bk_group_warn("null password",line,group_fields->gr_name,WARN);
		}		
		if(xshadowed_passwords) {
			if(strcmp(group_fields->gr_passwd,"x")!=0) {
				bk_group_warn("unshadowed password",line,group_fields->gr_name,WARN);
			}		
		}
		// gid
	 	if(group_fields->gr_gid==0) {
	 		if(DEBUG_MODE_ON) {
	 			bk_group_warn("gid evaluates to 0",line,group_fields->gr_name,DEBUG);
			}
			w=0;
			while(group_fields->gr_mem[w]) {
				trigger_safe=0;
				for(z=0; z<xnum_allow_gid; z++) {
						if(strcmp(group_fields->gr_mem[w],xallow_gid[z])==0) {
							trigger_safe=1;
						}		
				}
				if(!trigger_safe) {
					bk_group_warn("unauthorised gid 0",line,group_fields->gr_name,EMERG);
				}
				w++;					
			}			
        	}
        	
		
		// members - see above
		
		line++;
        }

        fclose(group);
}

void sys_passwd_check(char *passwd_file)
{
	FILE *passwd;
	struct passwd *passwd_fields;
	int line=1,w;
	
	if((passwd=fopen(passwd_file,"r"))==NULL) {
		fprintf(stderr,"%s: %s: %s\n",__progname,"could not open file",passwd_file);
		exit(1);
	}
	
	printf("\n%s:\n\n",passwd_file);
	
	while((passwd_fields=fgetpwent(passwd))) {
        	// username
	 	if(strlen(passwd_fields->pw_name)==0) {
	 		bk_passwd_warn("no username",line,passwd_fields->pw_name,INFO);
	 	}
	 	// password
	 	if(strlen(passwd_fields->pw_passwd)==0) {
	 		bk_passwd_warn("null password",line,passwd_fields->pw_name,WARN);
	 	}
	 	if(xshadowed_passwords) {
	 		if(strcmp(passwd_fields->pw_name,"x")==0) {
	 			bk_passwd_warn("unshadowed password",line,passwd_fields->pw_name,WARN);
	 		}
	 	}	 	
	 	// uid
	 	trigger_safe=0;
	 	if(passwd_fields->pw_uid==0) {
	 		if(DEBUG_MODE_ON) bk_passwd_warn("uid evaluates to 0",line,passwd_fields->pw_name,DEBUG);
	 		for(w=0; w<xnum_allow_uid; w++) {
	 			if(strcmp(passwd_fields->pw_name,xallow_uid[w])==0) {
	 				trigger_safe=1;	
	 			}
	 		}
	 		if(!trigger_safe) bk_passwd_warn("unauthorized uid 0",line,passwd_fields->pw_name,EMERG);	 		
	 	}
	 	// gid
	 	trigger_safe=0;
	 	if(passwd_fields->pw_gid==0) {
	 		if(DEBUG_MODE_ON) bk_passwd_warn("gid evaluates to 0",line,passwd_fields->pw_name,DEBUG);
	 		for(w=0; w<xnum_allow_gid; w++) {
	 			if(strcmp(passwd_fields->pw_name,xallow_gid[w])==0) {
	 				trigger_safe=1;	
	 			}
	 		}
	 		if(!trigger_safe) bk_passwd_warn("unauthorized gid 0",line,passwd_fields->pw_name,EMERG);
	 	}	 	
	 	// GECOS
	 	if(strlen(passwd_fields->pw_gecos)==0) {
	 		bk_passwd_warn("no gecos field",line,passwd_fields->pw_name,INFO);
	 	}	 	
	 	
	 	// home
	 	if(strlen(passwd_fields->pw_dir)==0) {
	 		bk_passwd_warn("no home directory",line,passwd_fields->pw_name,INFO);
	 	}
	 	// shell
	 	if(strlen(passwd_fields->pw_shell)==0) {
	 		bk_passwd_warn("no shell",line,passwd_fields->pw_name,INFO);
	 	}
	 	line++;	 	
	}
	
	fclose(passwd);
}

void passwd_check(char *passwd_file)
{
	FILE *passwd;
	int i=0,j,c,passwd_field_count,w,x,y,z;
	char passwd_line[MAX_PASSWD_LINES][MAX_PASSWD_LINE_LEN];
	char temp[MAX_PASSWD_LINE_LEN+7];
	char *p,*t,*q;
	// char passwd_fields[LINES][FIELDS][LENGTH];	
	char passwd_fields[MAX_PASSWD_LINES][10][MAX_PASSWD_FIELD];
	
	if((passwd=fopen(passwd_file,"r"))==NULL) {
		fprintf(stderr,"%s: %s: %s\n",__progname,"could not open file",passwd_file);
		exit(1);
	}

	while(!feof(passwd)) {
		if(i >= MAX_PASSWD_LINES) {
			fprintf(stderr,"%s: %s\n",__progname,"too many lines, recompile with greater MAX_PASSWD_LINES");
			exit(1);
		}
		if(fgets(passwd_line[i],(MAX_PASSWD_LINE_LEN-2),passwd)) i++;
		else break;
	}

	fclose(passwd);
	
	trigger_safe=0;
	
	printf("\n%s:\n\n",passwd_file);

	for(j=0; j<i; j++) {
		passwd_field_count=0;
		sprintf(passwd_line[j],"%s",chomp_str(passwd_line[j]));
		CHECK_NULL_PWD:
		q=passwd_line[j];
                if((t=strstr(q,"::")) != NULL) {
                	bk_passwd_warn("null field",j+1,passwd_line[j],DEBUG);
                	sprintf(temp,"%s",passwd_line[j]);
                	x=strlen(temp);
                	y=strlen(t);
                	z=x-y; // index of "::"
                	temp[z+1]=' ';
                	for(w=z+1; w<x+1; w++) {
                		temp[w+1]=passwd_line[j][w];
                	}
                	sprintf(passwd_line[j],"%s",temp);
                	goto CHECK_NULL_PWD;             	
                }

                if(passwd_line[j][(strlen(passwd_line[j])-1)]==':') { // null last field
                	bk_passwd_warn("null field",j+1,passwd_line[j],DEBUG);
                	x=strlen(passwd_line[j]);
                	passwd_line[j][x]=' ';
                	passwd_line[j][x+1]='\0';
                }
                if(passwd_line[j][0]==':') { // null first field
                	bk_passwd_warn("null field",j+1,passwd_line[j],DEBUG);
                	sprintf(temp,"%s",passwd_line[j]);
                	x=strlen(temp);
                	y=x-1;
                	z=x-y;
                	temp[0]=' ';
                	for(w=0; w<x+1; w++) {
                		temp[w+1]=passwd_line[j][w];
                	}
                	sprintf(passwd_line[j],"%s",temp);
                }
		
		
                sprintf(temp,"%s",passwd_line[j]);
		p=strtok(temp,":");
		if(strlen(p) > (MAX_PASSWD_FIELD-2)) {
			fprintf(stderr,"%s: %s: %s\n",__progname,"password field too big","recompile with larger MAX_PASSWD_FIELD");
			exit(1);
		}
		sprintf(passwd_fields[j][0],"%s",p);
		passwd_field_count++;
		do {
			p=strtok('\0',":");
			if(p) {
				if(strlen(p) > (MAX_PASSWD_FIELD-2)) {
					fprintf(stderr,"%s: %s: %s\n",__progname,"password field too big","recompile with larger MAX_PASSWD_FIELD");
					exit(1);
				}
				sprintf(passwd_fields[j][passwd_field_count],"%s",p);
				passwd_field_count++;
			}
		} while(p);
		if(passwd_field_count > 7) {
			bk_passwd_warn("too many fields (or colons)",j+1,passwd_line[j],WARN);
		}
		else if(passwd_field_count < 7) {
			bk_passwd_warn("fields missing (or missing colons)",j+1,passwd_line[j],WARN);
		}
	}
	if(VERBOSE_OUTPUT) {
		for(c=0; c<i; c++) {
			printf("[%d] \'%s\' \'%s\' \'%s\' \'%s\' \'%s\' \'%s\' \'%s\'\n",c+1,passwd_fields[c][0],passwd_fields[c][1],passwd_fields[c][2],passwd_fields[c][3],passwd_fields[c][4],passwd_fields[c][5],passwd_fields[c][6]);
		}
	}
	for(c=0; c<i; c++) {
	 	// username
	 	if(strlen(passwd_fields[c][0])==1 && passwd_fields[c][0][0]==' ') {
	 		bk_passwd_warn("no username",c+1,passwd_line[c],INFO);
	 	}
	 	// password
	 	if(strlen(passwd_fields[c][1])==1 && passwd_fields[c][1][0]==' ') {
	 		bk_passwd_warn("null password",c+1,passwd_line[c],WARN);
	 	}
	 	if(xshadowed_passwords) {
	 		if(strlen(passwd_fields[c][1]) != 1 || passwd_fields[c][1][0] != 'x') {
	 			bk_passwd_warn("unshadowed password",c+1,passwd_line[c],WARN);
	 		}
	 	}	 	
	 	// uid
	 	for(w=0; w<strlen(passwd_fields[c][2]); w++) {
	 		if(!isdigit(passwd_fields[c][2][w])) {
	 			bk_passwd_warn("non-numeric uid",c+1,passwd_line[c],DEBUG);
	 		}	
	 	}
	 	if(atoi(passwd_fields[c][2])==0) {
	 		if(DEBUG_MODE_ON) bk_passwd_warn("uid evaluates to 0",c+1,passwd_line[c],DEBUG);
	 		for(w=0; w<xnum_allow_uid; w++) {
	 			if(strcmp(passwd_fields[c][0],xallow_uid[w])==0) {
	 				trigger_safe=1;	
	 			}
	 		}
	 		if(!trigger_safe) bk_passwd_warn("unauthorized uid 0",c+1,passwd_line[c],EMERG);	 		
	 	}
	 	// gid
	 	
	 	trigger_safe=1;
	 	
		for(w=0; w<strlen(passwd_fields[c][3]); w++) {
	 		if(!isdigit(passwd_fields[c][3][w])) {
	 			trigger_safe=0;
	 		}	
	 	}
	 	if(!trigger_safe) {
	 		bk_passwd_warn("non-numeric gid",c+1,passwd_line[c],DEBUG);
	 	}
	 	trigger_safe=0;
	 	if(atoi(passwd_fields[c][3])==0) {
	 		if(DEBUG_MODE_ON) bk_passwd_warn("gid evaluates to 0",c+1,passwd_line[c],DEBUG);
	 		for(w=0; w<xnum_allow_gid; w++) {
	 			if(strcmp(passwd_fields[c][0],xallow_gid[w])==0) {
	 				trigger_safe=1;	
	 			}
	 		}
	 		if(!trigger_safe) bk_passwd_warn("unauthorized gid 0",c+1,passwd_line[c],EMERG);
	 	}	 	
	 	// GECOS
	 	if(strlen(passwd_fields[c][4])==1 && passwd_fields[c][4][0]==' ') {
	 		bk_passwd_warn("no gecos field",c+1,passwd_line[c],INFO);
	 	}	 	
	 	
	 	// home
	 	if(strlen(passwd_fields[c][5])==1 && passwd_fields[c][5][0]==' ') {
	 		bk_passwd_warn("no home directory",c+1,passwd_line[c],INFO);
	 	}
	 	// shell
	 	if(strlen(passwd_fields[c][6])==1 && passwd_fields[c][6][0]==' ') {
	 		bk_passwd_warn("no shell",c+1,passwd_line[c],INFO);
	 	}	 	
	}

}

void group_check(char *group_file)
{
	FILE *group;
	int i=0,j,c,group_field_count,w,x,y,z;
	char group_line[MAX_PASSWD_LINES][MAX_PASSWD_LINE_LEN];
	char temp[MAX_GROUP_FIELD+7];
	char *p,*t,*q;
	// char group_fields[LINES][FIELDS][LENGTH];	
	char group_fields[MAX_PASSWD_LINES][6][MAX_GROUP_FIELD];
	// char group_members[GROUPS][MEMBERS][LENGTH];
	char group_members[MAX_PASSWD_LINES][MAX_GROUP_MEMBERS][MAX_GROUP_FIELD];
	int num_group_members[MAX_PASSWD_LINES];	

	if((group=fopen(group_file,"r"))==NULL) {
		fprintf(stderr,"%s: %s: %s\n",__progname,"could not open file",group_file);
		exit(1);
	}

	while(!feof(group)) {
		if(i >= MAX_PASSWD_LINES) {
			fprintf(stderr,"%s: %s\n",__progname,"too many lines, recompile with greater MAX_PASSWD_LINES");
			exit(1);
		}
		if(fgets(group_line[i],(MAX_PASSWD_LINE_LEN-2),group)) i++;
		else break;
	}

	fclose(group);
	
	trigger_safe=0;
	
	printf("\n%s:\n\n",group_file);
		
	for(j=0; j<i; j++) {
		group_field_count=0;
		sprintf(group_line[j],"%s",chomp_str(group_line[j]));
		CHECK_NULL_GRP:
		q=group_line[j];
                if((t=strstr(q,"::")) != NULL) { // null field
                	bk_group_warn("null field",j+1,group_line[j],DEBUG);
                	sprintf(temp,"%s",group_line[j]);
                	x=strlen(temp);
                	y=strlen(t);
                	z=x-y; // index of "::"
                	temp[z+1]=' ';
                	for(w=z+1; w<x+1; w++) {
                		temp[w+1]=group_line[j][w];
                	}
                	sprintf(group_line[j],"%s",temp);
                	goto CHECK_NULL_GRP;             	
                }
                if(group_line[j][(strlen(group_line[j])-1)]==':') { // null last field
                	bk_group_warn("null field",j+1,group_line[j],DEBUG);
                	x=strlen(group_line[j]);
                	group_line[j][x]=' ';
                	group_line[j][x+1]='\0';
                }
                if(group_line[j][0]==':') { // null first field
                	bk_group_warn("null field",j+1,group_line[j],DEBUG); // here
                	sprintf(temp,"%s",group_line[j]);
	               	x=strlen(temp);
                	y=x-1;
                	z=x-y;
                	temp[0]=' ';
                	for(w=0; w<x+1; w++) {
                		temp[w+1]=group_line[j][w];
                	}
                	sprintf(group_line[j],"%s",temp);
                }
                sprintf(temp,"%s",group_line[j]);
		p=strtok(temp,":");
		if(strlen(p) > (MAX_GROUP_FIELD-2)) {
			fprintf(stderr,"%s: %s: %s\n",__progname,"group field too big","recompile with larger MAX_GROUP_FIELD");
			exit(1);
		}		
		sprintf(group_fields[j][0],"%s",p);
		group_field_count++;
		do {
			p=strtok('\0',":"); //here
			if(p) {
				if(strlen(p) > (MAX_GROUP_FIELD-2)) {
					fprintf(stderr,"%s: %s: %s\n",__progname,"group field too big","recompile with larger MAX_GROUP_FIELD");
					exit(1);
				}
				sprintf(group_fields[j][group_field_count],"%s",p);
				if(group_field_count==3) {
					// splice field into group_members[][]
					sprintf(temp,"%s",group_fields[j][group_field_count]);
					t=strtok(temp,",");
					num_group_members[j]=0;
					sprintf(group_members[j][num_group_members[j]],"%s",t);
					num_group_members[j]++;
					do {
						t=strtok('\0',","); // this eats
						if(t) {
							sprintf(group_members[j][num_group_members[j]++],"%s",t);						
						}					
					} while(t);             				
				}
				group_field_count++;
			}
		} while(p);
		if(group_field_count > 4) {
			bk_group_warn("too many fields (or colons)",j+1,group_line[j],WARN);
		}
		else if(group_field_count < 4) {
			bk_group_warn("fields missing (or missing colons)",j+1,group_line[j],WARN);
		}
		// split members into group_members[][][]
		
	}
	if(VERBOSE_OUTPUT) {
		for(c=0; c<i; c++) {
			printf("[%d] \'%s\' \'%s\' \'%s\' \'%s\'\n",c+1,group_fields[c][0],group_fields[c][1],group_fields[c][2],group_fields[c][3]);
		}
	}
		
	for(c=0; c<i; c++) {
		// group_name
		if(strlen(group_fields[c][0])==1 && group_fields[c][0][0]==' ') {
			bk_group_warn("nameless group",c+1,group_line[c],INFO);
		}
		// passwd
		if(strlen(group_fields[c][1])==1 && group_fields[c][1][0]==' ') {
			bk_group_warn("null password",c+1,group_line[c],WARN);
		}		
		if(xshadowed_passwords) {
			if(strlen(group_fields[c][1])!=1 || group_fields[c][1][0]!='x') {
				bk_group_warn("unshadowed password",c+1,group_line[c],WARN);
			}		
		}
		// gid
		
		trigger_safe=1;
	 	
		for(w=0; w<strlen(group_fields[c][2]); w++) {
	 		if(!isdigit(group_fields[c][2][w])) {
                		trigger_safe=0;
	 		}	
	 	}
	 	if(!trigger_safe) {
	 		bk_group_warn("non-numeric gid",c+1,group_line[c],DEBUG);
	 	}	 	
		trigger_safe=1;
	 	if(atoi(group_fields[c][2])==0) {
	 		if(DEBUG_MODE_ON) {
	 			bk_group_warn("gid evaluates to 0",c+1,group_line[c],DEBUG);
			}
			if(num_group_members[c]==0 || group_members[c][0][0]==' ') {
				bk_group_warn("unauthorised gid 0",c+1,group_line[c],EMERG);
			}
			for(w=0; w<num_group_members[c]; w++) {
					trigger_safe=0;
					for(z=0; z<xnum_allow_gid; z++) {
						if(strcmp(group_members[c][w],xallow_gid[z])==0) {
							trigger_safe=1;
						}					
					}
					if(!trigger_safe) {
						bk_group_warn("unauthorised gid 0",c+1,group_line[c],EMERG);
					}		
			}			

        	}
        	
		
		// members - see above
	
	}

}

void bk_passwd_warn(char *error_string,int line_num,char *line_string,int error_level)
{
	char *error_level_str[] = { 	"<DEBUG>",
					"<INFO>",
					"<WARN>",
					"<EMERG>" };
	char *f,g[256];				
	
	if(error_level >= output_level)
	{					
		if(USERNAME_ONLY) {
			if(line_string) {
				sprintf(g,"%s",line_string);
				f=strtok(g,":");
			}
			if(f) printf("%7s %-19s [%02d] %s\n",error_level_str[error_level],error_string,line_num,f);
			else printf("%7s %-19s [%02d]\n",error_level_str[error_level],error_string,line_num);
		}
		else {
			printf("%7s %-19s [%02d] %s\n",error_level_str[error_level],error_string,line_num,line_string);
		}
	}					
}

void bk_group_warn(char *error_string,int line_num,char *line_string,int error_level)
{
	char *error_level_str[] = {	"<DEBUG>",
					"<INFO>",
					"<WARN>",
					"<EMERG>" };
	char *f,g[256];				
	
	if(error_level >= output_level)
	{					
		if(USERNAME_ONLY) {
			if(line_string) {
				sprintf(g,"%s",line_string);
				f=strtok(g,":");
			}
			if(f) printf("%7s %-19s [%02d] %s\n",error_level_str[error_level],error_string,line_num,f);
			else printf("%7s %-19s [%02d]\n",error_level_str[error_level],error_string,line_num);
		}
		else {
			printf("%7s %-19s [%02d] %s\n",error_level_str[error_level],error_string,line_num,line_string);
		}
	}			
}

char *chomp_str(char *crlf_str)
{
	int str_len;

	str_len=strlen(crlf_str);
	if(crlf_str[str_len-1] == 13 || crlf_str[str_len-1] == 10) crlf_str[str_len-1]='\0';

	return crlf_str;
}
