#include "config_xor.h"


#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <unistd.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <fcntl.h>
#include <signal.h>

#include "slib.h"
#include "sh_calls.h"

#undef  FIL__
#define FIL__  _("slib.c")


/****************************************************************
 *
 *  The debug/trace subsystem
 *
 ****************************************************************/

#if defined(WITH_TRACE) || defined(WITH_TPT)

#define ALLE_STEINE -1

#if !defined(WITH_TRACE)
int stein_set = ALLE_STEINE;
#else
int stein_set = -99;
#endif

int stein     =  0;
int ebene     =  0;
int mindest   = -1;

static int use_console = 0;
void dbg_use_console()
{
  use_console = 1;
  return;
}

void dbg(int mod_ebene, char * file, int line, char * fmt, ...)
{
  FILE * dfile;
  va_list ap;
  char indent_it[32];
  int  i, imax;

  if (ebene < 1) 
    ++stein;

  if (mod_ebene > 0)
    ebene += mod_ebene;

  if (ebene < 31)
    imax = ebene;
  else
    imax = 31;

  /*
  fprintf (stderr, _("DBG: file=<%s>, line=<%d> %d %d \n"), 
		 file, line, stein_set, mindest);
  */

  for (i = 0; i < imax; ++i)
    indent_it[i] = ' ';
  indent_it[imax] = '\0';

  if ( (stein == stein_set || stein_set == ALLE_STEINE ) && ebene <= mindest)
    {
      if (ebene < 1)
#if !defined(WITH_TRACE)
	{
	  if (0 == use_console)
	    {
	      fprintf (stderr, _("DBG: file=<%s>, line=<%d>, "), 
		       file, line);
	      va_start(ap, fmt);
	      vfprintf  (stderr, fmt, ap);
	      va_end(ap);
	    }
	  else
	    {
	      dfile = fopen(_("/dev/console"), "w");
	      fprintf (dfile, _("DBG: file=<%s>, line=<%d>, "), file, line);
	      va_start(ap, fmt);
	      vfprintf  (dfile, fmt, ap);
	      va_end(ap);
	      fclose (dfile);
	    }
	}
#else
      {
	if (0 == use_console)
	  fprintf (stderr, 
		   _("MILEST: file=<%s>, line=<%d>, milestone=<%d>, "), 
		   file, line, stein);
	else
	  {
	    dfile = fopen(_("/dev/console"), "w");
	    fprintf (dfile, 
		     _("MILEST: file=<%s>, line=<%d>, milestone=<%d>, "), 
		     file, line, stein);
	    va_start(ap, fmt);
	    vfprintf  (dfile, fmt, ap);
	    va_end(ap);
	    fclose (dfile);
	  }
      }
#endif
      else
	{
	  if (0 == use_console)
	    {
	      fprintf (stderr, _("%sTRACE : file=<%s>, line=<%d>, "), 
		       indent_it, file, line);
	      va_start(ap, fmt);
	      vfprintf  (stderr, fmt, ap);
	      va_end(ap);
	    }
	  else
	    {
	      dfile = fopen(_("/dev/console"), "w");
	      fprintf (dfile, 
		       _("%sTRACE : file=<%s>, line=<%d>, "),
		       indent_it, file, line);
	      va_start(ap, fmt);
	      vfprintf  (dfile, fmt, ap);
	      va_end(ap);
	      fclose (dfile);
	    }
	}
    }

  if (mod_ebene < 0)
    ebene += mod_ebene;

  return;
}

int set_stein (char * c)
{
  stein_set = atoi(c);
  if (stein_set < -1)
    {
      stein_set = -99;
      return (-1);
    }
  return 0;
}
int set_ebene (char * c)
{
  mindest = atoi(c);
  if (mindest < 0)
    {
      mindest = -99;
      return (-1);
    }
  return 0;
}
/* WITH_TRACE */
#endif

/* ---------------------------------------------------------------- 
 *
 *    Free()/Malloc() with block
 *
 * ---------------------------------------------------------------- */

int myblock_flag = 0;

void my_free(void *ptr)
{
  if (myblock_flag == 0)
    {
      myblock_flag = 1;
      free(ptr);
      myblock_flag = 0;
    }
  else
    {
#ifdef SIGILL
      raise(SIGILL);       /* SIGILL is POSIX.1  */
#endif
      _exit(EXIT_FAILURE);
    }
  return;
}

void * my_malloc(size_t size)
{
  void * ptr = NULL; /* some compilers don't understand _exit() */

  if (myblock_flag == 0)
    {
      myblock_flag = 1;
      ptr = malloc(size);
      myblock_flag = 0;
    }
  else
    {
#ifdef SIGILL
      raise(SIGILL);       /* SIGILL is POSIX.1  */
#endif
      _exit(EXIT_FAILURE);
    }
  return ptr;
}
  

/*
 *  The global errno.
 *  On error, this is set to the return value of the function.
 */
long int sl_errno;


/* ---------------------------------------------------------------- 
 *
 *    String handling routines
 *
 * ---------------------------------------------------------------- */
  
#if !defined(HOST_IS_I86SOLARIS)
#if !defined (_GNU_SOURCE)
extern int vsnprintf ( char *str, size_t n,
		       const char *format, va_list ap );
#endif
#endif

#if !defined (VA_COPY)
#if defined (__GNUC__) && defined (__PPC__) && (defined (_CALL_SYSV) || defined (_WIN32))
#define VA_COPY(ap1, ap2)     (*(ap1) = *(ap2))
#elif defined (VA_COPY_AS_ARRAY)
#define VA_COPY(ap1, ap2)     memmove ((ap1), (ap2), sizeof (va_list))
#else /* va_list is a pointer */
#define VA_COPY(ap1, ap2)     ((ap1) = (ap2))
#endif
#endif 

#if !defined(HAVE_VSNPRINTF) || defined(HAVE_BROKEN_VSNPRINTF)
static
int sl_printf_count (const char * fmt, va_list  vl)
{
  int  length       = 1;
  int  fini         = 0;
  int  islong       = 0;
  int  islongdouble = 0;
  char * string_arg;

  SL_ENTER(_("sl_printf_count"));

  if (fmt == NULL)
    SL_IRETURN(SL_ENULL, _("sl_printf_count"));

  while (*fmt) {

    if ( (*fmt) == '%' ) { /* a format specifier */

      fmt++;        /* point to first char after '%' */

      fini = 0;
      islong = 0;
      islongdouble = 0;

      while (*fmt && (fini == 0) ) {
	
	switch (*fmt) {

	case '*':      /* field width supplied by an integer */
	  length = length + va_arg (vl, int);
	  ++fmt;
	  break;
	case '1':
	case '2':
	case '3':
	case '4':
	case '5':
	case '6':
	case '7':
	case '8':
	case '9':
	  length = length + strtol (fmt, (char**) &fmt, 10);
	  /* strtol makes FastForward to first invalid char */
	  break;

	case 'l':   /* 'long' modifier */
	  islong = 1;
	  ++fmt;
	  break;
	case 'L':  /* 'long double' modifier */ 
#ifdef HAVE_LONG_DOUBLE	  
	  islongdouble = 1;
#else
	  islong = 1;
#endif
	  ++fmt;
	  break;

	case 'd':
	case 'i': 
	case 'o':
	case 'u':
	case 'x':
	case 'X':
	  if (islong == 1)
	    (void) va_arg (vl, long);
	  else
	    (void) va_arg (vl, int);
	  length = length + 24;
	  ++fmt;
	  fini = 1;
	  break;

	case 'D':
	case 'O':
	case 'U':
	  (void) va_arg (vl, long);
	  length = length + 24;
	  fmt++;
	  fini = 1;
	  break;

	case 'e':
	case 'E':
	case 'f':
	case 'g':
#ifdef HAVE_LONG_DOUBLE	  
	  if (islongdouble == 1) {
	    (void) va_arg (vl, long double);
	    length = length + 20;
	    }
	  else
#endif
	    (void) va_arg (vl, double);
	  length = length + 20;
	  fini = 1;
	  ++fmt;
	  break;

	case 's':
	  string_arg = va_arg (vl, char *);
	  if (string_arg != NULL)
	    length = length + sl_strlen (string_arg);
	  else
	    length = length + 16;
	  fini = 1;
	  ++fmt;
	  break;

	case 'c':
	  (void) va_arg (vl, int);
	  length = length + 1;
	  fini = 1;
	  ++fmt;
	  break;

	case 'p':
	case 'n':
	  (void) va_arg (vl, void * );
	  length = length + 32;
	  fini = 1;
	  ++fmt;
	  break;

	case '%':            /* %% will print '%' */
	  length = length + 1;
	  fini = 1;
	  ++fmt;
	  break;

	default:
	  length = length + 1;
	  ++fmt;
	  break;

	}  /* end switch */
      }    
      /* end parsing a single format specifier */
    } else {
      length = length + 1;
      fmt++;
    }
  }
  SL_IRETURN(length, _("sl_printf_count"));
}
#endif  /* #ifndef  HAVE_VSNPRINTF */

/*
 * An implementation of vsnprintf. va_start/va_end are in the caller
 * function.
 * Returns SL_ENONE on success.
 * ENULL:  src || format == NULL
 * ERANGE: n out of range
 * ETRUNC: truncated
 */
int sl_vsnprintf(char *str, size_t n,
		 const char *format, va_list vl )
{
  int           total;
#if !defined(HAVE_VSNPRINTF) || defined(HAVE_BROKEN_VSNPRINTF)
  va_list       vl2;
#endif

  SL_ENTER(_("sl_vsnprintf"));
  if (str == NULL || format == NULL)
    SL_IRETURN(SL_ENULL, _("sl_vsnprintf"));

  total = 0;

#if defined(HAVE_VSNPRINTF) && !defined(HAVE_BROKEN_VSNPRINTF)
  vsnprintf (str, n, format, vl);
#else
  VA_COPY (vl2, vl);                   /* save the argument list           */
  total = sl_printf_count (format, vl);
  if (total < n) 
    vsprintf (str, format, vl2);       /* program has checked that it fits */
  else 
    {
      sl_strlcpy (str, format, n);
      va_end(vl2);
      SL_IRETURN(SL_ETRUNC, _("sl_vsnprintf"));
    }
  va_end(vl2);
#endif
  SL_IRETURN(SL_ENONE, _("sl_vsnprintf"));
}

/*
 * An implementation of snprintf.
 * Returns SL_ENONE on success.
 * ENULL:  src || format == NULL
 * ERANGE: n out of range
 * ETRUNC: truncated
 */
int sl_snprintf(char *str, size_t n,
		const char *format, ... )
{
  int           total = 0;
  va_list       vl;
#if !defined(HAVE_VSNPRINTF) || defined(HAVE_BROKEN_VSNPRINTF)
  va_list       vl2;
#endif

  SL_ENTER(_("sl_snprintf"));
  if (str == NULL || format == NULL)
    SL_IRETURN(SL_ENULL, _("sl_snprintf"));
  
  va_start (vl, format);
#if defined(HAVE_VSNPRINTF) && !defined(HAVE_BROKEN_VSNPRINTF)
  vsnprintf (str, n, format, vl);
  total = sl_strlen(str);
#else
  VA_COPY (vl2, vl);                   /* save the argument list           */
  total = sl_printf_count (format, vl);
  if (total < n) 
    vsprintf (str, format, vl2);       /* program has checked that it fits */
  else 
    {
      sl_strlcpy (str, format, n);
      va_end(vl2);
      SL_IRETURN(SL_ETRUNC, _("sl_snprintf"));
    }
  va_end(vl2);
#endif  
  va_end(vl);
  SL_IRETURN(SL_ENONE, _("sl_snprintf"));
}

/*
 * Appends src to string dst of size siz (unlike strncat, siz is the
 * full size of dst, not space left).  At most siz-1 characters
 * will be copied.  Always NUL terminates (unless siz == 0).
 * Returns SL_NONE on success, errcode on failure.
 *
 * ENULL:  dst == NULL
 * ERANGE: siz out of range
 * ETRUNC: src truncated
 */
int sl_strlcat(char * dst, const char *src, size_t siz)
{
  register size_t dst_end;
  register size_t dst_free;
  register size_t src_len;

  SL_ENTER(_("sl_strlcat"));

  if (dst == NULL)
    SL_IRETURN (SL_ENULL, _("sl_strlcat"));
  if (src == NULL || src == "") 
    SL_IRETURN (SL_ENONE, _("sl_strlcat"));

  /* How long is src ?
   */
  src_len  = strlen(src);

  if (siz > 0) {

    /* How much free space do we have ?
     */
    dst_end  = strlen(dst);
    dst_free = siz - dst_end;

    /* Copy at most dst_free chars.
     * The documentation on strncat() is inconclusive as to
     * whether the NULL counts or not, thus use strncpy().
     * dst[dst_end] is the NULL terminator of dst.
     */
    (void) strncpy(&dst[dst_end], 
		   src, 
		   ( (src_len < dst_free) ? src_len : dst_free));

    /* NULL terminate dst.
     */
    if (src_len < dst_free) 
      dst[dst_end+src_len] = '\0';
    dst[siz-1] = '\0';

    if (src_len >= dst_free) 
      SL_IRETURN (SL_ETRUNC, _("sl_strlcat"));
  }

  SL_IRETURN (SL_ENONE, _("sl_strlcat"));
}


/*
 * An alternative implementation of the OpenBSD strlcpy() function.
 *
 * Copy src to string dst of size siz.  At most siz-1 characters
 * will be copied.  Always NUL terminates (unless siz == 0).
 * Returns SL_NONE on success, errcode on failure.
 *
 * ENULL:  dst == NULL
 * ERANGE: siz out of range
 * ETRUNC: src truncated
 */
int sl_strlcpy(char * dst, const char * src, size_t siz)
{
  SL_ENTER(_("sl_strlcpy"));

  if (dst == NULL)
    SL_IRETURN (SL_ENULL, _("sl_strlcpy"));
  if (src == NULL)
    { 
      if (siz > 0) 
	dst[0] = '\0';
      SL_IRETURN (SL_ENONE, _("sl_strlcpy"));
    }


  if (siz > 0) {
    /* copy siz-1 characters 
     */
    (void) strncpy(dst, src, siz-1);

    /* NULL terminate
     */
    dst[siz-1] = '\0';

    if (strlen(src) >= siz)
      SL_IRETURN (SL_ETRUNC, _("sl_strlcpy"));
  }
  SL_IRETURN (SL_ENONE, _("sl_strlcpy"));
}

/*
 * A robust drop-in replacement of strncpy. strlcpy is preferable.
 */
char * sl_strncpy(char *dst, const char *src, size_t size)
{
  size_t len;

#ifdef SL_FAIL_ON_ERROR
  SL_REQUIRE(dst != NULL, _("dst != NULL"));
  SL_REQUIRE(src != NULL, _("src != NULL"));
  SL_REQUIRE(size > 0, _("size > 0"));
#endif

  if (dst == NULL)
    {
      sl_errno = SL_ENULL;
      return (NULL);
    }
  if (size < 1)
    {
      sl_errno = SL_ERANGE;
      return (dst);
    }
  if (!src)
    {
      sl_errno = SL_ENULL;
      dst[0] = '\0';
    }
  else if (src[0] == '\0')
    dst[0] = '\0';
  else
    strncpy(dst, src, size);

  if ((len = sl_strlen(src)) >= size)
    {
      errno = ENOSPC;
      dst[size-1] = '\0';
    }
  return (dst);
}

/*
 * A robust drop-in replacement of strncat. strlcat is preferable.
 */
char * sl_strncat(char *dst, const char *src, size_t n)
{
#ifdef SL_FAIL_ON_ERROR
  SL_REQUIRE(dst != NULL, _("dst != NULL"));
  SL_REQUIRE(src != NULL, _("src != NULL"));
  SL_REQUIRE(n > 0, _("n > 0"));
#endif

  if (dst == NULL)
    {
      sl_errno = SL_ENULL;
      return (NULL);
    }
  if (n < 1)
    {
      sl_errno = SL_ERANGE;
      return (dst);
    }
  if (!src)
    {
      sl_errno = SL_ENULL;
      return (dst);
    }
  else if (src[0] == '\0')
    dst[0] = '\0';
  else
    strncat(dst, src, n);

  return (dst);
}


int sl_strcmp(char * a, char * b)
{
#ifdef SL_FAIL_ON_ERROR
  SL_REQUIRE (a != NULL, _("a != NULL"));
  SL_REQUIRE (b != NULL, _("b != NULL"));
#endif

  if (a != NULL && b != NULL)
    return (strcmp(a, b));
  else if (a == NULL && b != NULL)
    return (-1);
  else if (a != NULL && b == NULL)
    return (1);
  else
    return (-7); /* arbitrary */
}

int sl_strncmp(char * a, char * b, size_t n)
{
#ifdef SL_FAIL_ON_ERROR
  SL_REQUIRE (a != NULL, _("a != NULL"));
  SL_REQUIRE (b != NULL, _("b != NULL"));
  SL_REQUIRE (n > 0, _("n > 0"));
#endif

  if (a != NULL && b != NULL)
    return (strncmp(a, b, n));
  else if (a == NULL && b != NULL)
    return (-1);
  else if (a != NULL && b == NULL)
    return (1);
  else
    return (-7); /* arbitrary */
}


size_t sl_strlen(const char *s)
{
#ifdef SL_FAIL_ON_ERROR
  SL_REQUIRE (s != NULL, _("s != NULL"));
#endif

  if (s == NULL)
    return (0);
  else
    return (strlen(s));
}

/* string searching
 */

char * sl_strstr (char * haystack, char * needle) 
{
#ifndef HAVE_STRSTR
  int             i;
#endif
  size_t          needle_len;
  size_t          haystack_len;
  
  if (haystack == NULL || needle == NULL)
    return NULL;
    
  needle_len   = strlen(needle);
  haystack_len = strlen(haystack);

  if (needle_len == 0 || haystack_len == 0)
    return NULL;

#if defined(HAVE_STRSTR)
  return (strstr(haystack, needle));
#else
  for (i = 0; i <= (haystack_len-needle_len); ++i)
    if (0 == sl_strncmp(&haystack[i], needle, needle_len))
      return (needle);
  return NULL;
#endif
}




/* ---------------------------------------------------------------- 
 *
 *    Privilege handling routines
 *
 * ---------------------------------------------------------------- */

  

static   uid_t   euid;
static   uid_t   ruid;
static   uid_t   ruid_orig;
static   gid_t   egid;
static   gid_t   rgid;
static   gid_t   rgid_orig;

static   int     uids_are_stored = SL_FALSE;
static   int     suid_is_set     = SL_TRUE;

#ifdef HAVE_SETRESUID
extern       int setresuid (uid_t truid, uid_t teuid, uid_t tsuid);
extern       int setresgid (gid_t trgid, gid_t tegid, gid_t tsgid);
#endif


/*
 * This function returns true if the program is SUID.
 * It calls abort() if the uid's are not saved already.
 */
int sl_is_suid()
{
  SL_ENTER(_("sl_is_suid"));

  SL_REQUIRE(uids_are_stored == SL_TRUE, _("uids_are_stored == SL_TRUE"));
  if (euid == ruid && egid == rgid)
    SL_IRETURN (0, _("sl_is_suid"));     /* FALSE */
  else
    SL_IRETURN (1, _("sl_is_suid"));     /* TRUE  */
}

/*
 * This function returns the saved euid.
 * It calls abort() if the uid's are not saved already.
 */
int sl_get_euid(uid_t * ret)
{
  SL_ENTER(_("sl_get_euid"));
  SL_REQUIRE(uids_are_stored == SL_TRUE, _("uids_are_stored == SL_TRUE"));
  *ret = euid;
  SL_IRETURN (SL_ENONE, _("sl_get_euid"));
}

/*
 * This function returns the saved egid.
 * It calls abort() if the uid's are not saved already.
 */
int sl_get_egid(gid_t * ret)
{
  SL_ENTER(_("sl_get_egid"));
  SL_REQUIRE(uids_are_stored == SL_TRUE, _("uids_are_stored == SL_TRUE"));
  *ret = egid;
  SL_IRETURN (SL_ENONE, _("sl_get_egid"));
}

/*
 * This function returns the saved ruid.
 * It calls abort() if the uid's are not saved already.
 */
int sl_get_ruid(uid_t * ret)
{
  SL_ENTER(_("sl_get_ruid"));
  SL_REQUIRE(uids_are_stored == SL_TRUE, _("uids_are_stored == SL_TRUE"));
  *ret = ruid;
  SL_IRETURN (SL_ENONE, _("sl_get_ruid"));
}

/*
 * This function returns the saved rgid.
 * It calls abort() if the uid's are not saved already.
 */
int sl_get_rgid(gid_t * ret)
{
  SL_ENTER(_("sl_get_rgid"));
  SL_REQUIRE(uids_are_stored == SL_TRUE, _("uids_are_stored == SL_TRUE"));
  *ret = rgid;
  SL_IRETURN (SL_ENONE, _("sl_get_rgid"));
}

/*
 * This function returns the saved original ruid.
 * It calls abort() if the uid's are not saved already.
 */
int sl_get_ruid_orig(uid_t * ret)
{
  SL_ENTER(_("sl_get_ruid_orig"));
  SL_REQUIRE(uids_are_stored == SL_TRUE, _("uids_are_stored == SL_TRUE"));
  *ret = ruid_orig;
  SL_IRETURN (SL_ENONE, _("sl_get_ruid_orig"));
}

/*
 * This function returns the saved original rgid.
 * It calls abort() if the uid's are not saved already.
 */
int sl_get_rgid_orig(gid_t * ret)
{
  SL_ENTER(_("sl_get_rgid_orig"));
  SL_REQUIRE(uids_are_stored == SL_TRUE, _("uids_are_stored == SL_TRUE"));
  *ret = rgid_orig;
  SL_IRETURN (SL_ENONE, _("sl_get_rgid_orig"));
}

static int suid_warn_flag = 1;
static void suid_warn(int a)
{
  fprintf(stderr, "ERROR:  open set/unset suid !!! %d\n", a);
  return;
}

/*
 * This function sets the effective uid 
 * to the saved effective uid.
 * It will abort on failure.
 */
int sl_set_suid ()
{
  int retval;

  SL_ENTER(_("sl_set_suid"));
  SL_REQUIRE(uids_are_stored == SL_TRUE, _("uids_are_stored == SL_TRUE"));  

  if (ruid == euid && rgid == egid) 
    {
      suid_is_set = SL_TRUE;
      SL_IRETURN(SL_ENONE, _("sl_set_suid"));
    }  
  SL_REQUIRE(suid_is_set     == SL_FALSE, _("suid_is_set == SL_FALSE"));  

#if defined(HAVE_SETRESUID)
  retval = setresuid (-1, euid, -1);
  if (retval == 0) 
    retval = setresgid (-1, egid, -1);

#elif defined(HAVE_SETEUID)
  retval = seteuid (egid);
  if (retval == 0) 
    retval = setegid (euid);

  /* on AIX, setreuid does not behave well for non-root users.
   */
#elif defined(HAVE_SETREUID)
  retval = setreuid (ruid, euid);
  if (retval == 0) 
    retval = setregid (rgid, egid);

#else
  retval = setuid (euid);
  if (retval == 0) 
    retval = setgid (egid);
#endif
  if (suid_warn_flag == 1)
    suid_warn(1);
  suid_warn_flag = 1;

  SL_REQUIRE(retval == 0, _("retval == 0"));
  suid_is_set = SL_TRUE;
  SL_IRETURN(SL_ENONE, _("sl_set_suid"));
}

/*
 * This function sets the effective uid to the real uid.
 * It will abort on failure.
 */
int sl_unset_suid ()
{
  register int retval;

  SL_ENTER(_("sl_unset_suid"));
  SL_REQUIRE(uids_are_stored == SL_TRUE, _("uids_are_stored == SL_TRUE"));

  if (ruid == euid && rgid == egid)
    {
      suid_is_set = SL_FALSE;
      SL_IRETURN(SL_ENONE, _("sl_unset_suid"));
    }  
  SL_REQUIRE(suid_is_set     == SL_TRUE, _("suid_is_set == SL_TRUE"));  

#if defined(HAVE_SETRESUID)
  retval = setresgid (-1, rgid, -1);
  if (retval == 0) 
    retval = setresuid (-1, ruid, -1);

#elif defined(HAVE_SETEUID)
  retval = setegid (rgid);
  if (retval == 0) 
    retval = seteuid (ruid);

#elif defined(HAVE_SETREUID)
  retval = setregid (egid, rgid);
  if (retval == 0) 
    retval = setreuid (euid, ruid);

#else
  retval = setgid (rgid);
  if (retval == 0) 
    retval = setuid (ruid);
#endif

  if (suid_warn_flag == 0)
    suid_warn(0);
  suid_warn_flag = 0;

  SL_REQUIRE(retval == 0, _("retval == 0"));
  suid_is_set = SL_FALSE;
  SL_IRETURN(SL_ENONE, _("sl_unset_suid"));
}


/*
 * This function saves the uid's.
 */
int sl_save_uids()
{
  SL_ENTER(_("sl_save_uids"));
  if (uids_are_stored == SL_TRUE) 
    SL_IRETURN(SL_EREPEAT, _("sl_save_uids"));

  ruid_orig = getuid();
  rgid_orig = getgid();
  egid = getegid();
  euid = geteuid();
  ruid = ruid_orig;
  rgid = rgid_orig;
  uids_are_stored = SL_TRUE;

  SL_IRETURN(SL_ENONE, _("sl_save_uids"));
}

/* 
 * This function drops SUID privileges irrevocably.
 * It set the effective uid to the original real uid.
 */
int sl_drop_privileges()
{
  SL_ENTER(_("sl_drop_privileges"));
  SL_REQUIRE(uids_are_stored == SL_TRUE, _("uids_are_stored == SL_TRUE"));

  SL_REQUIRE(setgid(rgid_orig) == 0, _("setgid(rgid_orig) == 0"));
  SL_REQUIRE(setuid(ruid_orig) == 0, _("setuid(ruid_orig) == 0"));

  /* make sure that setuid(0) fails
   */
  SL_REQUIRE(setuid(0) < 0, _("setuid(0) < 0"));

  euid = ruid_orig;
  egid = rgid_orig;
  ruid = ruid_orig;
  rgid = rgid_orig;

  SL_IRETURN(SL_ENONE, _("sl_drop_privileges"));
}

/* 
 * Define a policy: Stay root.
 * Do nothing if not SUID.
 */
int sl_policy_get_root()
{
  SL_ENTER(_("sl_policy_get_root"));
  SL_REQUIRE(uids_are_stored == SL_FALSE, _("uids_are_stored == SL_FALSE"));

  SL_REQUIRE (sl_save_uids() == SL_ENONE, _("sl_save_uids() == SL_ENONE"));

  if (euid != ruid || egid != rgid)
    {
      SL_REQUIRE(setgid(egid) == 0, _("setgid(egid) == 0"));
      SL_REQUIRE(setuid(euid) == 0, _("setuid(euid) == 0"));
      SL_REQUIRE(ruid == getuid() && rgid == getgid(),
		 _("ruid == getuid() && rgid == getgid()"));
      ruid = euid;
      rgid = egid;
    }
  suid_is_set = SL_TRUE;
  SL_IRETURN(SL_ENONE, _("sl_policy_get_root"));
}

/* 
 * Define a policy: Get real (irrevocably).
 * This function drops SUID privileges irrevocably.
 * Do nothing if not SUID.
 */
int sl_policy_get_real()
{
  SL_ENTER(_("sl_policy_get_real"));
  SL_REQUIRE(uids_are_stored == SL_FALSE, _("uids_are_stored == SL_FALSE"));
  SL_REQUIRE (sl_save_uids() == SL_ENONE, _("sl_save_uids() == SL_ENONE"));

  if (euid != ruid || egid != rgid)
    {
      SL_REQUIRE (sl_drop_privileges() == SL_ENONE,
		  _("sl_drop_privileges() == SL_ENONE"));
    }
  suid_is_set = SL_TRUE;
  SL_IRETURN(SL_ENONE, _("sl_policy_get_real"));
}

#include <pwd.h>

/* 
 * Define a policy: Get user.
 * Drops privileges.
 */
int sl_policy_get_user(char * user)
{
  struct passwd * tempres;

  SL_ENTER(_("sl_policy_get_user"));

  SL_REQUIRE(user != NULL, _("user != NULL"));
  SL_REQUIRE(uids_are_stored == SL_FALSE, _("uids_are_stored == SL_FALSE"));
  SL_REQUIRE (sl_save_uids() == SL_ENONE, _("sl_save_uids() == SL_ENONE"));

  if (euid != ruid || egid != rgid)
    {
      tempres = getpwnam(user);
      SL_REQUIRE (NULL != tempres, _("tempres != NULL"));

      rgid = tempres->pw_gid;
      ruid = tempres->pw_uid;

      SL_REQUIRE(sl_unset_suid() == SL_ENONE, 
		 _("sl_unset_suid() == SL_ENONE"));
    }
  SL_IRETURN(SL_ENONE, _("sl_policy_get_user"));
}



/* ---------------------------------------------------------------- 
 *
 *    File access routines
 *
 * ---------------------------------------------------------------- */

#define TOFFSET 0x1234 
#define MAXFD   FOPEN_MAX

typedef struct openfiles {
  SL_TICKET ticket;     /* The unique  ID.      */ 
  int fd;               /* The file descriptor. */
  char * path;          /* The file path.       */
} SL_OFILE; 

static SL_OFILE *ofiles[MAXFD]; 


static unsigned int nonce_counter = TOFFSET;

static
SL_TICKET sl_create_ticket (unsigned int myindex) 
{
  unsigned int high; /* index */ 
  unsigned int low;  /* nonce */

  SL_ENTER(_("sl_create_ticket"));

  if (myindex >= MAXFD) 
    SL_IRETURN (SL_EINTERNAL, _("sl_create_ticket")); 

  /* mask out the high bit and check that it is not used
   * -> verify that it fits into 16 bits as positive
   */
  high = (myindex + TOFFSET) & 0x7fff; 

  if (high != myindex + TOFFSET) 
    SL_IRETURN (SL_EINTERNAL, _("sl_create_ticket")); 

  low = nonce_counter & 0xffff;

  /* Overflow -> nonce too big.
   */
  if ((low != nonce_counter++) || low == 0)
    SL_IRETURN (SL_EINTERNAL, _("sl_create_ticket"));
 
  /* Wrap around the nonce counter.
   * This is a dirty trick.
   */
  if (nonce_counter > 0x7fff)
    nonce_counter = TOFFSET;

  SL_RETURN ((SL_TICKET) ((high << 16) | low), _("sl_create_ticket")); 
}

static 
int sl_read_ticket (SL_TICKET fno) 
{
  register unsigned myindex; 
  register SL_OFILE *of; 

  SL_ENTER(_("sl_read_ticket"));

  myindex = ((fno >> 16) & 0xffff) - TOFFSET;
  if (myindex >= MAXFD)
    SL_IRETURN(SL_ETICKET, _("sl_read_ticket"));

  if (ofiles[myindex] == NULL)
    SL_IRETURN(SL_ETICKET, _("sl_read_ticket"));

  if (ofiles[myindex]->ticket != fno)
    SL_IRETURN(SL_ETICKET, _("sl_read_ticket"));

  if ((of = ofiles[myindex])->fd < 0 || of->fd >= MAXFD )
    SL_IRETURN(SL_EINTERNAL, _("sl_read_ticket"));

  if (((of->ticket) & 0xffff) == 0)
    SL_IRETURN(SL_EINTERNAL, _("sl_read_ticket")); 

  SL_RETURN(myindex, _("sl_read_ticket")); 
}

#define SL_OPEN_MIN          113
#define SL_OPEN_FOR_READ     113
#define SL_OPEN_FOR_WRITE    114
#define SL_OPEN_FOR_RDWR     115
#define SL_OPEN_FOR_WTRUNC   116
#define SL_OPEN_FOR_RWTRUNC  117
#define SL_OPEN_SAFE_RDWR    118
#define SL_OPEN_FOR_FASTREAD 119
#define SL_OPEN_MAX          119

static mode_t  open_mode = (S_IWUSR|S_IRUSR|S_IRGRP);


static
int sl_open_file (char *filename, int mode, int priv)
{
  struct stat   lbuf;
  struct stat   buf;
  int           lstat_return;
  int           stat_return;
  int           fd;
  int           sflags;
  SL_TICKET     ticket;

  SL_ENTER(_("sl_open_file"));
  
#if !defined(O_NOATIME)
  /* 
   * bitwise 'or' with zero does not modify any bit 
   */
#define O_NOATIME 0
#endif
 
#if !defined(O_NONBLOCK)
#if defined(O_NDELAY)
#define O_NONBLOCK  O_NDELAY
#else
#define O_NONBLOCK  0
#endif
#endif

  if (filename == NULL)
    SL_IRETURN(SL_ENULL, _("sl_open_file"));
  if (mode < SL_OPEN_MIN || mode > SL_OPEN_MAX)
    SL_IRETURN(SL_EINTERNAL, _("sl_open_file"));
  
  
  /* "This system call always succeeds and the previous value of
   * the mask is returned." 
   */
  (void) umask (0); 

  if (mode == SL_OPEN_FOR_FASTREAD)
    {
      fd = aud_open (FIL__, __LINE__, priv, filename, 
		     O_RDONLY|O_NOATIME|O_NONBLOCK, 0);
      if (fd >= 0) {
	sflags = retry_fcntl(fd, F_GETFL, 0);
	retry_fcntl(fd, F_SETFL, sflags & ~O_NONBLOCK);
      }
      if (fd < 0)
	SL_IRETURN(SL_EBADFILE, _("sl_open_file"));
      goto createTicket;
    }

  if (priv == SL_YESPRIV)
    sl_set_suid();
  lstat_return = retry_lstat(filename, &lbuf);
  if (priv == SL_YESPRIV)
    sl_unset_suid();

  if (lstat_return == -1)
    {
      lstat_return = ENOENT;
      if ( (mode == SL_OPEN_FOR_READ && lstat_return == ENOENT) ||
	   (errno != ENOENT))
	{
	  TPT(( 0, FIL__, __LINE__, _("msg=<lstat: %s> errno=<%d>\n"), 
	    filename, errno));
	  SL_IRETURN(SL_EBADFILE, _("sl_open_file"));
	}
    }
  
  if ( (mode != SL_OPEN_FOR_READ) && (lstat_return != ENOENT) &&
       ( S_ISDIR(lbuf.st_mode) || (S_IWOTH & lbuf.st_mode) ) 
      )
    SL_IRETURN(SL_EBADFILE, _("sl_open_file"));

    
  /* O_NOATIME has an effect for read(). But write() ?.
   */
  switch (mode)
    {
    case SL_OPEN_FOR_READ:
      fd = aud_open (FIL__, __LINE__, priv, filename, 
		     O_RDONLY|O_NOATIME|O_NONBLOCK, 0);
      if (fd >= 0) {
	sflags = retry_fcntl(fd, F_GETFL, 0);
	retry_fcntl(fd, F_SETFL, sflags & ~O_NONBLOCK);
      }
      break;
    case SL_OPEN_FOR_WRITE:
      if (lstat_return == ENOENT)
      	fd = aud_open (FIL__, __LINE__, priv, filename, 
		       O_WRONLY|O_CREAT|O_EXCL,    open_mode);
      else
	fd = aud_open (FIL__, __LINE__, priv, filename, 
		       O_WRONLY|O_CREAT|O_NOATIME, open_mode);
      break;
    case SL_OPEN_SAFE_RDWR:
      if (lstat_return == ENOENT)
      	fd = aud_open (FIL__, __LINE__, priv, filename, 
		       O_RDWR|O_CREAT|O_EXCL,      open_mode);
      else
	SL_IRETURN(SL_EBADFILE, _("sl_open_file"));
      break;
    case SL_OPEN_FOR_RDWR:
      if (lstat_return == ENOENT)
      	fd = aud_open (FIL__, __LINE__, priv, filename, 
		       O_RDWR|O_CREAT|O_EXCL,      open_mode);
      else
	fd = aud_open (FIL__, __LINE__, priv, filename, 
		       O_RDWR|O_CREAT|O_NOATIME,   open_mode);
      break;
    case SL_OPEN_FOR_WTRUNC:
      if (lstat_return == ENOENT)
      	fd = aud_open (FIL__, __LINE__, priv, filename, 
		       O_WRONLY|O_CREAT|O_EXCL,            open_mode);
      else
	fd = aud_open (FIL__, __LINE__, priv, filename, 
		       O_WRONLY|O_CREAT|O_TRUNC|O_NOATIME, open_mode);
      break;
    case SL_OPEN_FOR_RWTRUNC:
      if (lstat_return == ENOENT)
      	fd = aud_open (FIL__, __LINE__, priv, filename, 
		       O_RDWR|O_CREAT|O_EXCL,            open_mode);
      else
	fd = aud_open (FIL__, __LINE__, priv, filename, 
		       O_RDWR|O_CREAT|O_TRUNC|O_NOATIME, open_mode);
      break;
    default:
      SL_IRETURN(SL_EINTERNAL, _("sl_open_file"));
      break;
    }

  if (fd < 0)
    {
      TPT(( 0, FIL__, __LINE__, _("msg=<Error opening: %s> errno=<%d>\n"), 
	    filename, errno));
      SL_IRETURN(SL_EBADFILE, _("sl_open_file"));
    }

  if (priv == SL_YESPRIV)
    sl_set_suid();
  stat_return = retry_fstat(fd, &buf);
  if (priv == SL_YESPRIV)
    sl_unset_suid();

  if (stat_return < 0)
    {
      close (fd);
      SL_IRETURN(SL_EBADFILE, _("sl_open_file"));
    }

  if (lstat_return != ENOENT && buf.st_ino != lbuf.st_ino)
    {
      close (fd);
      SL_IRETURN(SL_EBOGUS, _("sl_open_file"));
    }

 createTicket:

  /* Make entry.
   */
  if (fd >= MAXFD || ofiles[fd] != NULL)
    {
      close(fd);
      SL_IRETURN(SL_EINTERNAL, _("sl_open_file"));
    }


  if ( (ofiles[fd] = (SL_OFILE *) my_malloc(sizeof(SL_OFILE))) == NULL)
    {
      close(fd);
      SL_IRETURN(SL_EMEM, _("sl_open_file"));
    }
  if ( (ofiles[fd]->path = (char *) my_malloc( strlen(filename)+1) ) == NULL)
    {
      my_free(ofiles[fd]);
      ofiles[fd] = NULL;
      close(fd);
      SL_IRETURN(SL_EMEM, _("sl_open_file"));
    }

  /* Get a ticket.
   */
  ticket = sl_create_ticket(fd);

  if (SL_ISERROR(ticket))
    {
      (void) my_free (ofiles[fd]->path);
      (void) my_free (ofiles[fd]);
      close(fd);
      SL_IRETURN(ticket, _("sl_open_file"));
    }

  strcpy (ofiles[fd]->path, filename);                    /* Known to fit  */
  ofiles[fd]->ticket = ticket;
  ofiles[fd]->fd     = fd;

  SL_IRETURN(ticket, _("sl_open_file"));
}

static
int check_fname_priv (char * fname, int priv)
{
  SL_ENTER(_("check_fname_priv"));
  if (fname == NULL)
    SL_IRETURN(SL_ENULL, _("check_fname_priv"));
  if (priv != SL_YESPRIV && priv != SL_NOPRIV)
    SL_IRETURN(SL_EINTERNAL, _("check_fname_priv"));
  SL_IRETURN(SL_ENONE, _("check_fname_priv"));
}
  
SL_TICKET sl_open_write (char * fname, int priv)
{
  long status;
  SL_ENTER(_("sl_open_write"));

  if (SL_ENONE != (status = check_fname_priv (fname, priv)))
    SL_IRETURN(status, _("sl_open_write"));

  status = sl_open_file(fname, SL_OPEN_FOR_WRITE, priv);
  SL_IRETURN(status, _("sl_open_write"));
}

SL_TICKET sl_open_read (char * fname, int priv)
{
  long status;
  SL_ENTER(_("sl_open_read"));

  if (SL_ENONE != (status = check_fname_priv (fname, priv)))
    {
      TPT(( 0, FIL__, __LINE__, 
	    _("msg=<Error in check_fname_priv.> status=<%ld>\n"), 
	    fname, status));
      SL_IRETURN(status, _("sl_open_read"));
    }

  status = sl_open_file(fname, SL_OPEN_FOR_READ, priv);
  SL_IRETURN(status, _("sl_open_read"));
}

SL_TICKET sl_open_fastread (char * fname, int priv)
{
  long status;
  SL_ENTER(_("sl_open_fastread"));

  if (SL_ENONE != (status = check_fname_priv (fname, priv)))
    SL_IRETURN(status, _("sl_open_read"));

  status = sl_open_file(fname, SL_OPEN_FOR_FASTREAD, priv);
  SL_IRETURN(status, _("sl_open_fastread"));
}

SL_TICKET sl_open_rdwr (char * fname, int priv)
{
  long status;
  SL_ENTER(_("sl_open_rdwr"));

  if (SL_ENONE != (status = check_fname_priv (fname, priv)))
    SL_IRETURN(status, _("sl_open_rdwr"));

  status = sl_open_file(fname, SL_OPEN_FOR_RDWR, priv);
  SL_IRETURN(status, _("sl_open_rdwr"));
}

SL_TICKET sl_open_safe_rdwr (char * fname, int priv)
{
  long status;
  SL_ENTER(_("sl_open_rdwr"));

  if (SL_ENONE != (status = check_fname_priv (fname, priv)))
    SL_IRETURN(status, _("sl_open_rdwr"));

  status = sl_open_file(fname, SL_OPEN_SAFE_RDWR, priv);
  SL_IRETURN(status, _("sl_open_rdwr"));
}

SL_TICKET sl_open_write_trunc (char * fname, int priv)
{
  long status;
  SL_ENTER(_("sl_open_write_trunc"));

  if (SL_ENONE != (status = check_fname_priv (fname, priv)))
    SL_IRETURN(status, _("sl_open_write_trunc"));

  status = sl_open_file(fname, SL_OPEN_FOR_WTRUNC, priv);
  SL_IRETURN(status, _("sl_open_write_trunc"));
}

SL_TICKET sl_open_rdwr_trunc (char * fname, int priv)
{
  long status;
  SL_ENTER(_("sl_open_rdwr_trunc"));

  if (SL_ENONE != (status = check_fname_priv (fname, priv)))
    SL_IRETURN(status, _("sl_open_rdwr_trunc"));

  status = sl_open_file(fname, SL_OPEN_FOR_RWTRUNC, priv);
  SL_IRETURN(status, _("sl_open_rdwr_trunc"));
}


int get_the_fd (SL_TICKET ticket)
{
  int fd;

  SL_ENTER(_("get_the_fd"));

  if (SL_ISERROR(fd = sl_read_ticket(ticket)))
    SL_IRETURN(fd, _("get_the_fd"));

  if (ofiles[fd] == NULL || fd != ofiles[fd]->fd || fd < 0)
    SL_IRETURN(SL_EINTERNAL, _("get_the_fd"));
  SL_IRETURN(fd, _("get_the_fd"));
}

int sl_close (SL_TICKET ticket) 
{
  register int fd;

  SL_ENTER(_("sl_close"));

  if (SL_ISERROR(fd = get_the_fd (ticket)))
    SL_IRETURN(fd, _("sl_close"));

  /* This may fail, but what to do then ?
   */
  if (0 != close(fd) && ofiles[fd] != NULL)
    {
      TPT((0, FIL__, __LINE__, 
	   _("msg=<Error closing file.>, path=<%s>, fd=<%d>\n"), 
	   ofiles[fd]->path, fd));
    }

  if (ofiles[fd] != NULL)
    {
      (void) my_free(ofiles[fd]->path);
      (void) my_free(ofiles[fd]);
      ofiles[fd] = NULL;
    }

  SL_IRETURN(SL_ENONE, _("sl_close")); 
}

int sl_dropall(int fd)
{
  while (fd < MAXFD)
    {
      if (ofiles[fd] != NULL)
	{
	  if (ofiles[fd]->path != NULL)
	    (void) my_free(ofiles[fd]->path);
	  (void) my_free(ofiles[fd]);
	  ofiles[fd] = NULL;
	}
      ++fd;
    }
  return 0;
}


int sl_unlink (SL_TICKET ticket) 
{
  register int fd;

  SL_ENTER(_("sl_unlink"));

  if (SL_ISERROR(fd = get_the_fd(ticket)))
    SL_IRETURN(fd, _("sl_unlink"));

  if (retry_aud_unlink(FIL__, __LINE__, ofiles[fd]->path) < 0)
    SL_IRETURN(SL_EUNLINK, _("sl_unlink"));

  SL_IRETURN(SL_ENONE, _("sl_unlink")); 
}

  
int sl_seek (SL_TICKET ticket, off_t off_data) 
{
  register int fd;

  SL_ENTER(_("sl_seek"));

  if (SL_ISERROR(fd = get_the_fd(ticket)))
    SL_IRETURN(fd, _("sl_seek"));

  if (lseek(fd, off_data, SEEK_SET) == (off_t)-1)
    SL_IRETURN(SL_EREWIND, _("sl_seek"));

  SL_IRETURN(SL_ENONE, _("sl_seek")); 
}
    
int sl_rewind (SL_TICKET ticket) 
{
  register int fd;

  SL_ENTER(_("sl_rewind"));

  if (SL_ISERROR(fd = get_the_fd(ticket)))
    SL_IRETURN(fd, _("sl_rewind"));

  if (lseek (fd, 0L, SEEK_SET) == (off_t)-1)
    SL_IRETURN(SL_EREWIND, _("sl_rewind"));

  SL_IRETURN(SL_ENONE, _("sl_rewind")); 
}

int sl_forward (SL_TICKET ticket) 
{
  register int fd;

  SL_ENTER(_("sl_forward"));

  if (SL_ISERROR(fd = get_the_fd(ticket)))
    SL_IRETURN(fd, _("sl_forward"));

  if (lseek (fd, 0L, SEEK_END) == (off_t)-1)
    SL_IRETURN(SL_EFORWARD, _("sl_forward"));

  SL_IRETURN(SL_ENONE, _("sl_forward")); 
}

int sl_sync (SL_TICKET ticket) 
{
  register int fd;

  SL_ENTER(_("sl_sync"));

  if (SL_ISERROR(fd = get_the_fd(ticket)))
    SL_IRETURN(fd, _("sl_sync"));

  if (fsync (fd) == -1)
    SL_IRETURN(SL_ESYNC, _("sl_sync"));

  SL_IRETURN(SL_ENONE, _("sl_sync")); 
}


int sl_read (SL_TICKET ticket, void * buf_in, size_t count)
{
  int fd;
  int byteread = 0;
  int bytes    = 0;

  char * buf;

  SL_ENTER(_("sl_read"));

  if (count < 1)
    SL_IRETURN(SL_ERANGE, _("sl_read"));
  if (buf_in == NULL)
    SL_IRETURN(SL_ENULL, _("sl_read"));

  if (SL_ISERROR(fd = get_the_fd(ticket)))
    SL_IRETURN(fd, _("sl_read"));

  buf = (char *) buf_in;

  do 
    {
      byteread = read (fd, buf, count);
      if (byteread > 0) 
	{
	  bytes += byteread; count -= byteread;
	}  
      buf += byteread;
    } while ( byteread > 0 || 
	      ( byteread < 0 && (errno == EINTR || errno == EAGAIN)) 
	      );

 
  if (byteread == (-1))
    SL_IRETURN(SL_EREAD, _("sl_read"));

   SL_IRETURN(bytes, _("sl_read"));
}


int sl_write (SL_TICKET ticket, void * msg_in, long nbytes)
{
  long bytewritten;
  long bytecount;
  int  fd;

  char * msg; 

  SL_ENTER(_("sl_write"));

  if (nbytes < 1)
    SL_IRETURN(SL_ERANGE, _("sl_write"));
  if (msg_in == NULL)
    SL_IRETURN(SL_ENULL, _("sl_write"));
  if (SL_ISERROR(fd = get_the_fd(ticket)))
    SL_IRETURN(fd, _("sl_write"));

  msg = (char *) msg_in;

  /* write
   */
  bytecount    = 0;
  bytewritten  = 0;
  while (bytecount < nbytes) 
    {    
      if ((bytewritten = write (fd, msg, nbytes-bytecount)) > 0) 
	{
	  bytecount += bytewritten;
	  msg       += bytewritten;    /* move buffer pointer forward */
	}
      else if (bytewritten <= 0)
	{
	  if ( errno == EINTR || errno == EAGAIN) /* try again */
	      continue;
	  else 
	    SL_IRETURN(SL_EWRITE, _("sl_write"));
	}
    }
  SL_IRETURN(SL_ENONE, _("sl_write"));
}

int sl_write_line (SL_TICKET ticket, void * msg, long nbytes)
{
  int  status;

  SL_ENTER(_("sl_write_line"));

  status = sl_write(ticket,  msg, nbytes); 
  if (!SL_ISERROR(status))
    status = sl_write(ticket,  "\n", 1);

  SL_IRETURN(status, _("sl_write_line"));
}


/* ---------------------------------------------------------------- 
 *
 *    Trustfile interface
 *
 * ---------------------------------------------------------------- */

extern long rootonly[];
extern int  EUIDSLOT;

extern char tf_path[MAXFILENAME];	/* Error path for trust function. */
extern long tf_euid;	                /* Space for EUID of process.     */


char * sl_error_string(int errorcode)
{
  switch (errorcode)
    {
    case SL_EBOGUS: 
      return _("Bogus file. Modified during access.");
    case SL_EWRITE: 
      return _("Write error.");
    case SL_EREAD: 
      return _("Read error.");
    case SL_ESYNC: 
      return _("Error in fsync().");
    case SL_EFORWARD: 
      return _("Error in lseek().");
    case SL_EREWIND: 
      return _("Error in lseek().");
    case SL_EUNLINK: 
      return _("Error in unlink().");
    case SL_EMEM: 
      return _("Out of memory.");
    case SL_EINTERNAL: 
      return _("Internal error.");
    case SL_ETICKET:
      return _("Bad ticket.");
    case SL_EREPEAT: 
      return _("Illegal repeated use of function.");
    case SL_ERANGE: 
      return _("Argument out of range.");
    case SL_ENULL: 
      return _("Dereferenced NULL pointer.");

    case SL_EBADUID: 
      return _("Owner not trustworthy.");
    case SL_EBADGID:
      return _("Group writeable and member not trustworthy.");
    case SL_EBADOTH:
      return _("World writeable.");
    case SL_EBADFILE:
      return _("File access error.");
    case SL_EBADNAME:
      return _("Invalid filename (prob. too long).");

    case SL_ETRUNC:
      return _("Truncation occured.");
    case SL_ESTAT:
      return _("stat() failed.");
    default:
      return _("Unknown error.");
    }
}



char * sl_trust_errfile()
{
  return &tf_path[0];
}


int  sl_trust_add_user (long pwid)
{
  static int count = 0;
  SL_ENTER(_("sl_trust_add_user"));

  if (count == 6)
    SL_IRETURN(SL_ERANGE, _("sl_trust_add_user"));
  
  rootonly[EUIDSLOT] = pwid;
  ++EUIDSLOT;
  ++count;
  SL_IRETURN(SL_ENONE, _("sl_trust_add_user"));
}

int sl_trustfile_euid(char * filename, long teuid)
{
  long status;
  SL_ENTER(_("sl_trustfile_euid"));

  tf_path[0] = '\0';
  if (filename == NULL)
    SL_IRETURN(SL_EBADFILE, _("sl_trustfile_euid"));

  tf_euid = teuid;
  status = sl_trustfile(filename, NULL, NULL);
  SL_IRETURN(status, _("sl_trustfile_euid"));
}


