/* SAMHAIN file system integrity testing                                   */
/* Copyright (C) 1999 Rainer Wichmann                                      */
/*                                                                         */
/*  This program is free software; you can redistribute it                 */
/*  and/or modify                                                          */
/*  it under the terms of the GNU General Public License as                */
/*  published by                                                           */
/*  the Free Software Foundation; either version 2 of the License, or      */
/*  (at your option) any later version.                                    */
/*                                                                         */
/*  This program is distributed in the hope that it will be useful,        */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of         */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the          */
/*  GNU General Public License for more details.                           */
/*                                                                         */
/*  You should have received a copy of the GNU General Public License      */
/*  along with this program; if not, write to the Free Software            */
/*  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.              */

#include "config_xor.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <errno.h>

#ifdef SH_USE_UTMP

#ifdef HAVE_UTMPX_H

#include <utmpx.h>
#define SH_UTMP_S utmpx
#undef  ut_name
#define ut_name ut_user
#ifdef HAVE_UTXTIME
#undef  ut_time
#define ut_time        ut_xtime
#else
#undef  ut_time
#define ut_time        ut_tv.tv_sec
#endif

#else
#include <utmp.h>
#define SH_UTMP_S utmp
#endif


#ifdef HAVE_PATHS_H
#include <paths.h>
#endif

#undef  FIL__
#define FIL__  _("sh_utmp.c")

#if defined (SH_WITH_CLIENT) || defined (SH_STANDALONE) 


#include "samhain.h"
#include "sh_utils.h"
#include "sh_error.h"
#include "sh_modules.h"
#include "sh_utmp.h"


#ifdef TM_IN_SYS_TIME
#include <sys/time.h>
#else
#include <time.h>
#endif

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif 

#ifdef HAVE_DIRENT_H
#include <dirent.h>
#define NAMLEN(dirent) sl_strlen((dirent)->d_name)
#else
#define dirent direct
#define NAMLEN(dirent) (dirent)->d_namlen
#ifdef HAVE_SYS_NDIR_H
#include <sys/ndir.h>
#endif
#ifdef HAVE_SYS_DIR_H
#include <sys/dir.h>
#endif
#ifdef HAVE_NDIR_H
#include <ndir.h>
#endif
#endif

#ifndef HAVE_LSTAT
#define lstat stat
#endif 

#ifndef UT_LINESIZE
#define UT_LINESIZE           12
#endif
#ifndef UT_NAMESIZE
#define UT_NAMESIZE            8
#endif
#ifndef UT_HOSTSIZE
#define UT_HOSTSIZE           16
#endif

#ifdef HAVE_UTMPX_H

#ifndef _PATH_UTMP
#ifdef   UTMPX_FILE
#define _PATH_UTMP   UTMPX_FILE
#else  
#error  You must define UTMPX_FILE in the file config.h 
#endif
#endif
#ifndef _PATH_WTMP
#ifdef   WTMPX_FILE
#define _PATH_WTMP   WTMPX_FILE
#else
#error  You must define WTMPX_FILE in the file config.h
#endif
#endif

#else

#ifndef _PATH_UTMP
#ifdef   UTMP_FILE
#define _PATH_UTMP   UTMP_FILE
#else  
#error  You must define UTMP_FILE in the file config.h 
#endif
#endif
#ifndef _PATH_WTMP
#ifdef   WTMP_FILE
#define _PATH_WTMP   WTMP_FILE
#else
#error  You must define WTMP_FILE in the file config.h
#endif
#endif

#endif

typedef struct log_user {
  char                ut_tty[UT_LINESIZE+1];    
  char                name[UT_NAMESIZE+1];
  char                ut_host[UT_HOSTSIZE+1]; 
  time_t              time; 
  struct log_user   * next;
} blah_utmp;

#ifdef HAVE_UTTYPE
static char   terminated_line[UT_HOSTSIZE]; 
#endif

static struct SH_UTMP_S save_utmp;

static void sh_utmp_logout_morechecks(struct log_user   * user);
static void sh_utmp_login_morechecks(struct SH_UTMP_S * ut);
static void sh_utmp_addlogin (struct SH_UTMP_S * ut);
static void sh_utmp_check_internal(int mode);

int    ShUtmpLoginSolo    = 7;
int    ShUtmpLoginMulti   = 7;
int    ShUtmpLogout       = 7;
int    ShUtmpActive       = BAD;
time_t ShUtmpInterval     = 300;

sh_rconf sh_utmp_table[] = {
  {
    N_("severitylogin"),
    sh_utmp_set_login_solo
  },
  {
    N_("severityloginmulti"),
    sh_utmp_set_login_multi
  },
  {
    N_("severitylogout"),
    sh_utmp_set_logout_good
  },
  {
    N_("logincheckactive"),
    sh_utmp_set_login_activate
  },
  {
    N_("logincheckinterval"),
    sh_utmp_set_login_timer
  },
  {
    NULL,
    NULL
  },
};




#if defined (HAVE_SETUTENT) && defined (USE_SETUTENT)

#ifdef HAVE_UTMPX_H

#define sh_utmp_utmpname     utmpxname
#define sh_utmp_setutent     setutxent
#define sh_utmp_endutent     endutxent
#define sh_utmp_getutent     getutxent
#define sh_utmp_getutid      getutxid
#define sh_utmp_getutline    getutxline

#else

#define sh_utmp_utmpname     utmpname
#define sh_utmp_setutent     setutent
#define sh_utmp_endutent     endutent
#define sh_utmp_getutent     getutent
#define sh_utmp_getutid      getutid
#define sh_utmp_getutline    getutline

#endif

#else

/* BSD lacks getutent() etc.
 * utmpname(), setutent(), and endutent() return void,
 * so we do not perform much error handling.
 * Errors must be recognized by getutent() returning NULL.
 * Apparently, the application cannot check whether wtmp is empty,
 * or whether there was an fopen() error.
 */

static FILE * sh_utmpfile = NULL;
static char   sh_utmppath[80] = _PATH_UTMP;
static long   sh_utmp_feed_forward = 0;

static void sh_utmp_utmpname(const char * str)
{
  if (sh_utmpfile != NULL)
    {
      MBLK( fclose (sh_utmpfile); )
      sh_utmpfile = NULL;
    }

  sl_strlcpy (sh_utmppath, str, 80);
  return;
}

static void sh_utmp_setutent(void)
{
  int error;
  int fd;

  SL_ENTER(_("sh_utmp_setutent"));

  ASSERT((sh_utmppath != NULL), _("sh_utmppath != NULL"));

  if (sh_utmppath == NULL)
    return;

  if (sh_utmpfile == NULL) 
    {
      fd = aud_open (FIL__, __LINE__, SL_NOPRIV, sh_utmppath, O_RDONLY, 0);
      if (fd >= 0)
	{
	  MBLK( sh_utmpfile = fdopen(fd, "r"); )
	}

      /* -- If (sh_utmpfile == NULL) then either the open() or the fdopen()
       *    has failed.
       */
      if (sh_utmpfile == NULL) 
	{
	  error = errno;
	  sh_error_handle ((-1), FIL__, __LINE__, error, MSG_E_ACCESS,
			   (long) sh.real.uid, sh_utmppath);
	  SL_RET0(_("sh_utmp_setutent"));
	}
    }
  (void) fseek (sh_utmpfile, 0L, SEEK_SET);
  (void) fseek (sh_utmpfile, sh_utmp_feed_forward, SEEK_CUR); 
  clearerr (sh_utmpfile);
  SL_RET0(_("sh_utmp_setutent"));
}

static void sh_utmp_endutent(void)
{
  MBLK( fclose(sh_utmpfile); )
  sh_utmpfile = NULL;
  return;
}

static struct SH_UTMP_S * sh_utmp_getutent(void)
{
  size_t in;
  static struct SH_UTMP_S out;

  SL_ENTER(_("sh_utmp_getutent"));

  ASSERT_RET((sh_utmpfile != NULL), _("sh_utmpfile != NULL"), (NULL))

  in = fread (&out, sizeof(struct SH_UTMP_S), 1, sh_utmpfile);

  if (in != 1) 
    {
      if (ferror (sh_utmpfile) != 0) 
	{
	  clearerr (sh_utmpfile);
	  SL_RETURN(NULL, _("sh_utmp_getutent"));
	} 
      else 
	{
	  SL_RETURN(NULL, _("sh_utmp_getutent"));
	}
    }
  SL_RETURN(&out, _("sh_utmp_getutent"));
}

#ifdef USE_UNUSED

static struct SH_UTMP_S * sh_utmp_getutline(struct SH_UTMP_S * ut)
{
  struct SH_UTMP_S * out;
 
  while (1) {
      if ((out = sh_utmp_getutent()) == NULL) {
       	return NULL;
      }
#ifdef HAVE_UTTYPE  
      if (out->ut_type == USER_PROCESS || out->ut_type == LOGIN_PROCESS)
	if (sl_strcmp(ut->ut_line, out->ut_line) == 0) 
	  return out;
#else
      if ( 0 != sl_strncmp (out->ut_name, "reboot",   6) &&
	   0 != sl_strncmp (out->ut_name, "shutdown", 8) &&
	   0 != sl_strncmp (out->ut_name, "date",     4) )
	return out;
#endif
  }
  return NULL;
}

static struct SH_UTMP_S * sh_utmp_getutid(struct SH_UTMP_S * ut)
{
#ifdef HAVE_UTTYPE  
  struct SH_UTMP_S * out;

  if (ut->ut_type == RUN_LVL  || ut->ut_type == BOOT_TIME ||
      ut->ut_type == NEW_TIME || ut->ut_type == OLD_TIME) 
    {
      while (1) {
	if ((out = sh_utmp_getutent()) == NULL) {
	  return NULL;
	}
	if (out->ut_type == ut->ut_type) 
	  return out;
      }
    } 
  else if (ut->ut_type == INIT_PROCESS || ut->ut_type == LOGIN_PROCESS ||
	   ut->ut_type == USER_PROCESS || ut->ut_type == DEAD_PROCESS ) 
    {
      while (1) {
	if ((out = sh_utmp_getutent()) == NULL) {
	  return NULL;
	}
	if (sl_strcmp(ut->ut_id, out->ut_id) == 0) 
	  return out;
      }
    }
#endif
  return NULL;
}
/* #ifdef USE_UNUSED */
#endif

/* #ifdef HAVE_SETUTENT */
#endif



static struct log_user   * userlist   = NULL;
static time_t  lastcheck;

/*************
 *
 * module init
 *
 *************/
int sh_utmp_init ()
{
  if (ShUtmpActive == BAD)
    return (-1);

  lastcheck  = time (NULL);
  userlist   = NULL;
  memset (&save_utmp, 0, sizeof(struct SH_UTMP_S));
  sh_utmp_check_internal (2); /* current logins */
  sh_utmp_check_internal (0);
  return (0);
}

/*************
 *
 * module cleanup
 *
 *************/
int sh_utmp_end ()
{
  struct log_user * user    = userlist;
  struct log_user * userold;
  while (user)
    {
      userold = user;
      user    = user->next;
      SH_FREE(userold);
    }
  return (0);
}


/*************
 *
 * module timer
 *
 *************/
int sh_utmp_timer (unsigned long tcurrent)
{
  if ((time_t) (tcurrent - lastcheck) > ShUtmpInterval)
    {
      lastcheck  = tcurrent;
      return (-1);
    }
  return 0;
}

/*************
 *
 * module check
 *
 *************/
int sh_utmp_check ()
{
  sh_error_handle ((-1), FIL__, __LINE__, 0, MSG_UT_CHECK);
  sh_utmp_check_internal (1);

  return 0;
}

/*************
 *
 * module setup
 *
 *************/

int sh_utmp_set_login_solo  (char * c)
{
  char tmp[32];
  tmp[0] = '='; tmp[1] = '\0';
  sl_strlcat (tmp, c, 32);
  sh_error_set_level (tmp, &ShUtmpLoginSolo);
  return 0;
}

int sh_utmp_set_login_multi (char * c)
{
  char tmp[32];
  tmp[0] = '='; tmp[1] = '\0';
  sl_strlcat (tmp, c, 32);
  sh_error_set_level (tmp, &ShUtmpLoginMulti);
  return 0;
}

int sh_utmp_set_logout_good (char * c)
{
  char tmp[32];
  tmp[0] = '='; tmp[1] = '\0';
  sl_strlcat (tmp, c, 32);
  sh_error_set_level (tmp, &ShUtmpLogout);
  return 0;
}

int sh_utmp_set_login_timer (char * c)
{
  long val;

  val = strtol (c, (char **)NULL, 10);
  if (val <= 0)
    sh_error_handle ((-1), FIL__, __LINE__, EINVAL, MSG_EINVALS,
                      _("utmp timer"), c);

  val = (val <= 0 ? 60 : val);

  ShUtmpInterval = (time_t) val;
  return 0;
}

int sh_utmp_set_login_activate (char * c)
{
  if (c != NULL)
    {
      if (c[0] == '0' || c[0] == 'n' || c[0] == 'N')
	{
	  ShUtmpActive = BAD;
	  return (0);
	}
      else if (c[0] == '1' || c[0] == 'y' || c[0] == 'Y')
	{
	  ShUtmpActive = GOOD;
	  return (0);
	}
    }

  sh_error_handle ((-1), FIL__, __LINE__, EINVAL, MSG_EINVALS,
                      _("utmp activate"), c);

  ShUtmpActive = BAD;
  return (-1);
}



/* for each login:
 *    - allocate a log record
 *    - link device.ut_record -> log_record
 *    - link user.ut_record   -> log_record
 */

static void sh_utmp_addlogin (struct SH_UTMP_S * ut)
{
  struct log_user   * user     = userlist;
  struct log_user   * username = userlist;
  struct log_user   * userold  = userlist;

  char   ttt[TIM_MAX];

  if (ut->ut_line[0] == '\0')
    return;

  /* for some stupid reason, AIX repeats the wtmp entry for logouts
   * with ssh
   */
  if (memcmp (&save_utmp, ut, sizeof(struct SH_UTMP_S)) == 0)
    {
      memset(&save_utmp, '\0', sizeof(struct SH_UTMP_S));
      return;
    }
  memcpy (&save_utmp, ut, sizeof(struct SH_UTMP_S));
      

  /* ------- find user -------- 
   */
  while (user != NULL) 
    {
      if (0 == sl_strncmp(user->ut_tty, ut->ut_line, UT_LINESIZE) ) 
	break;
      userold = user;
      user = user->next;
    }

  while (username != NULL) 
    {
      if (0 == sl_strncmp(username->name, ut->ut_name, UT_NAMESIZE) ) 
	break;
      username = username->next;
    }

  
#ifdef HAVE_UTTYPE  
  /* ---------- LOGIN -------------- */
  if (ut->ut_type == USER_PROCESS) 
    {
      if (user == NULL)
	{
	  user = SH_ALLOC(sizeof(struct log_user));
	  user->next       = userlist;
	  userlist         = user;
	}
      sl_strlcpy(user->ut_tty,  ut->ut_line, UT_LINESIZE+1);
      sl_strlcpy(user->name,    ut->ut_name, UT_NAMESIZE+1);
#ifdef HAVE_UTHOST
      sl_strlcpy(user->ut_host, ut->ut_host, UT_HOSTSIZE+1);
#endif
      user->time = ut->ut_time;


      if (username == NULL                              /* not yet logged in */
	  || 0 == sl_strncmp(ut->ut_line, _("ttyp"), 4) /* in virt. console  */
	  || 0 == sl_strncmp(ut->ut_line, _("ttyq"), 4) /* in virt. console  */
	  ) {
	sl_strlcpy(ttt, sh_unix_time (user->time), TIM_MAX);
	sh_error_handle( ShUtmpLoginSolo, FIL__, __LINE__, 0,
#ifdef HAVE_UTHOST
			 MSG_UT_LG1A,
#else
			 MSG_UT_LG1B,
#endif
			 user->name,
			 user->ut_tty,
#ifdef HAVE_UTHOST
			 user->ut_host,
#endif
			 ttt
			 );
      } else
	if (0 != sl_strncmp(ut->ut_line, _("ttyp"), 4) &&
	    0 != sl_strncmp(ut->ut_line, _("ttyq"), 4))
	{       
	sl_strlcpy(ttt, sh_unix_time (user->time), TIM_MAX);
	sh_error_handle( ShUtmpLoginMulti, FIL__, __LINE__, 0,
#ifdef HAVE_UTHOST
			 MSG_UT_LG2A,
#else
			 MSG_UT_LG2B,
#endif
			 user->name,
			 user->ut_tty,
#ifdef HAVE_UTHOST
			 user->ut_host,
#endif
			 ttt
			 );
      }
      
      sh_utmp_login_morechecks(ut);
      return;
    }


  /* ---------  LOGOUT ---------------- */
  else if (ut->ut_name[0] == '\0'
	   || ut->ut_type == DEAD_PROCESS  /* solaris does not clear ut_name */
	   )
    {
      if (user != NULL)
	{
	  sl_strlcpy(ttt, sh_unix_time (user->time), TIM_MAX);
	  sh_error_handle( ShUtmpLogout, FIL__, __LINE__, 0,
#ifdef HAVE_UTHOST
			   MSG_UT_LG3A,
#else
			   MSG_UT_LG3B,
#endif
			   user->name,
			   user->ut_tty,
#ifdef HAVE_UTHOST
			   user->ut_host,
#endif
			   ttt
			   );
	  userold->next = user->next;
	  if (user == userlist)
	    userlist = user->next;
	  sh_utmp_logout_morechecks(user);
	  SH_FREE(user);
	  user = NULL;
	}
      else
	{
	  sl_strlcpy(terminated_line, ut->ut_line, UT_HOSTSIZE);
	  sl_strlcpy(ttt, sh_unix_time (ut->ut_time), TIM_MAX);
	  sh_error_handle( ShUtmpLogout, FIL__, __LINE__, 0,
			   MSG_UT_LG3C,
			   terminated_line,
			   ttt
			   );
	}
      return;
    }

  /* default */
  return;

  /* #ifdef HAVE_UTTYPE                   */
#else

  if (user == NULL)   /* probably a login */
    {
      user = SH_ALLOC(sizeof(struct log_user));
      sl_strlcpy(user->ut_tty,  ut->ut_line, UT_LINESIZE+1);
      sl_strlcpy(user->name,    ut->ut_name, UT_NAMESIZE+1);
#ifdef HAVE_UTHOST
      sl_strlcpy(user->ut_host, ut->ut_host, UT_HOSTSIZE+1);
#endif
      user->time       = ut->ut_time;
      user->next       = userlist;
      userlist         = user;

      sl_strlcpy(ttt, sh_unix_time (user->time), TIM_MAX);


      sh_error_handle( ShUtmpLoginSolo, FIL__, __LINE__, 0,
#ifdef HAVE_UTHOST
		       MSG_UT_LG1A,
#else
		       MSG_UT_LG1B,
#endif
		       user->name,
		       user->ut_tty,
#ifdef HAVE_UTHOST
		       user->ut_host,
#endif
		       ttt
		       );
      sh_utmp_login_morechecks(ut);
    }
  else  /* probably a logout */
    {
      sl_strlcpy(ttt, sh_unix_time (user->time), TIM_MAX);

      sh_error_handle( ShUtmpLogout, FIL__, __LINE__, 0,
#ifdef HAVE_UTHOST
		       MSG_UT_LG2A,
#else
		       MSG_UT_LG2B,
#endif
		       user->name,
		       user->ut_tty,
#ifdef HAVE_UTHOST
		       user->ut_host,
#endif
		       ttt
		       );
      sh_utmp_logout_morechecks(user);
      userold->next = user->next;
      SH_FREE(user);
      user = NULL;
    }

  return;
#endif
}

static time_t        lastmod  = 0;
static off_t         lastsize = 0;
static unsigned long lastread = 0;

char * mode_path[] = { _PATH_WTMP, _PATH_WTMP, _PATH_UTMP };
    
static void sh_utmp_check_internal (int mode)
{
  struct stat   buf;
  int           error;
  struct SH_UTMP_S * ut;
  unsigned long this_read = 0;

  /* error if no access
   */
  if (0 != retry_lstat(mode_path[mode], &buf)) 
    {
      error = errno;
      sh_error_handle((-1), FIL__, __LINE__, error, MSG_E_ACCESS,
		      (long) sh.real.uid, mode_path[mode]);
      return;
    }

  /* modification time
   */
  if (mode < 2)
    {
      if (buf.st_mtime <= lastmod) 
	return;
      else
	lastmod = buf.st_mtime;
    }

  /* file size
   */
  if (buf.st_size < lastsize && mode < 2) 
    { 
      sh_error_handle((-1), FIL__, __LINE__, 0, MSG_UT_ROT,
		      mode_path[mode]);
      lastread = 0;
#ifndef USE_SETUTENT
      sh_utmp_feed_forward = 0L;
#endif
    }

  if (mode < 2)
    lastsize = buf.st_size;

  if (buf.st_size == 0) 
    return;

  sh_utmp_utmpname(mode_path[mode]);
  sh_utmp_setutent();

  /* 
   * feed forward if initializing
   * we need to do this here
   */
  if (mode < 2)
    {
      while (this_read < lastread) {
	ut = sh_utmp_getutent();
	++this_read;
      }
    }

  /* start reading
   */
  this_read = 0;
  while (1) {
    ut = sh_utmp_getutent();
    if (ut == NULL) 
      break;
    /* modified: ut_user --> ut_name */
    if (mode == 1 || (mode == 2 && ut->ut_name[0] != '\0'
#ifdef HAVE_UTTYPE
		      && ut->ut_type != DEAD_PROCESS
#endif
		      ))
      sh_utmp_addlogin (ut);
    /*************************************************
    printf("%8s | %10s | %10s | %d %d | %16s | %ld\n", 
	   ut->ut_name, ut->ut_id, ut->ut_line, 
	   (int) ut->ut_type, (int) ut->ut_pid, 
	   ut->ut_host, ut->ut_time);
    ***************************************************/
    ++this_read;
  }

  sh_utmp_endutent();

  if (mode < 2)
    {
      lastread += this_read;
#ifndef USE_SETUTENT
      sh_utmp_feed_forward += (long) (this_read * sizeof(struct SH_UTMP_S));
      lastread = 0;
#endif
    }

  return;
}


static void sh_utmp_login_morechecks(struct SH_UTMP_S * ut)
{
  if (ut)
    return;
  return;
}

static void sh_utmp_logout_morechecks(struct log_user   * user)
{
  if (user)
    return;
  return;
}

#endif


/* #ifdef SH_USE_UTMP */
#endif



