#include "config_xor.h"


#include <stdio.h>

#include <sys/types.h>
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif
#ifdef HAVE_MEMORY_H
#include <memory.h>
#endif

#if defined(HAVE_MLOCK) && !defined(HAVE_BROKEN_MLOCK)
#include <sys/mman.h>
#endif

#include "sh_tiger.h"

#include "sh_unix.h"
#include "sh_error.h"
#include "sh_utils.h"

#ifdef HAVE_LONG_LONG
typedef unsigned long long int word64;
#endif

#if defined(HAVE_INT_32)
typedef unsigned int sh_word32;
#define MYFORMAT   (_("%08X%08X%08X%08X%08X%08X"))
#define GPGFORMAT (_("%08X %08X %08X  %08X %08X %08X"))
#elif defined(HAVE_LONG_32)
typedef unsigned long sh_word32;
#define MYFORMAT   (_("%08lX%08lX%08lX%08lX%08lX%08lX"))
#define GPGFORMAT (_("%08lX %08lX %08lX  %08lX %08lX %08lX"))
#elif defined(HAVE_SHORT_32)
typedef unsigned short sh_word32;
#define MYFORMAT   (_("%08X%08X%08X%08X%08X%08X"))
#define GPGFORMAT (_("%08X %08X %08X  %08X %08X %08X"))
#else
#error No 32 byte type found !
#endif

typedef unsigned char sh_byte;

#undef  FIL__
#define FIL__  _("sh_tiger0.c")

#ifdef HAVE_LONG_LONG
void tiger_t(word64 *str, word64 length, word64 res[3]);
void tiger(word64 *str, word64 length, word64 res[3]);

#ifdef TIGER_DBG
static void tiger_dbg(word64 res[3], int step, 
		      unsigned long nblocks, unsigned long ncount)
{
  return;
}
#endif
#else
void tiger(sh_word32 *str, sh_word32 length, sh_word32 res[6]);
void tiger_t(sh_word32 *str, sh_word32 length, sh_word32 res[6]);

#ifdef TIGER_DBG
static 
void tiger_dbg(sh_word32 res[6], int step, 
	       unsigned long nblocks, unsigned long ncount)
{
    fprintf(stderr,                                     
            _("ST %d BLK %2ld CT %2ld %08lX %08lX %08lX %08lX %08lX %08lX\n"),
	    step,
	    nblocks,
	    ncount,
            (sh_word32)(res[1]), 
            (sh_word32)(res[0]), 
            (sh_word32)(res[3]), 
            (sh_word32)(res[2]), 
            (sh_word32)(res[5]), 
            (sh_word32)(res[4]) );
}
#endif
#endif

/* this is the wrapper function -- not part of the tiger reference
 * implementation
 */
SL_TICKET tiger_fd = (-1);


#ifdef HAVE_LONG_LONG
static
word64 * sh_tiger_hash_val (char * filename, 
			    TigerType what, 
			    unsigned long Length)
#else
static
sh_word32 * sh_tiger_hash_val (char * filename, 
			       TigerType what, 
			       unsigned long Length)
#endif
{
  SL_TICKET  fd;
  int  i, j;
  int  count;
  int  blk;
  char    * tmp;
  sh_byte * bptr;
  sh_byte buffer[PRIV_MAX+2];
  sh_byte bbuf[64];

  uid_t   euid;

  unsigned long ncount = 0, nblocks = 0;
  unsigned long  t, msb, lsb;

#ifdef HAVE_LONG_LONG
#define TIGER_CAST (word64*)
  static word64 res[3];
  res[0]=0x0123456789ABCDEFLL;
  res[1]=0xFEDCBA9876543210LL;
  res[2]=0xF096A5B4C3B2E187LL;
#else
#define TIGER_CAST (sh_word32*)
  static sh_word32 res[6];
  res[0]=0x89ABCDEF;
  res[1]=0x01234567;
  res[2]=0x76543210;
  res[3]=0xFEDCBA98;
  res[4]=0xC3B2E187;
  res[5]=0xF096A5B4;
#endif

  SL_ENTER(_("sh_tiger_hash_val"));

  if (what == TIGER_FILE || what == TIGER_FD) 
    {
      if (what == TIGER_FILE)
	{
	  TPT((0,FIL__, __LINE__, _("msg=<TIGER_FILE>, path=<%s>\n"),
	       filename));
	  fd = sl_open_read (filename, SL_YESPRIV);
	}
      else
	{
	  fd = tiger_fd;
	  TPT((0,FIL__, __LINE__, _("msg=<TIGER_FD>, fd=<%ld>\n"), tiger_fd));
	}

      if (SL_ISERROR (fd)) 
	{
	  TPT((0, FIL__, __LINE__, _("msg=<SL_ISERROR (%ld)>\n"), tiger_fd));
	  tmp = sh_util_safe_name (filename);
	  sl_get_euid(&euid);
	  sh_error_handle (ShDFLevel[SH_ERR_T_FILE], FIL__, __LINE__, EACCES,
			   MSG_E_ACCESS, (long) euid, tmp);
	  SH_FREE(tmp);
	  SL_RETURN( NULL, _("sh_tiger_hash_val"));
	}
      
#if defined(HAVE_MLOCK) && !defined(HAVE_BROKEN_MLOCK)
    if (skey->mlock_failed == GOOD) 
      {
	sl_set_suid ();
        if ( (-1) == mlock((char *) buffer, (PRIV_MAX+2) * sizeof(sh_byte))) 
          skey->mlock_failed = BAD;
	sl_unset_suid ();
      }
#else
    if (skey->mlock_failed == GOOD)
      skey->mlock_failed = BAD;
#endif

#ifdef TIGER_DBG
    tiger_dbg (res, 0, nblocks, ncount);
#endif

    while (1) 
      {
	count = sl_read (fd, buffer, PRIV_MAX);

	if (SL_ISERROR (count)) 
	  {
	    TPT((0, FIL__ , __LINE__ , _("msg=<SL_ISERROR (%ld)>\n"), count)); 
	    tmp = sh_util_safe_name (filename);
	    sh_error_handle (ShDFLevel[SH_ERR_T_FILE], FIL__, __LINE__, 0,
			     MSG_E_READ, tmp);
	    SH_FREE(tmp);
	    SL_RETURN( NULL, _("sh_tiger_hash_val"));
	  }

	/* TPT((0, FIL__, __LINE__ , _("msg=<Read %ld bytes>\n"), count)); */

	blk      = (count / 64); /* number of 64-byte words */

	/* nblocks += blk; number of 64-byte words */
	ncount = count - ((count/64)*64);
	
	bptr = buffer;
	for (i = 0; i < blk; ++i)
	  {
	    bptr = &buffer[64 * i];
	    
	    tiger_t(TIGER_CAST bptr, 64, res);
	    ++nblocks;
	    sh.stat.bytes_hashed += 64;
	    
#ifdef TIGER_DBG
	    tiger_dbg (res, 3, nblocks, ncount);
#endif
	  }
	
	if (blk < 8) 
	  break;
      }

    TPT((0, FIL__, __LINE__ , _("msg=<Last block.>\n")));

    /* copy incomplete block
     */
    j = 0; for (i = 0; i < 64; ++i) 
      bbuf[i] = '\0';
    for (i = (count/64) * 64; i < count; ++i)
      bbuf[j++] = buffer[i];

#ifdef TIGER_DBG
    tiger_dbg (res, 5, nblocks, ncount);
#endif

    msb = 0;
    t = nblocks;
    if( (lsb = t << 6) < t )    
      msb++;
    msb += t >> 26;
    t = lsb;
    if( (lsb = t + ncount) < t ) 
      msb++;
    t = lsb;
    if( (lsb = t << 3) < t )    
      msb++;
    msb += t >> 29;

    if( j < 56 ) 
      { 
        bbuf[j++] = 0x01; ++ncount;
        while( j < 56 )
	  { bbuf[j++] = 0; ++ncount; } 
      }
    else 
      { 
        bbuf[j++] = 0x01;
        while( j < 64 )
	  bbuf[j++] = 0;
	tiger_t(TIGER_CAST bbuf, 64, res);
	sh.stat.bytes_hashed += 64;
	++nblocks; ncount = 0;
        memset(bbuf, 0, 56 ); 
      }

#ifdef TIGER_DBG
    tiger_dbg (res, 6, nblocks, ncount);
#endif

    bbuf[56] = lsb      ;
    bbuf[57] = lsb >>  8;
    bbuf[58] = lsb >> 16;
    bbuf[59] = lsb >> 24;
    bbuf[60] = msb      ;
    bbuf[61] = msb >>  8;
    bbuf[62] = msb >> 16;
    bbuf[63] = msb >> 24;

    tiger_t(TIGER_CAST bbuf, 64, res);
    sh.stat.bytes_hashed += 64;
    
#ifdef TIGER_DBG
    tiger_dbg (res, 7, nblocks, ncount);
#endif

    memset (bbuf,   0, 64);
    memset (buffer, 0, PRIV_MAX);

    MUNLOCK( (char *) buffer, (PRIV_MAX+2) * sizeof(sh_byte));

    if (what == TIGER_FILE)
      sl_close (fd);
    else
      tiger_fd = (-1);


    SL_RETURN( res, _("sh_tiger_hash_val"));
  }

  if (what == TIGER_DATA && filename != NULL) 
    {
      /* TPT((0, FIL__, __LINE__, _("msg=<TIGER_DATA>\n"))); */
      tiger(TIGER_CAST filename,  (sh_word32) Length, res); 
      SL_RETURN(res, _("sh_tiger_hash_val"));
    }
  SL_RETURN( NULL, _("sh_tiger_hash_val"));
}

char * sh_tiger_hash (char * filename, TigerType what, unsigned long Length)
{
#ifdef HAVE_LONG_LONG
  word64 * res;
#else
  sh_word32 * res;
#endif
  static char out[KEY_LEN+1];

  SL_ENTER( _("sh_tiger_hash"));

  res = sh_tiger_hash_val (filename, 
			   what, 
			   Length);
  if (res != NULL)
    {
#ifdef HAVE_LONG_LONG
      sprintf(out,                                   /* known to fit  */
	      MYFORMAT,
	      (sh_word32)(res[0]>>32), 
	      (sh_word32)(res[0]), 
	      (sh_word32)(res[1]>>32), 
	      (sh_word32)(res[1]), 
	      (sh_word32)(res[2]>>32), 
	      (sh_word32)(res[2]) );
#else
      sprintf(out,                                   /* known to fit  */
	      MYFORMAT,
	      (sh_word32)(res[1]), 
	      (sh_word32)(res[0]), 
	      (sh_word32)(res[3]), 
	      (sh_word32)(res[2]), 
	      (sh_word32)(res[5]), 
	      (sh_word32)(res[4]) );
#endif
      out[KEY_LEN] = '\0';
      SL_RETURN( out, _("sh_tiger_hash"));

    }

   SL_RETURN( _("000000000000000000000000000000000000000000000000"), 
	      _("sh_tiger_hash"));
}

char * sh_tiger_hash_gpg (char * filename, TigerType what, 
			  unsigned long Length)
{
  int  len;
  char * out;
  char   outhash[48+6+1];
#ifdef HAVE_LONG_LONG
  word64 * res;
#else
  sh_word32 * res;
#endif

  SL_ENTER(_("sh_tiger_hash_gpg"));

  res = sh_tiger_hash_val (filename, 
			   what, 
			   Length);
  if (res != NULL)
    {
#ifdef HAVE_LONG_LONG
      sprintf(outhash,                               /* known to fit  */
	      GPGFORMAT,
	      (sh_word32)(res[0]>>32), 
	      (sh_word32)(res[0]), 
	      (sh_word32)(res[1]>>32), 
	      (sh_word32)(res[1]), 
	      (sh_word32)(res[2]>>32), 
	      (sh_word32)(res[2]) );
#else
      sprintf(outhash,                               /* known to fit  */
	      GPGFORMAT,
	      (sh_word32)(res[1]), 
	      (sh_word32)(res[0]), 
	      (sh_word32)(res[3]), 
	      (sh_word32)(res[2]), 
	      (sh_word32)(res[5]), 
	      (sh_word32)(res[4]) );
#endif
      outhash[48 + 6] = '\0';
    }
  else
    {
      sprintf(outhash,                               /* known to fit  */
	      _("00000000 00000000 00000000  00000000 00000000 00000000"));
    }

  if (what == TIGER_FILE)
    len = sl_strlen (filename) + 2 + 48 + 6;
  else
    len = 48 + 6;

  out = SH_ALLOC(len + 1);

  if (what == TIGER_FILE)
    {
      sl_strlcpy (out, filename, len+1);
      sl_strlcat (out,  _(": "), len+1);
      sl_strlcat (out,  outhash, len+1);
    }
  else
    {
      sl_strlcpy (out,  outhash, len+1);
    }
  SL_RETURN( out, _("sh_tiger_hash_gpg"));
}


UINT32 * sh_tiger_hash_uint32 (char * filename, 
			       TigerType what, 
			       unsigned long Length)
{
#ifdef HAVE_LONG_LONG
  word64 * res;
#else
  sh_word32 * res;
#endif

  static UINT32 out[6];

  SL_ENTER(_("sh_tiger_hash_uint32"));

  memset(out, 0, 6 * sizeof(UINT32));

  res = sh_tiger_hash_val (filename, 
			   what, 
			   Length);
  if (res != NULL)
    {
#ifdef HAVE_LONG_LONG
	out[0] =  (UINT32)(res[0]>>32); 
	out[1] =  (UINT32)(res[0]);
	out[2] =  (UINT32)(res[1]>>32); 
	out[3] =  (UINT32)(res[1]); 
	out[4] =  (UINT32)(res[2]>>32); 
	out[5] =  (UINT32)(res[2]);
#else
	out[0] =  (UINT32)(res[1]); 
	out[1] =  (UINT32)(res[0]);
	out[2] =  (UINT32)(res[3]); 
	out[3] =  (UINT32)(res[2]); 
	out[4] =  (UINT32)(res[5]); 
	out[5] =  (UINT32)(res[4]);
#endif
    }

  SL_RETURN(out, _("sh_tiger_hash_uint32"));
}
  



