/* SAMHAIN file system integrity testing                                   */
/* Copyright (C) 2001 Rainer Wichmann                                      */
/*                                                                         */
/*  This program is free software; you can redistribute it                 */
/*  and/or modify                                                          */
/*  it under the terms of the GNU General Public License as                */
/*  published by                                                           */
/*  the Free Software Foundation; either version 2 of the License, or      */
/*  (at your option) any later version.                                    */
/*                                                                         */
/*  This program is distributed in the hope that it will be useful,        */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of         */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the          */
/*  GNU General Public License for more details.                           */
/*                                                                         */
/*  You should have received a copy of the GNU General Public License      */
/*  along with this program; if not, write to the Free Software            */
/*  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.              */

#include "config_xor.h"


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <errno.h>
#include <limits.h>


#ifdef SH_USE_SUIDCHK

#undef  FIL__
#define FIL__  _("sh_suidchk.c")

#if defined (SH_WITH_CLIENT) || defined (SH_STANDALONE) 

#if TIME_WITH_SYS_TIME
#include <sys/time.h>
#include <time.h>
#else
#if HAVE_SYS_TIME_H
#include <sys/time.h>
#else
#include <time.h>
#endif
#endif

#ifdef HAVE_DIRENT_H
#include <dirent.h>
#define NAMLEN(dirent) sl_strlen((dirent)->d_name)
#else
#define dirent direct
#define NAMLEN(dirent) (dirent)->d_namlen
#ifdef HAVE_SYS_NDIR_H
#include <sys/ndir.h>
#endif
#ifdef HAVE_SYS_DIR_H
#include <sys/dir.h>
#endif
#ifdef HAVE_NDIR_H
#include <ndir.h>
#endif
#endif

#include "samhain.h"
#include "sh_utils.h"
#include "sh_error.h"
#include "sh_modules.h"
#include "sh_suidchk.h"
#include "sh_hash.h"


sh_rconf sh_suidchk_table[] = {
  {
    N_("severitysuidcheck"),
    sh_suidchk_set_severity
  },
  {
    N_("suidcheckactive"),
    sh_suidchk_set_activate
  },
  {
    N_("suidcheckinterval"),
    sh_suidchk_set_timer
  },
  {
    NULL,
    NULL
  },
};


static time_t  lastcheck         = (time_t) 0;
static int     ShSuidchkActive   = S_TRUE;
static int     ShSuidchkInterval = 7200;
static int     ShSuidchkSeverity = SH_ERR_SEVERE;


static char *
filesystem_type (char * path, char * relpath, struct stat * statp);

#ifndef PATH_MAX
#define PATH_MAX 1023
#endif

int sh_suidchk_check_internal (char * iname)
{
  DIR *           thisDir = NULL;
  struct dirent * thisEntry;
  char          * tmpcat;
  char          * tmp;
  struct stat     buf;
  int             status;
  char          * fs;
  long            sl_status;

  if (iname == NULL)
    {
      TPT((0, FIL__, __LINE__ , _("msg=<directory name is NULL>\n")));
      return (-1);
    }

  sl_set_suid();
  thisDir = opendir (iname);
  sl_unset_suid();

  if (thisDir == NULL)
    {
      status = errno;
      tmp = sh_util_safe_name(iname);
      sh_error_handle ((-1), FIL__, __LINE__, status, MSG_E_OPENDIR,
		       sh_error_message(status),
		       tmp );
      SH_FREE(tmp);
      return (-1);
    }

  do {
    thisEntry = readdir (thisDir);

    if (thisEntry != NULL) {

      if (sl_strcmp (thisEntry->d_name, ".") == 0)
	continue;

      if (sl_strcmp (thisEntry->d_name, "..") == 0)
	continue;

      tmpcat = SH_ALLOC(PATH_MAX+1);
      sl_status = sl_strlcpy(tmpcat, iname,                 PATH_MAX+1);
      if (! SL_ISERROR(sl_status) && tmpcat[1] != '\0')
	sl_status = sl_strlcat(tmpcat, "/",                 PATH_MAX+1);
      if (! SL_ISERROR(sl_status))
	sl_status = sl_strlcat(tmpcat, thisEntry->d_name,   PATH_MAX+1);
      if (SL_ISERROR(sl_status))
	{
	  tmp = sh_util_safe_name(tmpcat);
	  sh_error_handle ((-1), FIL__, __LINE__, (int) sl_status, 
			   MSG_E_SUBGPATH,
			   _("path too long"),
			   tmp,
			   _("sh_suidchk_check_internal") );
	  SH_FREE(tmp);
	  continue;
	}
      sl_set_suid();
      status = lstat(tmpcat, &buf);
      sl_unset_suid();
      if (status != 0)
	{
	  status = errno;
	  tmp = sh_util_safe_name(tmpcat);
	  sh_error_handle ((-1), FIL__, __LINE__, status, MSG_E_LSTAT,
			   sh_error_message(status),
			   tmpcat );
	  SH_FREE(tmp);
	}
      else
	{
	  if (S_ISDIR(buf.st_mode))
	    {
	      /* fs is a STATIC string
	       */
	      fs = filesystem_type (tmpcat, tmpcat, &buf);
	      if (fs != NULL && 
		  0 != strcmp (_("nfs"),     fs) && 
		  0 != strcmp (_("proc"),    fs) &&
		  0 != strncmp(_("iso9660"), fs, 7) &&
		  0 != strcmp (_("vfat"),    fs) &&
		  0 != strcmp (_("msdos"),   fs)
		  )
		{
		  /* fprintf(stderr, "%s: %s\n", fs, tmpcat); */
		  sh_suidchk_check_internal(tmpcat);
		}
	    }
	  else if (S_ISREG(buf.st_mode) &&
		   (0 !=(S_ISUID & buf.st_mode)||0 !=(S_ISGID & buf.st_mode)))
	    {
	      if (1 != sh_hash_have_it (tmpcat))
		{
		  tmp = sh_util_safe_name(tmpcat);
		  sh_error_handle (ShSuidchkSeverity, FIL__, __LINE__, 
				   0, MSG_SUID_POLICY,
				   _("suid/sgid file not in database"),
				   tmp );
		  SH_FREE(tmp);
		}
	    }
	}
      SH_FREE(tmpcat);
    }
  }  while (thisEntry != NULL);

  closedir (thisDir);
  return 0;
}

/*************
 *
 * module init
 *
 *************/
int sh_suidchk_init ()
{
  if (ShSuidchkActive == S_FALSE)
    return (-1);

  return (0);
}

/*************
 *
 * module cleanup
 *
 *************/
int sh_suidchk_end ()
{
  return (0);
}


/*************
 *
 * module timer
 *
 *************/
int sh_suidchk_timer (unsigned long tcurrent)
{
  if ((time_t) (tcurrent - lastcheck) > ShSuidchkInterval)
    {
      lastcheck  = tcurrent;
      return (-1);
    }
  return 0;
}

/*************
 *
 * module check
 *
 *************/
int sh_suidchk_check ()
{
  sh_error_handle (SH_ERR_INFO, FIL__, __LINE__, EINVAL, MSG_E_SUBGEN,
		   _("Checking for SUID programs"),
		   _("suidchk_check") );
  return (sh_suidchk_check_internal ("/"));
}

/*************
 *
 * module setup
 *
 *************/

int sh_suidchk_set_severity  (char * c)
{
  char tmp[32];
  tmp[0] = '='; tmp[1] = '\0';
  sl_strlcat (tmp, c, 32);
  sh_error_set_level (tmp, &ShSuidchkSeverity);
  return 0;
}

int sh_suidchk_set_timer (char * c)
{
  long val;

  val = strtol (c, (char **)NULL, 10);
  if (val <= 0)
    sh_error_handle ((-1), FIL__, __LINE__, EINVAL, MSG_EINVALS,
                      _("suidchk timer"), c);

  val = (val <= 0 ? 7200 : val);

  ShSuidchkInterval = (time_t) val;
  return 0;
}

int sh_suidchk_set_activate (char * c)
{
  return sh_util_flagval(c, &ShSuidchkActive);
}



/* fstype.c -- determine type of filesystems that files are on
   Copyright (C) 1990, 91, 92, 93, 94 Free Software Foundation, Inc.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  */

/* Written by David MacKenzie <djm@gnu.ai.mit.edu>. */

/* Modified by R. Wichmann: 
   - replaced error()   by sh_error_handle()
   - replaced xstrdup() by sl_strdup()
   - replaced strstr()  by sl_strstr()
*/

/* modetype.h -- file type bits definitions for POSIX systems
   Requires sys/types.h sys/stat.h.
   Copyright (C) 1990 Free Software Foundation, Inc.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  */

/* POSIX.1 doesn't mention the S_IFMT bits; instead, it uses S_IStype
   test macros.  To make storing file types more convenient, define
   them; the values don't need to correspond to what the kernel uses,
   because of the way we use them. */
#ifndef S_IFMT			/* Doesn't have traditional Unix macros. */
#define S_IFBLK 1
#define S_IFCHR 2
#define S_IFDIR 4
#define S_IFREG 8
#ifdef S_ISLNK
#define S_IFLNK 16
#endif
#ifdef S_ISFIFO
#define S_IFIFO 32
#endif
#ifdef S_ISSOCK
#define S_IFSOCK 64
#endif
#endif /* !S_IFMT */

#ifdef STAT_MACROS_BROKEN
#undef S_ISBLK
#undef S_ISCHR
#undef S_ISDIR
#undef S_ISREG
#undef S_ISFIFO
#undef S_ISLNK
#undef S_ISSOCK
#undef S_ISMPB
#undef S_ISMPC
#undef S_ISNWK
#endif

/* Do the reverse: define the POSIX.1 macros for traditional Unix systems
   that don't have them.  */
#if !defined(S_ISBLK) && defined(S_IFBLK)
#define	S_ISBLK(m) (((m) & S_IFMT) == S_IFBLK)
#endif
#if !defined(S_ISCHR) && defined(S_IFCHR)
#define	S_ISCHR(m) (((m) & S_IFMT) == S_IFCHR)
#endif
#if !defined(S_ISDIR) && defined(S_IFDIR)
#define	S_ISDIR(m) (((m) & S_IFMT) == S_IFDIR)
#endif
#if !defined(S_ISREG) && defined(S_IFREG)
#define	S_ISREG(m) (((m) & S_IFMT) == S_IFREG)
#endif
#if !defined(S_ISFIFO) && defined(S_IFIFO)
#define	S_ISFIFO(m) (((m) & S_IFMT) == S_IFIFO)
#endif
#if !defined(S_ISLNK) && defined(S_IFLNK)
#define	S_ISLNK(m) (((m) & S_IFMT) == S_IFLNK)
#endif
#if !defined(S_ISSOCK) && defined(S_IFSOCK)
#define	S_ISSOCK(m) (((m) & S_IFMT) == S_IFSOCK)
#endif
#if !defined(S_ISMPB) && defined(S_IFMPB) /* V7 */
#define S_ISMPB(m) (((m) & S_IFMT) == S_IFMPB)
#define S_ISMPC(m) (((m) & S_IFMT) == S_IFMPC)
#endif
#if !defined(S_ISNWK) && defined(S_IFNWK) /* HP/UX */
#define S_ISNWK(m) (((m) & S_IFMT) == S_IFNWK)
#endif


static char * sl_strdup (char * str) 
{
  char * p = NULL;
  int    len;

  if (str != NULL)
    {
      len = sl_strlen(str);
      p   = SH_ALLOC(len + 1);
      sl_strlcpy (p, str, len+1);
    }
  return p;
}



static char *filesystem_type_uncached (char *path, char *relpath, 
				       struct stat *statp);

#ifdef FSTYPE_MNTENT		/* 4.3BSD etc.  */
static int xatoi (char *cp);
#endif

#ifdef FSTYPE_MNTENT		/* 4.3BSD, SunOS, HP-UX, Dynix, Irix.  */
#include <mntent.h>
#if !defined(MOUNTED)
# if defined(MNT_MNTTAB)	/* HP-UX.  */
#  define MOUNTED MNT_MNTTAB
# endif
# if defined(MNTTABNAME)	/* Dynix.  */
#  define MOUNTED MNTTABNAME
# endif
#endif
#endif

#ifdef FSTYPE_GETMNT		/* Ultrix.  */
#include <sys/param.h>
#include <sys/mount.h>
#include <sys/fs_types.h>
#endif

#ifdef FSTYPE_USG_STATFS	/* SVR3.  */
#include <sys/statfs.h>
#include <sys/fstyp.h>
#endif

#ifdef FSTYPE_STATVFS		/* SVR4.  */
#include <sys/statvfs.h>
#include <sys/fstyp.h>
#endif

#ifdef FSTYPE_STATFS		/* 4.4BSD.  */
#include <sys/param.h>		/* NetBSD needs this.  */
#include <sys/mount.h>

#ifndef MFSNAMELEN		/* NetBSD defines this.  */
static char *
fstype_to_string (t)
     short t;
{
#ifdef INITMOUNTNAMES		/* Defined in 4.4BSD, not in NET/2.  */
  static char *mn[] = INITMOUNTNAMES;
  if (t >= 0 && t <= MOUNT_MAXTYPE)
    return mn[t];
  else
    return "?";
#else /* !INITMOUNTNAMES */
  switch (t)
    {
    case MOUNT_UFS:
      return _("ufs");
    case MOUNT_NFS:
      return _("nfs");
#ifdef MOUNT_PC
    case MOUNT_PC:
      return _("pc");
#endif
#ifdef MOUNT_MFS
    case MOUNT_MFS:
      return _("mfs");
#endif
#ifdef MOUNT_LO
    case MOUNT_LO:
      return _("lofs");
#endif
#ifdef MOUNT_TFS
    case MOUNT_TFS:
      return _("tfs");
#endif
#ifdef MOUNT_TMP
    case MOUNT_TMP:
      return _("tmp");
#endif
#ifdef MOUNT_MSDOS
    case MOUNT_MSDOS:
      return _("msdos");
#endif
#ifdef MOUNT_ISO9660
    case MOUNT_ISO9660:
      return _("iso9660fs");
#endif
    default:
      return "?";
    }
#endif /* !INITMOUNTNAMES */
}
#endif /* !MFSNAMELEN */
#endif /* FSTYPE_STATFS */

#ifdef FSTYPE_AIX_STATFS	/* AIX.  */
#include <sys/vmount.h>
#include <sys/statfs.h>

#define FSTYPE_STATFS		/* Otherwise like 4.4BSD.  */
#define f_type f_vfstype

static char *
fstype_to_string (t)
     short t;
{
  switch (t)
    {
    case MNT_AIX:
#if 0				/* NFS filesystems are actually MNT_AIX. */
      return "aix";
#endif
    case MNT_NFS:
      return "nfs";
    case MNT_JFS:
      return "jfs";
    case MNT_CDROM:
      return "cdrom";
    default:
      return "?";
    }
}
#endif /* FSTYPE_AIX_STATFS */

#ifdef AFS
#include <netinet/in.h>
#include <afs/venus.h>
#if __STDC__
/* On SunOS 4, afs/vice.h defines this to rely on a pre-ANSI cpp.  */
#undef _VICEIOCTL
#define _VICEIOCTL(id)  ((unsigned int ) _IOW('V', id, struct ViceIoctl))
#endif
#ifndef _IOW
/* AFS on Solaris 2.3 doesn't get this definition.  */
#include <sys/ioccom.h>
#endif

static int
in_afs (path)
     char *path;
{
  static char space[2048];
  struct ViceIoctl vi;

  vi.in_size = 0;
  vi.out_size = sizeof (space);
  vi.out = space;

  if (pioctl (path, VIOC_FILE_CELL_NAME, &vi, 1)
      && (errno == EINVAL || errno == ENOENT))
	return 0;
  return 1;
}
#endif /* AFS */

/* Nonzero if the current filesystem's type is known.  */
static int fstype_known = 0;

/* Return a static string naming the type of filesystem that the file PATH,
   described by STATP, is on.
   RELPATH is the file name relative to the current directory.
   Return "unknown" if its filesystem type is unknown.  */

static char *
filesystem_type (char * path, char * relpath, struct stat * statp)
{
  static char *current_fstype = NULL;
  static dev_t current_dev;

  if (current_fstype != NULL)
    {
      if (fstype_known && statp->st_dev == current_dev)
	return current_fstype;	/* Cached value.  */
      SH_FREE (current_fstype);
    }
  current_dev = statp->st_dev;
  current_fstype = filesystem_type_uncached (path, relpath, statp);
  return current_fstype;
}

/* Return a newly allocated string naming the type of filesystem that the
   file PATH, described by STATP, is on.
   RELPATH is the file name relative to the current directory.
   Return "unknown" if its filesystem type is unknown.  */

static char *
filesystem_type_uncached (path, relpath, statp)
     char *path;
     char *relpath;
     struct stat *statp;
{
  char *type = NULL;


#ifdef FSTYPE_MNTENT		/* 4.3BSD, SunOS, HP-UX, Dynix, Irix.  */
  char *table = MOUNTED;
  FILE *mfp;
  struct mntent *mnt;

  if (path == NULL || relpath == NULL)
    return NULL;

  mfp = setmntent (table, "r");
  if (mfp == NULL)
    {
      sh_error_handle ((-1), FIL__, __LINE__, 0, MSG_E_SUBGEN,
		       _("setmntent() failed"),
		       _("filesystem_type_uncached") );
      return NULL;
    }

  /* Find the entry with the same device number as STATP, and return
     that entry's fstype. */
  while (type == NULL && (mnt = getmntent (mfp)))
    {
      char *devopt;
      dev_t dev;
      struct stat disk_stats;

#ifdef MNTTYPE_IGNORE
      if (!strcmp (mnt->mnt_type, MNTTYPE_IGNORE))
	continue;
#endif

      /* Newer systems like SunOS 4.1 keep the dev number in the mtab,
	 in the options string.	 For older systems, we need to stat the
	 directory that the filesystem is mounted on to get it.

	 Unfortunately, the HPUX 9.x mnttab entries created by automountq
	 contain a dev= option but the option value does not match the
	 st_dev value of the file (maybe the lower 16 bits match?).  */

#if !defined(hpux) && !defined(__hpux__)
      devopt = sl_strstr (mnt->mnt_opts, "dev=");
      if (devopt)
	{
	  if (devopt[4] == '0' && (devopt[5] == 'x' || devopt[5] == 'X'))
	    dev = xatoi (devopt + 6);
	  else
	    dev = xatoi (devopt + 4);
	}
      else
#endif /* not hpux */
	{
	  if (stat (mnt->mnt_dir, &disk_stats) == -1)
	    {
	      sh_error_handle ((-1), FIL__, __LINE__, 0, MSG_E_SUBGEN,
			       _("stat() failed"),
			       _("filesystem_type_uncached") );
	      return NULL;
	    }
	  dev = disk_stats.st_dev;
	}

      if (dev == statp->st_dev)
	type = mnt->mnt_type;
    }

  if (endmntent (mfp) == 0)
    {
      sh_error_handle ((-1), FIL__, __LINE__, 0, MSG_E_SUBGEN,
		       _("endmntent() failed"),
		       _("filesystem_type_uncached") );
    }
#endif

#ifdef FSTYPE_GETMNT		/* Ultrix.  */
  int offset = 0;
  struct fs_data fsd;

  if (path == NULL || relpath == NULL)
    return NULL;

  while (type == NULL
	 && getmnt (&offset, &fsd, sizeof (fsd), NOSTAT_MANY, 0) > 0)
    {
      if (fsd.fd_req.dev == statp->st_dev)
	type = gt_names[fsd.fd_req.fstype];
    }
#endif

#ifdef FSTYPE_USG_STATFS	/* SVR3.  */
  struct statfs fss;
  char typebuf[FSTYPSZ];

  if (path == NULL || relpath == NULL)
    return NULL;

  if (statfs (relpath, &fss, sizeof (struct statfs), 0) == -1)
    {
      /* Don't die if a file was just removed. */
      if (errno != ENOENT)
	{
	  sh_error_handle ((-1), FIL__, __LINE__, errno, MSG_E_SUBGEN,
			   _("statfs() failed"),
			   _("filesystem_type_uncached") );
	  return NULL;
	}
    }
  else if (!sysfs (GETFSTYP, fss.f_fstyp, typebuf))
    type = typebuf;
#endif

#ifdef FSTYPE_STATVFS		/* SVR4.  */
  struct statvfs fss;

  if (path == NULL || relpath == NULL)
    return NULL;

  if (statvfs (relpath, &fss) == -1)
    {
      /* Don't die if a file was just removed. */
      if (errno != ENOENT)
	{
	  sh_error_handle ((-1), FIL__, __LINE__, errno, MSG_E_SUBGEN,
			   _("statvfs() failed"),
			   _("filesystem_type_uncached") );
	  return NULL;
	}
    }
  else
    type = fss.f_basetype;
#endif

#ifdef FSTYPE_STATFS		/* 4.4BSD.  */
  struct statfs fss;
  char *p;

  if (path == NULL || relpath == NULL)
    return NULL;

  if (S_ISLNK (statp->st_mode))
    p = dirname (relpath);
  else
    p = relpath;

  if (statfs (p, &fss) == -1)
    {
      /* Don't die if symlink to nonexisting file, or a file that was
	 just removed. */
      if (errno != ENOENT)
	{
	  sh_error_handle ((-1), FIL__, __LINE__, errno, MSG_E_SUBGEN,
			   _("statfs() failed"),
			   _("filesystem_type_uncached") );
	  return NULL;
	}
    }
  else
    {
#ifdef MFSNAMELEN		/* NetBSD.  */
      type = sl_strdup (fss.f_fstypename);
#else
      type = fstype_to_string (fss.f_type);
#endif
    }
  if (p != relpath)
    SH_FREE (p);
#endif

#ifdef AFS
  if ((!type || !strcmp (type, "xx")) && in_afs (relpath))
    type = "afs";
#endif

  /* An unknown value can be caused by an ENOENT error condition.
     Don't cache those values.  */
  fstype_known = (type != NULL);

  return sl_strdup (type ? type : "unknown");
}

#ifdef FSTYPE_MNTENT		/* 4.3BSD etc.  */
/* Return the value of the hexadecimal number represented by CP.
   No prefix (like '0x') or suffix (like 'h') is expected to be
   part of CP. */

static int
xatoi (cp)
     char *cp;
{
  int val;
  
  val = 0;
  while (*cp)
    {
      if (*cp >= 'a' && *cp <= 'f')
	val = val * 16 + *cp - 'a' + 10;
      else if (*cp >= 'A' && *cp <= 'F')
	val = val * 16 + *cp - 'A' + 10;
      else if (*cp >= '0' && *cp <= '9')
	val = val * 16 + *cp - '0';
      else
	break;
      cp++;
    }
  return val;
}
#endif



#endif


/* #ifdef SH_USE_UTMP */
#endif



