/* SAMHAIN file system integrity testing                                   */
/* Copyright (C) 1999, 2000 Rainer Wichmann                                */
/*                                                                         */
/*  This program is free software; you can redistribute it                 */
/*  and/or modify                                                          */
/*  it under the terms of the GNU General Public License as                */
/*  published by                                                           */
/*  the Free Software Foundation; either version 2 of the License, or      */
/*  (at your option) any later version.                                    */
/*                                                                         */
/*  This program is distributed in the hope that it will be useful,        */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of         */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the          */
/*  GNU General Public License for more details.                           */
/*                                                                         */
/*  You should have received a copy of the GNU General Public License      */
/*  along with this program; if not, write to the Free Software            */
/*  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.              */

#include "config_xor.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>


#include "samhain.h"

#ifdef USE_SRP_PROTOCOL

#if (defined (SH_WITH_CLIENT) || defined (SH_WITH_SERVER))

#include "sh_tiger.h"
#include "sh_mem.h"
#include "sh_utils.h"
#include "bignum.h"
#include "sh_srp.h"

#undef  FIL__
#define FIL__  _("sh_srp.c")

typedef struct sh_srp_struc {
  char   x[KEY_LEN+1];
  bignum a;
  bignum p;
  bignum g;
} sh_srp_t;

static sh_srp_t sh_srp;

void sh_srp_x (char * salt, char * password)
{

  char           *combi;
  long            len;
  register int i;
  unsigned char * dez = NULL;

  SL_ENTER(_("sh_srp_x"));

  /* patch by Andreas Piesk
   */
  if (password == NULL)
    dez = (unsigned char *) &(skey->pw[0]);
  else 
    dez = (unsigned char *) password;

  for (i = 0; i < PW_LEN; ++i)
    {
      skey->vernam[i] = (*dez); 
      ++dez;
    }
  skey->vernam[PW_LEN] = '\0';

  sl_strlcpy (skey->vernam,
	      sh_tiger_hash(skey->vernam, TIGER_DATA, PW_LEN), KEY_LEN);
  skey->vernam[KEY_LEN] = '\0';

  len = sl_strlen(salt) + sl_strlen(skey->vernam) + 1;

  /* H(s,P)
   */
  combi = SH_ALLOC(len);
  sl_strlcpy (combi, salt, len);
  sl_strlcat (combi, skey->vernam, len);
  sl_strlcpy (sh_srp.x, 
	      sh_tiger_hash(combi, TIGER_DATA, sl_strlen(combi)),
	      KEY_LEN+1);
  SH_FREE (combi);

  SL_RET0(_("sh_srp_x"));
}

char * sh_srp_M (char * x1, char * x2, char * x3)
{
  char           *combi;
  long            len;
  static char     hash[KEY_LEN+1];
  
  SL_ENTER(_("sh_srp_M"));

  ASSERT_RET((x1 != NULL && x2 != NULL && x3 !=NULL),
	     _("x1 != NULL && x2 != NULL && x3 !=NULL"), NULL);

  len = sl_strlen(x1) + sl_strlen(x2) + sl_strlen(x3) + 1;
  
  /* H(x1,x2,x3)
   */
  combi = SH_ALLOC(len);
  sl_strlcpy (combi, x1, len);
  sl_strlcat (combi, x2, len);
  sl_strlcat (combi, x3, len);
  sl_strlcpy (hash, 
	      sh_tiger_hash(combi, TIGER_DATA, (len-1)),
	      KEY_LEN+1);
  SH_FREE (combi);
  
  SL_RETURN(hash, _("sh_srp_M"));
}


void sh_srp_exit()
{
  SL_ENTER(_("sh_srp_exit"));
  big_destroy(&sh_srp.g);	     
  big_destroy(&sh_srp.p);
  big_destroy(&sh_srp.a);

  big_release_pkg();

  big_errno = BIG_OK;
  SL_RET0(_("sh_srp_exit"));
}


int sh_srp_init()
{
  bigerr_t res;
  char     modulus[80*4];

  SL_ENTER(_("sh_srp_init"));
  
  big_errno = BIG_OK; 

  res = big_init_pkg();
  
  if (res == BIG_OK)
    {
      res = big_create(&sh_srp.p);
      if (res == BIG_OK)
	res = big_create(&sh_srp.g);
      if (res == BIG_OK)
        res = big_create(&sh_srp.a);
      if (res == BIG_OK)
	{
	  sl_strlcpy(modulus, SRP_MODULUS_1024_1, sizeof(modulus));
	  sl_strlcat(modulus, SRP_MODULUS_1024_2, sizeof(modulus));
	  sl_strlcat(modulus, SRP_MODULUS_1024_3, sizeof(modulus));
	  sl_strlcat(modulus, SRP_MODULUS_1024_4, sizeof(modulus));
	}
      if (res == BIG_OK)
	res = big_set_string (modulus,                  16, &sh_srp.p);
      if (res == BIG_OK)
	res = big_set_string (SRP_GENERATOR_1024,       16, &sh_srp.g);
      if (res == BIG_OK)
	SL_RETURN (0, _("sh_srp_init"));
      else
	sh_srp_exit();
    }
  SL_RETURN ((-1), _("sh_srp_init"));
}


int sh_srp_make_a ()
{
  UINT32 randl[6];
  int    i;
  int    res;
  char   hash[KEY_LEN+1];

  SL_ENTER(_("sh_srp_make_a"));

  for (i = 0; i < 6; ++i)
    randl[i] = (UINT32) taus_get (&(skey->rng0[0]), 
				  &(skey->rng1[0]),
				  &(skey->rng2[0]));
  sl_strlcpy (hash, 
	      sh_tiger_hash((char *)&randl[0], TIGER_DATA, 6*sizeof(UINT32)), 
	      KEY_LEN+1);

  hash[KEY_LEN] = '\0';

  res = big_set_string (hash,       16, &sh_srp.a);
  if (res == BIG_OK)
    SL_RETURN((0), _("sh_srp_make_a"));
  else
    SL_RETURN((-1), _("sh_srp_make_a"));
}

/* return 0 if AB is NOT zero
 */
int sh_srp_check_zero (char * AB_str)
{
  bignum   AB, q, r;
  bigerr_t res;
  int      val;

  SL_ENTER(_("sh_srp_check_zero"));

  ASSERT_RET((AB_str != NULL), _("AB_str != NULL"), (-1));

  res = big_create(&AB);
  if (res == BIG_OK)
    res = big_create(&q);
  if (res == BIG_OK)
    res = big_create(&r);

  if (res == BIG_OK)
    res = big_set_string (AB_str,       16, &AB);
  if (res == BIG_OK)
    res = big_trunc(&AB, &sh_srp.p, &q, &r); /* is last one the remainder ? */
  
  if (res != BIG_OK)         val = (-1);
  else if ( big_zerop(&AB) ) val = (-1);
  else                       val =    0;

  big_destroy(&AB);	
  big_destroy(&q);	
  big_destroy(&r);	
  
  SL_RETURN((val), _("sh_srp_check_zero"));
}

#ifdef SH_WITH_CLIENT
  

char * sh_srp_A ()
{
  bignum   A;
  char    *str;
  char    *combi;
  long     len;
  bigerr_t res;

  SL_ENTER(_("sh_srp_A"));

  res = big_create(&A);
  
  if (res == BIG_OK)
    res = big_exptmod (&sh_srp.g, &sh_srp.a, &sh_srp.p, &A);
  
  if (res == BIG_OK)
    str = big_string (&A, 16);
  else
    str = NULL;
  
  if (str != NULL)
    {
      len = sl_strlen(str) + 1;
      combi = SH_ALLOC(len);
      sl_strlcpy (combi, str, len);
    }
  else
    combi = NULL;
  
  big_destroy(&A);	     
  SL_RETURN(combi, _("sh_srp_A"));
}

/* #ifdef SH_WITH_CLIENT */
#endif  
  
#ifdef SH_WITH_SERVER

char * sh_srp_B (char * verifier)
{
  bignum   B, v, t, dummy;
  char    *str;
  char    *combi;
  long     len;
  bigerr_t res;

  SL_ENTER(_("sh_srp_B"));

  ASSERT_RET((verifier != NULL), _("verifier != NULL"), (NULL));

  res = big_create(&dummy);

  if (res == BIG_OK)
    res = big_create(&t);
  if (res == BIG_OK)
    res = big_create(&v);
  if (res == BIG_OK)
    res = big_create(&B);

  if (res == BIG_OK)
    res = big_exptmod (&sh_srp.g, &sh_srp.a, &sh_srp.p, &t);
  
  if (res == BIG_OK)
    big_set_string (verifier,       16, &v);

  if (res == BIG_OK)
    res = big_add (&t, &v, &dummy);

  if (res == BIG_OK)
    {
      if ( big_greaterp(&dummy, &sh_srp.p) ) 
	res = big_sub(&dummy, &sh_srp.p, &B);
      else                                   
	res = big_set_big(&dummy, &B);
    }

  if (res == BIG_OK)
    str = big_string (&B, 16);
  else
    str = NULL;
  
  if (str != NULL)
    {
      len = sl_strlen(str) + 1;
      combi = SH_ALLOC(len);
      sl_strlcpy (combi, str, len);
    }
  else
    combi = NULL;
  
  big_destroy(&B);	
  big_destroy(&v);	
  big_destroy(&t);	
  big_destroy(&dummy);	
  
  SL_RETURN(combi, _("sh_srp_B"));
}
/* #ifdef SH_WITH_SERVER */
#endif  
  
  
#ifdef SH_WITH_CLIENT
  
char * sh_srp_S_c (char * u_str, char * B_str)
{
  bignum   u, B, x, t, base, z1, z2;
  char    *str;
  char    *combi;
  long     len;
  bigerr_t res;

  SL_ENTER(_("sh_srp_S_c"));

  ASSERT_RET((u_str != NULL && B_str != NULL),
	     _("u_str != NULL && B_str != NULL"), (NULL));

  big_errno = BIG_OK;

  res = big_create(&z2);
  if (res == BIG_OK)
   res = big_create(&z1);
  if (res == BIG_OK)
   res = big_create(&base);
  if (res == BIG_OK)
   res = big_create(&t);
  if (res == BIG_OK)
   res = big_create(&x);
  if (res == BIG_OK)
   res = big_create(&B);
  if (res == BIG_OK)
   res = big_create(&u);
  
  if (res == BIG_OK)
   res = big_set_string (B_str,          16, &B);
  if (res == BIG_OK)
   res = big_set_string (sh_srp.x,       16, &x);
  if (res == BIG_OK)
   res = big_set_string (u_str,          16, &u);
  
  /* the base  (B - g^x)
   */
  if (res == BIG_OK)
    res = big_exptmod (&sh_srp.g, &x, &sh_srp.p, &t);

  if (res == BIG_OK)
    {
      if ( big_greaterp(&B, &t) ) 
	{
	  res = big_sub(&B, &t, &base);
	}
      else 
	{
	  res = big_add(&B, &sh_srp.p, &z2);
	  if (res == BIG_OK)
	    res = big_sub(&z2, &t, &base);
	}
    }

  /* the exponent (a + ux)
   */
  if (res == BIG_OK)
    res = big_mul (&u, &x, &t);
  if (res == BIG_OK)
    res = big_trunc(&t, &sh_srp.p, &z1, &z2); /* is last one the remainder ? */
  if (res == BIG_OK)
    res = big_add(&sh_srp.a, &z2, &z1);
  if (res == BIG_OK)
    {
      if ( big_greaterp(&z1, &sh_srp.p) ) 
	res = big_sub(&z1, &sh_srp.p, &z2);
      else 
	res = big_set_big(&z1, &z2);
    }

  if (res == BIG_OK)
    res = big_exptmod (&base, &z2, &sh_srp.p, &t);

  if (res == BIG_OK)
    str = big_string (&t, 16);
  else
    str = NULL;

  if (str != NULL)
    {
      len = sl_strlen(str) + 1;
      combi = SH_ALLOC(len);
      sl_strlcpy (combi, str, len);
    }
  else
    combi = NULL;

  big_destroy(&z1);	     
  big_destroy(&z2);	     
  big_destroy(&base);	     
  big_destroy(&t);	
  big_destroy(&x);	
  big_destroy(&B);	
  big_destroy(&u);	
  
  SL_RETURN(combi, _("sh_srp_S_c"));
}
  
/* #ifdef SH_WITH_CLIENT */
#endif  
  
#ifdef SH_WITH_SERVER

  
char * sh_srp_S_s (char * u_str, char * A_str, char * v_str)
{
  bignum   u, A, v, t, base, z1, z2;
  char    *str;
  char    *combi;
  long     len;
  bigerr_t res;

  SL_ENTER(_("sh_srp_S_s"));

  ASSERT_RET((u_str != NULL && A_str != NULL && v_str != NULL),
	     _("u_str != NULL && A_str != NULL && v_str != NULL"),
	     (NULL));

  big_errno = BIG_OK;

  res = big_create(&z2);
  if (res == BIG_OK)
    res = big_create(&z1);
  if (res == BIG_OK)
    res = big_create(&base);
  if (res == BIG_OK)
    res = big_create(&t);
  if (res == BIG_OK)
    res = big_create(&v);
  if (res == BIG_OK)
    res = big_create(&A);
  if (res == BIG_OK)
    res = big_create(&u);
  
  if (res == BIG_OK)
    res = big_set_string (A_str,          16, &A);
  if (res == BIG_OK)
    res = big_set_string (v_str,          16, &v);
  if (res == BIG_OK)
    res = big_set_string (u_str,          16, &u);
  
  /* the base  (Av^u)
   */
  if (res == BIG_OK)
    res = big_exptmod (&v, &u, &sh_srp.p, &t);
  if (res == BIG_OK)
    res = big_mul (&A, &t, &z1);
  if (res == BIG_OK)
    res = big_trunc(&z1, &sh_srp.p, &z2, &base); /* is last the remainder ? */

  if (res == BIG_OK)
    res = big_exptmod (&base, &sh_srp.a, &sh_srp.p, &t);

  if (res == BIG_OK)
    str = big_string (&t, 16);
  else
    str = NULL;
  
  if (str != NULL)
    {
      len = sl_strlen(str) + 1;
      combi = SH_ALLOC(len);
      sl_strlcpy (combi, str, len);
    }
  else
    combi = NULL;
  
  big_destroy(&z1);	     
  big_destroy(&z2);	     
  big_destroy(&base);	     
  big_destroy(&t);	
  big_destroy(&v);	
  big_destroy(&A);	
  big_destroy(&u);	
  
  SL_RETURN(combi, _("sh_srp_S_s"));
}

/* #ifdef SH_WITH_SERVER */
#endif  


char * sh_srp_verifier (void)
{
  bignum   x, v;
  char    *combi;
  char    *str;
  long     len;
  bigerr_t res;
  
  SL_ENTER(_("sh_srp_verifier"));
  
  res = big_create(&x);
  if (res == BIG_OK)
    res = big_create(&v);
  
  if (res == BIG_OK)
    res = big_set_string (sh_srp.x,               16, &x);
  
  if (res == BIG_OK)
    res = big_exptmod (&sh_srp.g, &x, &sh_srp.p, &v);
  
  if (res == BIG_OK)
    str = big_string (&v, 16);
  else
    str = NULL;
  
  if (str != NULL)
    {
      len = sl_strlen(str) + 1;
      combi = SH_ALLOC(len);
      sl_strlcpy (combi, str, len);
    }
  else
    combi = NULL;
  
  big_destroy(&x);	     
  big_destroy(&v);	     
  
  SL_RETURN(combi, _("sh_srp_verifier"));
}
  

/* #if (defined (SH_WITH_CLIENT) || defined (SH_WITH_SERVER)) */

#endif

/* #ifdef USE_SRP_PROTOCOL */

#endif




