/* SAMHAIN file system integrity testing                                   */
/* Copyright (C) 1999, 2000 Rainer Wichmann                                */
/*                                                                         */
/*  This program is free software; you can redistribute it                 */
/*  and/or modify                                                          */
/*  it under the terms of the GNU General Public License as                */
/*  published by                                                           */
/*  the Free Software Foundation; either version 2 of the License, or      */
/*  (at your option) any later version.                                    */
/*                                                                         */
/*  This program is distributed in the hope that it will be useful,        */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of         */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the          */
/*  GNU General Public License for more details.                           */
/*                                                                         */
/*  You should have received a copy of the GNU General Public License      */
/*  along with this program; if not, write to the Free Software            */
/*  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.              */

#include "config_xor.h"


#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>


#include "samhain.h"
#include "sh_error.h"
#include "sh_unix.h"
#include "sh_utils.h"
#include "sh_files.h"
#include "sh_mail.h"
#include "sh_calls.h"
#include "sh_tiger.h"
#include "sh_forward.h"
#include "sh_modules.h"
#include "sh_gpg.h"
#ifdef WITH_EXTERNAL
#include "sh_extern.h"
#endif

#undef  FIL__
#define FIL__  _("sh_readconf.c")

typedef enum {
  SH_SECTION_NONE,
  SH_SECTION_LOG,
  SH_SECTION_MISC,
  SH_SECTION_ATTRIBUTES,
  SH_SECTION_READONLY,
  SH_SECTION_LOGFILES,
  SH_SECTION_LOGGROW,
  SH_SECTION_NOIGNORE,
  SH_SECTION_ALLIGNORE,
#if defined (SH_WITH_MAIL) 
  SH_SECTION_MAIL,
#endif
#if defined (SH_WITH_CLIENT) 
  SH_SECTION_CLT,
#endif
#ifdef WITH_EXTERNAL
  SH_SECTION_EXTERNAL,
#endif
#if defined (SH_WITH_CLIENT) || defined (SH_STANDALONE) 
  SH_SECTION_OTHER,
#endif
#ifdef SH_WITH_SERVER
  SH_SECTION_CLIENTS,
  SH_SECTION_SRV,
#endif
  SH_SECTION_THRESHOLD
} ShSectionType;

typedef struct str_ListSections {
  char * name;
  int    type;
} sh_str_ListSections;

struct str_ListSections tab_ListSections[] = {
  { N_("[Log]"),              SH_SECTION_LOG},
  { N_("[Misc]"),             SH_SECTION_MISC},
  { N_("[Attributes]"),       SH_SECTION_ATTRIBUTES},
  { N_("[ReadOnly]"),         SH_SECTION_READONLY},
  { N_("[LogFiles]"),         SH_SECTION_LOGFILES},
  { N_("[GrowingLogFiles]"),  SH_SECTION_LOGGROW},
  { N_("[IgnoreAll]"),        SH_SECTION_ALLIGNORE},
  { N_("[IgnoreNone]"),       SH_SECTION_NOIGNORE},
#ifdef WITH_EXTERNAL
  { N_("[External]"),         SH_SECTION_EXTERNAL}, 
#endif
  { N_("[EventSeverity]"),    SH_SECTION_THRESHOLD},
#ifdef SH_WITH_SERVER
  { N_("[Clients]"),          SH_SECTION_CLIENTS},
  { N_("[Server]"),           SH_SECTION_SRV},
#endif
#if defined (SH_WITH_CLIENT) 
  { N_("[Client]"),           SH_SECTION_CLT},
#endif
#if defined (SH_WITH_MAIL) 
  { N_("[Mail]"),             SH_SECTION_MAIL},
#endif
  { NULL,                     SH_SECTION_NONE}
};

   
static void sh_readconfig_line (char * line);

static ShSectionType read_mode = SH_SECTION_NONE;

static int conf_line = 0;

/* --- Read the configuration file. ---
 */
int sh_readconf_read (void)
{
#if defined (SH_WITH_CLIENT) || defined (SH_STANDALONE) 
  /* This is for modules. 
   */
  int    modnum;
#endif

  int i;

  SL_TICKET    fd    = -1;
#if defined(SH_STEALTH) && !defined(SH_STEALTH_MICRO)
  SL_TICKET    fdTmp = -1;
  SL_TICKET open_tmp (void);
#endif
  char * tmp;
  char * lptr;

  char   line_in[512+2];
  char * line;
  int    line_int;

  char   myident[3*SH_MINIBUF+3];

  /* This is for nested conditionals.
   */
  int    some_other_host[16]   = { 0 };
  int    some_other_system[16] = { 0 };
  int    seen_host   = 0;
  int    seen_system = 0;
  int    host_int    = 0;
  int    sys_int     = 0;

  int    invert = 0;
  int    length = sl_strlen(sh.host.name);

  int    local_file = 1;
  char   local_flag = 'R';

#if defined(WITH_GPG) || defined(WITH_PGP)
  int    signed_content = S_FALSE;
#endif

  uid_t  euid;

  SL_ENTER(_("sh_readconf_read"));

  /* --- Open config file, exit on failure. ---
   */
#if defined(SH_WITH_CLIENT)
  if (0 == sl_strcmp(file_path('C', 'R'), _("REQ_FROM_SERVER")))
    {
      sh_error_handle ((-1), FIL__, __LINE__, 0, MSG_D_START);

      fd = sh_forward_req_file(_("CONF"));

      if (!SL_ISERROR(fd))
	local_file = 0;
      else if (sh.flag.checkSum != SH_CHECK_INIT)
	aud_exit (FIL__, __LINE__, EXIT_FAILURE);
      else
	{
	  sh_error_handle ((-1), FIL__, __LINE__, 0, MSG_D_FAIL);
	  local_file = 1;
	  local_flag = 'I';
	}
    }
#endif

  /* Use a local configuration file.
   */
  if (local_file == 1)
    {
      if (0 != tf_trust_check (file_path('C', local_flag), SL_YESPRIV)
	  ||
	  SL_ISERROR(fd = sl_open_read(file_path('C',local_flag),SL_YESPRIV)))
	{
	  sl_get_euid(&euid);
	  sh_error_handle ((-1), FIL__, __LINE__, EACCES, MSG_NOACCESS, 
			   (long) euid, file_path('C', local_flag));
	  aud_exit (FIL__, __LINE__, EXIT_FAILURE);
	}
    }

  /* Compute the checksum of the open file.
   */
  tiger_fd = fd;
  sl_strlcpy(sh.conf.hash, 
	     sh_tiger_hash(file_path('C',local_flag),TIGER_FD, 0),
	     KEY_LEN+1);
  sl_rewind (fd);

#if defined(SH_STEALTH) && !defined(SH_STEALTH_MICRO)
    /* extract the data and copy to temporary file
     */
  fdTmp = open_tmp();
  while ( sh_unix_getline_stealth (fd, line_in, 512) > 0) {
    if (line_in[0] == '\n')
      {
	sl_write(fdTmp, line_in, 1);
      }
    else
      {
	sl_write_line(fdTmp, line_in, sl_strlen(line_in));
      }
  }
  sl_close(fd);
  fd = fdTmp;
  sl_rewind (fd);
#endif


  /* The system type, release, and machine.
   */
  sprintf(myident, _("%s:%s:%s"),                  /* known to fit  */
	  sh.host.system, sh.host.release, sh.host.machine);


  /* ---  Start reading lines.  ---
   */
  while ( sh_unix_getline (fd, line_in, 512) > 0) {

    ++conf_line;

    line = &(line_in[0]);

    /* fprintf(stderr, "<%s>\n", line); */

    /* Sun May 27 18:40:05 CEST 2001
     */
#if defined(WITH_GPG) || defined(WITH_PGP)
    if (signed_content == S_FALSE)
      { 
	if (0 == sl_strcmp(line, _("-----BEGIN PGP SIGNED MESSAGE-----")))
	  signed_content = S_TRUE;
	else 
	  continue;
      }
    else if (signed_content == S_TRUE)
      { 
	if (0 == sl_strcmp(line, _("-----END PGP SIGNATURE-----")))
	  break;
	else if (0 == sl_strcmp(line, _("-----BEGIN PGP SIGNED MESSAGE-----")))
	  {
	    sh_error_handle((-1), FIL__, __LINE__, 0, MSG_E_SUBGEN,
			    _("second signed message in file"),
			    _("sh_readconf_read"));
	    sh_error_handle ((-1), FIL__, __LINE__, 0, MSG_EXIT_ABORT1,
			     sh.prg_name);
	    aud_exit (FIL__, __LINE__,EXIT_FAILURE);
	  }
      }
#endif

    /* Skip leading white space.
     */
    while (*line)
      {
	line_int = *line;
	if (!isspace(line_int))
	  break;
	++line;
      }

    /* Skip header etc. 
     */
    if (line[0] == '#' || line[0] == '\0' || line[0] == ';' || line[0] == '/')
      continue; 
  
    /* Clip off trailing white space.                 
     */
    tmp = line + sl_strlen( line ); --tmp;
    while( isspace((int) *tmp ) && tmp >= line ) *tmp-- = '\0';


    /* -------  starts a section  ------------  */
    
    if (line[0] == '['                       && 
	some_other_host[seen_host] == 0      &&
	some_other_system[seen_system] == 0) 
      { 
	read_mode = SH_SECTION_NONE;

	if (sl_strncmp (line,  _("[EOF]"), 
			  5) == 0)
	  {
	    goto nopel;
	  }

	i = 0;

	while (tab_ListSections[i].name != 0)
	  {
	    if (sl_strncmp (line, _(tab_ListSections[i].name), 
			    sl_strlen(tab_ListSections[i].name)) == 0)
	      { 
		read_mode = tab_ListSections[i].type;
		break;
	      }
	    ++i;
	  }

#if defined (SH_WITH_CLIENT) || defined (SH_STANDALONE) 
	if (read_mode == SH_SECTION_NONE)
	  {
	    for (modnum = 0; modList[modnum].name != NULL; ++modnum) 
	      {
		if (0 == sl_strncmp (line,  _(modList[modnum].conf_section),
				     sl_strlen(modList[modnum].conf_section)) )
		  read_mode = SH_SECTION_OTHER;
	      }
	  }
#endif
	if (read_mode == SH_SECTION_NONE)
	  sh_error_handle (SH_ERR_ALL, FIL__, __LINE__, 0, MSG_EINVALL,
			   _("config file"), (long) conf_line);
      } 

    /* ---  an @host directive -------------- */

    else if (line[0] == '@' || (line[0] == '!' && line[1] == '@')) 
      {
	if (line[0] == '!')
	  {
	    lptr   = &line[2];
	    invert = 1;
	  }
	else
	  {
	    lptr   = &line[1];
	    invert = 0;
	  }

	if (sl_strncmp (lptr, _("end"), 3) == 0)
	  {
	    if (0 == seen_host)
	      sh_error_handle ((-1), FIL__, __LINE__, 0, MSG_EINVALD,
			       _("config file"), (long) conf_line);

	    else if (host_int == 0)
	      {
		/* end  of an @host directive 
		 */
		some_other_host[seen_host] = 0;
		--seen_host;
		seen_host = (seen_host < 0) ? 0 : seen_host;
	      }

	    else 
	      {
		--host_int;
		host_int = (host_int < 0) ? 0 : host_int;
	      }
	  }
	else if (some_other_host[seen_host] == 0      &&
		 some_other_system[seen_system] == 0  && 
		 seen_host < 15)
	  {
	    if  (sl_strncmp (lptr,  sh.host.name, length) == 0
#ifdef HAVE_REGEX_H
		 || sh_util_regcmp (lptr, sh.host.name) == 0
#endif
		 )
	      {
		/* if match and '@',  set some_other_host = 0;
		 * if match and '!@', set some_other_host = 1;
		 */
		++seen_host;
		some_other_host[seen_host] = invert; 
	      }
	    else
	      {
		/* if no match and '@',  set some_other_host = 1;
		 * if no match and '!@', set some_other_host = 0;
		 */
		++seen_host;
		some_other_host[seen_host] = (invert == 0) ? 1 : 0;
	      }
	  }
	else
	  ++host_int;
      } 

    /* ---  an $system directive -------------- */

    else if (line[0] == '$' || (line[0] == '!' && line[1] == '$')) 
      {
	if (line[0] == '!')
	  {
	    lptr   = &line[2];
	    invert = 1;
	  }
	else
	  {
	    lptr   = &line[1];
	    invert = 0;
	  }

	if (sl_strncmp (lptr, _("end"), 3) == 0)
	  {
	    if (0 == seen_system)
	      sh_error_handle ((-1), FIL__, __LINE__, 0, MSG_EINVALD,
			       _("config file"), (long) conf_line);

	    else if (sys_int == 0)
	      {
		/* end  of an $system directive 
		 */
		some_other_system[seen_system] = 0;
		--seen_system;
		seen_system = (seen_system < 0) ? 0 : seen_system;
	      }
	    else 
	      {
		--sys_int;
		sys_int = (sys_int < 0) ? 0 : sys_int;
	      }
	  }
	else if (some_other_host[seen_host] == 0      &&
		 some_other_system[seen_system] == 0  && 
		 seen_system < 15)
	  {
	    if  (sl_strncmp (lptr,  myident, sl_strlen(myident)) == 0
#ifdef HAVE_REGEX_H
		 || sh_util_regcmp (lptr, myident) == 0
#endif
		 )
	      {
		++seen_system;
		some_other_system[seen_system] = invert;
	      }
	    else
	      {
		++seen_system;
		some_other_system[seen_system] = (invert == 0) ? 1 : 0;
	      }
	  }
	else
	  ++sys_int;
      }

    /* ------  no new section -------------- */


    else if (some_other_host[seen_host] == 0          && 
	     some_other_system[seen_system] == 0      && 
	     read_mode != SH_SECTION_NONE) 
      { 
	sh_readconfig_line (line);
      }

  }

 nopel:
	   
  if (0 != seen_host || 0 != seen_system)
    sh_error_handle ((-1), FIL__, __LINE__, 0, MSG_EINVALDD,
		     _("config file"), (long) conf_line);

#if defined(WITH_GPG) || defined(WITH_PGP)
  /* Validate signature of open file.
   */
  sl_rewind (fd);
  sh_error_only_stderr (S_FALSE);
  if (0 != sh_gpg_check_sign (fd, 0, 1))
    aud_exit (FIL__, __LINE__, EXIT_FAILURE);
#endif

  sl_close (fd);
  SL_RETURN( 0, _("sh_readconf_read"));
}

int sh_readconf_set_path (char * which, char * what)
{
  int len;
  SL_ENTER( _("sh_readconf_set_path"));

  if (which == NULL || what == NULL)
    {
      TPT((0, FIL__, __LINE__ , _("msg=<Input error>\n")));
      SL_RETURN( -1, _("sh_readconf_set_path"));
    }

  if (0 == sl_strcmp(what, _("AUTO")))
    {
      len = sl_strlen(which);
      if ( (len + sl_strlen(sh.host.name) + 2) > SH_PATHBUF)
	{
	  TPT((0, FIL__, __LINE__ , _("msg=<Path too large: %s:%s>\n"), 
	       which, sh.host.name));
	  SL_RETURN( -1, _("sh_readconf_set_path"));
	}
      else
	{
	  which[len] = ':'; which[len+1] = '\0';
	  sl_strlcat(which, sh.host.name, SH_PATHBUF);
	}
    }
  else  /* not auto */
    {
      if (sl_strlen(what) > (SH_PATHBUF-1))
	{
	  TPT((0, FIL__, __LINE__ , _("msg=<Path too large: %s>\n"), what));
	  SL_RETURN( -1, _("sh_readconf_set_path"));
	}
      else
	{
	  sl_strlcpy(which, what, SH_PATHBUF);
	}
    }
  SL_RETURN( 0, _("sh_readconf_set_path"));
}

int sh_readconf_set_database_path (char * what)
{
  return (sh_readconf_set_path(sh.data.path, what));
}

int sh_readconf_set_logfile_path (char * what)
{
  return (sh_readconf_set_path(sh.srvlog.name, what));
}

int sh_readconf_set_lockfile_path (char * what)
{
  return( sh_readconf_set_path(sh.srvlog.alt, what));
}




typedef enum {
  SET_MAILTIME,
  SET_FILETIME 
} ShTimerItem;
 
    
int sh_readconf_setTime (char * str, ShTimerItem what)
{
  unsigned long i = atoi (str);

  SL_ENTER( _("sh_readconf_setTime"));

  if (i < LONG_MAX) 
    {
      if      (what == SET_MAILTIME)
	{
	  TPT((0, FIL__, __LINE__, _("msg=<Set mail timer to %ld>\n"), i));
	  sh.mailTime.alarm_interval = i;
	}
      else if (what == SET_FILETIME)
	{
	  TPT((0, FIL__, __LINE__, _("msg=<Set filecheck timer to %ld>\n"),i));
	  sh.fileCheck.alarm_interval  = i;
	}

      SL_RETURN( 0, _("sh_readconf_setTime"));
    } 
  else 
    {
      sh_error_handle ((-1), FIL__, __LINE__, 0, MSG_EINVALL,
		     _("set timer"), (long) i);
      SL_RETURN( (-1), _("sh_readconf_setTime"));
    }
}

int sh_readconf_setMailtime (char * c)
{
  return sh_readconf_setTime (c, SET_MAILTIME);
}

int sh_readconf_setFiletime (char * c)
{
  return sh_readconf_setTime (c, SET_FILETIME);
}


typedef struct _cfg_options {
  char * optname;
  ShSectionType   section;
  ShSectionType   alt_section;
  int (*func)(char * opt);
} cfg_options;

cfg_options ext_table[] = {
#if defined(WITH_EXTERNAL)
  { N_("opencommand"),     SH_SECTION_EXTERNAL, SH_SECTION_NONE,  
    sh_ext_setcommand },
  { N_("setcommandline"),  SH_SECTION_EXTERNAL, SH_SECTION_NONE,  
    sh_ext_add_argv },
  { N_("setchecksum"),     SH_SECTION_EXTERNAL, SH_SECTION_NONE,  
    sh_ext_checksum },
  { N_("setdefault"),      SH_SECTION_EXTERNAL, SH_SECTION_NONE,  
    sh_ext_add_default },
  { N_("setenviron"),      SH_SECTION_EXTERNAL, SH_SECTION_NONE,  
    sh_ext_add_environ },
  { N_("setdeadtime"),     SH_SECTION_EXTERNAL, SH_SECTION_NONE,  
    sh_ext_deadtime },
  { N_("settype"),         SH_SECTION_EXTERNAL, SH_SECTION_NONE,  
    sh_ext_type },
  { N_("setcredentials"),  SH_SECTION_EXTERNAL, SH_SECTION_NONE,  
    sh_ext_priv },
  { N_("setfilternot"),    SH_SECTION_EXTERNAL, SH_SECTION_NONE,  
    sh_ext_add_not },
  { N_("setfilterand"),    SH_SECTION_EXTERNAL, SH_SECTION_NONE,  
    sh_ext_add_and },
  { N_("setfilteror"),     SH_SECTION_EXTERNAL, SH_SECTION_NONE,  
    sh_ext_add_or },
  { N_("externalseverity"),SH_SECTION_LOG,      SH_SECTION_NONE,  
    sh_error_set_external },
  { N_("externalclass"),   SH_SECTION_LOG,      SH_SECTION_NONE,  
    sh_error_external_mask },
#endif

#if defined (SH_WITH_CLIENT) || defined (SH_STANDALONE)
  { N_("dir"),            SH_SECTION_ATTRIBUTES, SH_SECTION_NONE, 
    sh_files_pushdir_attr },
  { N_("file"),           SH_SECTION_ATTRIBUTES, SH_SECTION_NONE, 
    sh_files_pushfile_attr },
  { N_("dir"),            SH_SECTION_READONLY,   SH_SECTION_NONE, 
    sh_files_pushdir_ro },
  { N_("file"),           SH_SECTION_READONLY,   SH_SECTION_NONE, 
    sh_files_pushfile_ro },
  { N_("dir"),            SH_SECTION_LOGFILES,   SH_SECTION_NONE, 
    sh_files_pushdir_log },
  { N_("file"),           SH_SECTION_LOGFILES,   SH_SECTION_NONE, 
    sh_files_pushfile_log },
  { N_("dir"),            SH_SECTION_LOGGROW,    SH_SECTION_NONE, 
    sh_files_pushdir_glog },
  { N_("file"),           SH_SECTION_LOGGROW,    SH_SECTION_NONE, 
    sh_files_pushfile_glog },
  { N_("dir"),            SH_SECTION_NOIGNORE,   SH_SECTION_NONE, 
    sh_files_pushdir_noig },
  { N_("file"),           SH_SECTION_NOIGNORE,   SH_SECTION_NONE, 
sh_files_pushfile_noig },
  { N_("dir"),            SH_SECTION_ALLIGNORE,  SH_SECTION_NONE, 
    sh_files_pushdir_allig },
  { N_("file"),           SH_SECTION_ALLIGNORE,  SH_SECTION_NONE, 
    sh_files_pushfile_allig },
  { N_("setrecursionlevel"),  SH_SECTION_MISC,   SH_SECTION_NONE, 
    sh_files_setrecursion },
  { N_("checksumtest"),       SH_SECTION_MISC,   SH_SECTION_NONE, 
    sh_util_setchecksum },
  { N_("reportonlyonce"),     SH_SECTION_MISC,   SH_SECTION_NONE, 
    sh_files_reportonce },
  { N_("reportfulldetail"),   SH_SECTION_MISC,   SH_SECTION_NONE, 
    sh_files_fulldetail },

  { N_("redefreadonly"),        SH_SECTION_MISC,   SH_SECTION_NONE, 
    sh_files_redef_readonly },

  { N_("redeflogfiles"),        SH_SECTION_MISC,   SH_SECTION_NONE, 
    sh_files_redef_logfiles },

  { N_("redefgrowinglogfiles"), SH_SECTION_MISC,   SH_SECTION_NONE, 
    sh_files_redef_loggrow },

  { N_("redefattributes"),      SH_SECTION_MISC,   SH_SECTION_NONE, 
    sh_files_redef_attributes },

  { N_("redefignorenone"),      SH_SECTION_MISC,   SH_SECTION_NONE, 
    sh_files_redef_noignore },

  { N_("redefignoreall"),       SH_SECTION_MISC,   SH_SECTION_NONE, 
    sh_files_redef_allignore },

#endif

#ifdef SH_WITH_SERVER
  { N_("setclienttimelimit"),  SH_SECTION_SRV,  SH_SECTION_MISC, 
    sh_forward_set_time_limit },
  { N_("useclientseverity"),   SH_SECTION_SRV,  SH_SECTION_MISC, 
  sh_forward_use_clt_sev },
  { N_("useclientclass"),      SH_SECTION_SRV,  SH_SECTION_MISC, 
  sh_forward_use_clt_class },
  { N_("severitylookup"),      SH_SECTION_SRV,  SH_SECTION_MISC, 
  sh_forward_lookup_level },
  { N_("client"),              SH_SECTION_CLIENTS,           SH_SECTION_NONE, 
    sh_forward_register_client },
#endif

#ifdef SH_WITH_CLIENT
  { N_("exportseverity"),      SH_SECTION_LOG,  SH_SECTION_NONE, 
    sh_error_setexport },
  { N_("exportclass"),         SH_SECTION_LOG,  SH_SECTION_NONE, 
    sh_error_export_mask },
  { N_("setlogserver"),        SH_SECTION_CLT,  SH_SECTION_MISC, 
    sh_forward_setlogserver },
#endif
  { N_("setfilechecktime"),  SH_SECTION_MISC,   SH_SECTION_NONE, 
    sh_readconf_setFiletime },
  { N_("setlooptime"),     SH_SECTION_MISC,  SH_SECTION_NONE, 
    sh_util_setlooptime },

#ifdef SH_WITH_MAIL
  { N_("mailseverity"),      SH_SECTION_LOG,   SH_SECTION_NONE, 
    sh_error_setseverity },
  { N_("mailclass"),         SH_SECTION_LOG,   SH_SECTION_NONE, 
    sh_error_mail_mask },
  { N_("setmailtime"),       SH_SECTION_MAIL,  SH_SECTION_MISC, 
    sh_readconf_setMailtime },
  { N_("setmailnum"),        SH_SECTION_MAIL,  SH_SECTION_MISC, 
    sh_mail_setNum },
  { N_("setmailaddress"),    SH_SECTION_MAIL,  SH_SECTION_MISC, 
    sh_mail_setaddress },
  { N_("setmailrelay"),      SH_SECTION_MAIL,  SH_SECTION_MISC, 
    sh_mail_set_relay },
  { N_("mailsingle"),        SH_SECTION_MAIL,  SH_SECTION_MISC, 
    sh_mail_setFlag },
#endif

  { N_("daemon"),            SH_SECTION_MISC,  SH_SECTION_NONE, 
    sh_unix_setdeamon },
  { N_("samhainpath"),       SH_SECTION_MISC,  SH_SECTION_NONE, 
    sh_unix_self_hash },
  { N_("trusteduser"),       SH_SECTION_MISC,  SH_SECTION_NONE, 
    tf_add_trusted_user },
  { N_("settimeserver"),     SH_SECTION_MISC,  SH_SECTION_NONE, 
    sh_unix_settimeserver },

  { N_("printseverity"),     SH_SECTION_LOG,   SH_SECTION_NONE, 
    sh_error_setprint },
  { N_("printclass"),        SH_SECTION_LOG,   SH_SECTION_NONE, 
    sh_error_print_mask },

  { N_("logseverity"),       SH_SECTION_LOG,   SH_SECTION_NONE, 
    sh_error_setlog },
  { N_("logclass"),          SH_SECTION_LOG,   SH_SECTION_NONE, 
    sh_error_log_mask },

  { N_("syslogseverity"),    SH_SECTION_LOG,   SH_SECTION_NONE, 
    sh_error_set_syslog },
  { N_("syslogclass"),       SH_SECTION_LOG,   SH_SECTION_NONE, 
    sh_error_syslog_mask },

  { N_("logcalls"),          SH_SECTION_LOG,   SH_SECTION_NONE, 
    sh_aud_set_functions },

  { N_("messageheader"),     SH_SECTION_MISC,  SH_SECTION_NONE, 
    sh_error_ehead },

  { N_("setconsole"),        SH_SECTION_MISC,  SH_SECTION_NONE, 
    sh_log_set_console },

#ifdef WITH_MESSAGE_QUEUE
  { N_("setconsole"),        SH_SECTION_MISC,  SH_SECTION_NONE, 
    sh_log_set_console },

  { N_("messagequeueactive"),SH_SECTION_MISC,  SH_SECTION_NONE, 
    enable_msgq },
#endif

  { N_("setdatabasepath"),    SH_SECTION_MISC,  SH_SECTION_NONE, 
    sh_readconf_set_database_path },

  { N_("setlogfilepath"),     SH_SECTION_MISC,  SH_SECTION_NONE, 
    sh_readconf_set_logfile_path },

  { N_("setlockfilepath"),    SH_SECTION_MISC,  SH_SECTION_NONE, 
    sh_readconf_set_lockfile_path },

  { N_("hidesetup"),         SH_SECTION_MISC,  SH_SECTION_NONE, 
    sh_util_hidesetup },

  { N_("syslogfacility"),    SH_SECTION_LOG,   SH_SECTION_MISC, 
    sh_log_set_facility },

  { N_("mactype"),     SH_SECTION_MISC,  SH_SECTION_NONE, 
    sh_util_sigtype },

  { NULL,    0,   0,  NULL}
};




static void sh_readconfig_line (char * line)
{    
  char * c;
  char * tmp;
  int    i;
#if defined (SH_WITH_CLIENT) || defined (SH_STANDALONE) 
  int    modnum, modkey;
#endif

  static char  *ident[] = {
    N_("severityreadonly"),
    N_("severitylogfiles"),
    N_("severitygrowinglogs"),
    N_("severityignorenone"),
    N_("severityignoreall"),
    N_("severityattributes"),
    N_("severitydirs"),
    N_("severityfiles"),
    N_("severitynames"),
    NULL
  };

  static int      identnum[] = { 
    SH_ERR_T_RO,    
    SH_ERR_T_LOGS,  
    SH_ERR_T_GLOG,  
    SH_ERR_T_NOIG,  
    SH_ERR_T_ALLIG, 
    SH_ERR_T_ATTR, 
    SH_ERR_T_DIR,   
    SH_ERR_T_FILE, 
    SH_ERR_T_NAME        
  };
    
  SL_ENTER(_("sh_readconf_line"));

  /* interpret line                                    */

  c = strchr(line, '=');
  if (c == NULL || (*c) == '\0')
    {
      if (line != NULL)
	{
	  TPT(( 0, FIL__, __LINE__, _("msg=<ConfigFile: not key=value: %s>\n"),
		line));
	}
      SL_RET0(_("sh_readconf_line"));
    }
  else
    ++c;

  while ((*c) == ' ' || (*c) == '\t')
    ++c;

  if ((*c) == '\0')     /* no value                    */
    {
      if (line != NULL)
	{
	  TPT(( 0, FIL__, __LINE__, _("msg=<ConfigFile: not key=value: %s>\n"),
		line));
	}
      SL_RET0(_("sh_readconf_line"));
    }

  /* convert to lowercase                              */

  tmp = line;
  while (*tmp != '=')
    {
      *tmp = tolower( (int) *tmp);
      ++tmp;
    }

  if (!sl_is_suid())
    {
      TPT(( 0, FIL__, __LINE__, _("msg=<ConfigFile: %s>\n"), line));
    }


#if defined (SH_WITH_CLIENT) || defined (SH_STANDALONE) 
  if      (read_mode == SH_SECTION_OTHER) 
    {
      for (modnum = 0; modList[modnum].name != NULL; ++modnum) 
	{
	  for (modkey = 0; modList[modnum].conf_table[modkey].the_opt != NULL; 
	       ++modkey) 
	    {
	      if (sl_strncmp (line,  
			      _(modList[modnum].conf_table[modkey].the_opt),
			      sl_strlen(modList[modnum].conf_table[modkey].the_opt)) == 0)
		{
		  modList[modnum].conf_table[modkey].func(c);
		  if (!sl_is_suid())
		    {
		      TPT(( 0, FIL__, __LINE__, 
			    _("msg=<line = %s, option = %s>\n"), line,
			    _(modList[modnum].conf_table[modkey].the_opt)));
		    }
		}
	    }
	}
    }
#endif


  if (read_mode == SH_SECTION_THRESHOLD) 
    {
      i = 0;
      while (ident[i] != NULL) {
	if (sl_strncmp (line, _(ident[i]), sl_strlen(ident[i])-1) == 0)
	  sh_error_set_iv (identnum[i], c);
	++i;
      }
    }
  else  
    {
      i = 0;
      while (ext_table[i].optname != NULL)
	{
	  if ((ext_table[i].section == read_mode || 
	       ext_table[i].alt_section == read_mode) &&
	      sl_strncmp (line, _(ext_table[i].optname), 
			  sl_strlen(ext_table[i].optname)) == 0)
	    {
	      if (0 != ext_table[i].func (c))
		sh_error_handle ((-1), FIL__, __LINE__, 0, MSG_EINVALS,
				 _(ext_table[i].optname), c);
	      break;
	    }
	  ++i;
	}
    }
     
  SL_RET0(_("sh_readconf_line"));
}
  
    
